//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexBufferAssetBuilder.h"

#include <FeatStd/Util/StaticObject.h>
#include <Candera/Engine3D/Core/VertexGeometry.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/VertexBufferCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/VertexAttributeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDecompression.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultResourceProvider.h>

namespace Candera {
    using namespace Diagnostics;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        VertexBufferPtr VertexBufferBuilder::Create(LoaderContext& /*context*/)
        {
            return VertexBuffer::Create();
        }

        bool VertexBufferReader::ReadFirstPass(
            VertexBuffer& vertexBuffer,
            LoaderContext& context)
        {
            UInt16 vertexElementCount = static_cast<UInt16>(CFFReader::GetVertexBufferVertexAttributesCount(context.handle));
            VertexGeometry::VertexElementFormat* vertexFormat = FEATSTD_NEW_ARRAY(VertexGeometry::VertexElementFormat, vertexElementCount);
            if (vertexFormat == 0) {
                FEATSTD_LOG_ERROR("Failed to create VertexElementFormat for VertexBuffer " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }
            for (UInt16 i = 0; i < vertexElementCount; i++) {
                const AssetDataHandle& vertexElementHandle = CFFReader::GetVertexBufferVertexAttributesElementAt(context.handle, CFFReader::CFF_VERTEX_ATTRIBUTE_SIZE, i);
                if (!vertexElementHandle.IsValid()) {
                    FEATSTD_DELETE_ARRAY(vertexFormat);
                    FEATSTD_LOG_ERROR("Failed to read VertexElement of VertexBuffer " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }
                vertexFormat[i].Offset = CFFReader::GetVertexAttributeOffset(vertexElementHandle);
                vertexFormat[i].Type = static_cast<VertexGeometry::VertexDataType>(CFFReader::GetVertexAttributeType(vertexElementHandle));
                vertexFormat[i].Usage = static_cast<VertexGeometry::VertexUsage>(CFFReader::GetVertexAttributeId(vertexElementHandle));
                vertexFormat[i].UsageIndex = CFFReader::GetVertexAttributeUsageIndex(vertexElementHandle);
            }


            const ResourceDataHandle& vertexBufferHandle = DefaultResourceProvider::GetResourceProviderInstance().CreateRegionDataHandle(
                VertexBufferLib,
                vertexBuffer.GetId(),
                static_cast<UInt16>(CFFReader::GetVertexBufferVerticesRegion(context.handle)),
                static_cast<UInt32>(CFFReader::GetVertexBufferVerticesLength(context.handle)));

            if (!vertexBufferHandle.m_accessor.IsValid()) {
                FEATSTD_LOG_ERROR("VertexBuffer " AssetIdLogStr " has no vertex buffer attached", AssetIdLogArgs(context.id));
                FEATSTD_DELETE_ARRAY(vertexFormat);
                return false;
            }

            ResourceDataHandle indexBufferHandle = ResourceDataHandle::InvalidHandle();
            if (CFFReader::GetVertexBufferIndexesLength(context.handle) != 0) {
                indexBufferHandle = DefaultResourceProvider::GetResourceProviderInstance().CreateRegionDataHandle(
                    VertexBufferLib,
                    vertexBuffer.GetId(),
                    static_cast<UInt16>(CFFReader::GetVertexBufferIndexesRegion(context.handle)),
                    static_cast<UInt32>(CFFReader::GetVertexBufferIndexesLength(context.handle)));

                if (!vertexBufferHandle.m_accessor.IsValid()) {
                    FEATSTD_LOG_ERROR("VertexBuffer " AssetIdLogStr " has no index buffer attached", AssetIdLogArgs(context.id));
                    FEATSTD_DELETE_ARRAY(vertexFormat);
                    return false;
                }
            }

            VertexGeometry* vertexGeometry = FEATSTD_NEW(VertexGeometry)(
                vertexBufferHandle,
                indexBufferHandle,
                VertexGeometry::FormatArrayResource::CreateHandle(vertexFormat, VertexGeometry::VertexElementFormatDisposer::Dispose, vertexElementCount * static_cast<UInt32>(sizeof(VertexGeometry::VertexElementFormat))),
                CFFReader::GetVertexBufferVertexStride(context.handle),
                static_cast<VertexGeometry::MemoryPool>(CFFReader::GetVertexBufferMemoryPool(context.handle)),
                static_cast<VertexGeometry::BufferType>(CFFReader::GetVertexBufferVertexBufferType(context.handle)),
                static_cast<VertexGeometry::BufferUsage>(CFFReader::GetVertexBufferUsage(context.handle)));

            if (vertexGeometry != 0) {
                if (!vertexBuffer.SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose))
                {
                    FEATSTD_LOG_ERROR("Failed to associate VertexGeometry to VertexBuffer " AssetIdLogStr, AssetIdLogArgs(context.id));
                    FEATSTD_DELETE(vertexGeometry);
                    return false;
                }
                vertexBuffer.SetPrimitiveType(static_cast<VertexBuffer::PrimitiveType>(CFFReader::GetVertexBufferPrimitiveType(context.handle)));
            }
            else {
                FEATSTD_LOG_ERROR("Failed to create VertexGeometry for VertexBuffer " AssetIdLogStr, AssetIdLogArgs(context.id));
                FEATSTD_DELETE_ARRAY(vertexFormat);
                return false;
            }

            return true;
        }
    }
}
