//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextureAssetBuilder.h"
#include <Candera/Engine3D/Core/ProxyTextureImage.h>
#include <Candera/Engine3D/Core/CubeMapTextureImage.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TextureCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        SharedPointer<Texture> AssetBuilderBase<SharedPointer<Texture> >::Create(LoaderContext& /*context*/)
        {
            return Texture::Create();
        }

        bool AssetReaderBase<Texture>::ReadFirstPass(Texture& texture, LoaderContext& context)
        {
            SharedPointer<TextureImage> textureImage = SharedPointer<TextureImage>(0);

            TextureImage::TextureTargetType textureType = static_cast<TextureImage::TextureTargetType>(CFFReader::GetTextureTextureType(context.handle));
            switch(textureType) {
                case TextureImage::Texture2D: {
                    if (CFFReader::GetTextureImagesLength(context.handle) != 1) {
                        FEATSTD_LOG_ERROR("Asset error. Wrong number of image sources for Texture2D TextureImage type.");
                        return false;
                    }
                    AssetId imageSourceAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTextureImagesElementAt(context.handle, 0));
                    if (!imageSourceAssetId.IsValid()) {
                        FEATSTD_LOG_DEBUG("Image Source AssetId is not valid");
                    }
                    textureImage = context.provider->GetBitmapTextureImageByAssetId(imageSourceAssetId);
                    if (textureImage == 0) {
                        GraphicDeviceUnit* fbRenderTarget = context.provider->GetGraphicDeviceUnitByAssetId(imageSourceAssetId);
                        if (fbRenderTarget != 0) {
                            textureImage = SharedPointer<TextureImage>(ProxyTextureImage::Create(fbRenderTarget->ToImageSource3D()));
                        }
                        else {
                            FEATSTD_LOG_WARN("Graphics device could not be found.");
                        }
                    }
                } break;
                case TextureImage::TextureCubeMap:{
                    if (CFFReader::GetTextureImagesLength(context.handle) != CubeMapTextureImage::TargetFaceCount) {
                        FEATSTD_LOG_ERROR("Asset error. Wrong number of image sources for TextureCubeMap TextureImage type.");
                        return false;
                    }

                    SharedPointer<CubeMapTextureImage> image = CubeMapTextureImage::Create();
                    for (UInt8 cubeMapFace = 0; cubeMapFace < CubeMapTextureImage::TargetFaceCount; ++cubeMapFace) {
                        Candera::Internal::AssetId resultAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTextureImagesElementAt(context.handle, cubeMapFace));
                        if (!resultAssetId.IsValid()) {
                            FEATSTD_LOG_DEBUG("GetAssetId is not valid");
                        }
                        FeatStd::MemoryManagement::SharedPointer<Bitmap> result = context.provider->GetBitmapByAssetId(resultAssetId);
                        if (result.PointsToNull()){
                            FEATSTD_LOG_WARN("GetBitmapByAssetId returns a null pointer.");
                        }
                        if (!image->SetBitmapFace(static_cast<CubeMapTextureImage::TargetFace>(cubeMapFace), result)) {
                            FEATSTD_LOG_WARN("SetBitmapFace failed for CubeMapTextureImage %s", image->GetName());
                        }
                    }
                    textureImage = image;
                } break;
                default: FEATSTD_LOG_ERROR("Invalid texture type for Texture " AssetIdLogStr, AssetIdLogArgs(context.id));
            }

            if (textureImage == 0) {
                FEATSTD_LOG_ERROR("Could not create TextureImage for Texture " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }
            texture.SetTextureImage(textureImage);

            texture.SetMipMapFilter(static_cast<Texture::MipMapFilter>(CFFReader::GetTextureMipMapFilter(context.handle)));
            texture.SetMagnificationFilter(static_cast<Texture::MinMagFilter>(CFFReader::GetTextureMagnificationFilter(context.handle)));
            texture.SetMinificationFilter(static_cast<Texture::MinMagFilter>(CFFReader::GetTextureMinificationFilter(context.handle)));
            texture.SetWrapModeU(static_cast<Texture::WrapMode>(CFFReader::GetTextureWrapModeU(context.handle)));
            texture.SetWrapModeV(static_cast<Texture::WrapMode>(CFFReader::GetTextureWrapModeV(context.handle)));
            texture.SetMaxAnisotropy(CFFReader::GetTextureAnisotropy(context.handle));

            return true;
        }
    }
}
