//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ShaderParamSetterAssetBuilder.h"
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/UniformSetterCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/UniformCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <FeatStd/Util/PointerUtil.h>

namespace Candera {
    namespace Internal {
        using namespace Diagnostics;
        using namespace MemoryManagement;
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        class UniformBuffer{

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(UniformBuffer);

            static SharedPointer  Create(UInt32 bufferSize){ return SharedPointer(FEATSTD_NEW(UniformBuffer)(bufferSize)); }
            void* GetBuffer() { return m_buffer; }
            ~UniformBuffer();

        private:
            UniformBuffer(UInt32 bufferSize) : m_buffer(0) { m_buffer = FEATSTD_ALLOC(bufferSize); }

            void* m_buffer;

            FEATSTD_SHARED_POINTER_DECLARATION();
        };

        UniformBuffer::~UniformBuffer()
        {
            if (m_buffer != 0) {
                FEATSTD_FREE(m_buffer);
                m_buffer = 0;
            }
        }

        class UniformArraysProperties {
        public:
            static bool SetUniformsList(ShaderParamSetter& sps, UniformBuffer::SharedPointer uniform)
            {
                return sps.SetValue(CdaDynamicPropertyInstance(UniformsList), uniform);
            }

            static const UniformBuffer::SharedPointer& DefaultList()
            {
                static UniformBuffer::SharedPointer defaultUniformBuffer = UniformBuffer::SharedPointer(0);
                return defaultUniformBuffer;
            }

            static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host)
            {
                    FEATSTD_UNUSED(host);
                    return 0;
            }
        private:
            CdaDynamicProperties(UniformArraysProperties, Candera::DynamicProperties::DynamicPropertyHost);
                CdaDynamicPropertyUnregistered(UniformsList, UniformBuffer::SharedPointer);
                    CdaDynamicPropertyDefaultValue(DefaultList());
                CdaDynamicPropertyEnd();
            CdaDynamicPropertiesEnd();
        };

        static UInt8 GetUniformSize(Shader::UniformType uniformType)
        {
            UInt8 uniformSize = 0;
            switch (uniformType) {
            case Shader::Integer:
            case Shader::Float:
            case Shader::Bool:
            case Shader::Sampler2D:
            case Shader::SamplerCube:
                uniformSize = 1; break;
            case Shader::IntegerVec2:
            case Shader::FloatVec2:
            case Shader::BoolVec2:
                uniformSize = 2; break;
            case Shader::IntegerVec3:
            case Shader::FloatVec3:
            case Shader::BoolVec3:
                uniformSize = 3; break;
            case Shader::IntegerVec4:
            case Shader::FloatVec4:
            case Shader::FloatMat2:
            case Shader::BoolVec4:
                uniformSize = 4; break;
            case Shader::FloatMat3:
                uniformSize = 9; break;
            case Shader::FloatMat4:
                uniformSize = 16; break;
            default:
                break;
            }
            return uniformSize;
        }

        bool AssetReaderBase<GenericShaderParamSetter>::ReadFirstPass(GenericShaderParamSetter& shaderParamSetter, LoaderContext& context)
        {
            bool result = true;
            UInt32 totalSize = 0;

            shaderParamSetter.SetProjectionMatrix4Enabled(CFFReader::GetUniformSetterIsProjectionMatrix4Enabled_Cff(context.handle));
            shaderParamSetter.SetModelMatrix4Enabled(CFFReader::GetUniformSetterIsModelMatrix4Enabled_Cff(context.handle));
            shaderParamSetter.SetModelMatrix3Enabled(CFFReader::GetUniformSetterIsModelMatrix3Enabled_Cff(context.handle));
            shaderParamSetter.SetNormalModelMatrix3Enabled(CFFReader::GetUniformSetterIsNormalModelMatrix3Enabled_Cff(context.handle));
            shaderParamSetter.SetModelViewMatrix4Enabled(CFFReader::GetUniformSetterIsModelViewMatrix4Enabled_Cff(context.handle));
            shaderParamSetter.SetModelViewMatrix3Enabled(CFFReader::GetUniformSetterIsModelViewMatrix3Enabled_Cff(context.handle));
            shaderParamSetter.SetNormalModelViewMatrix3Enabled(CFFReader::GetUniformSetterIsNormalModelViewMatrix3Enabled_Cff(context.handle));
            shaderParamSetter.SetModelViewProjectionMatrix4Enabled(CFFReader::GetUniformSetterIsModelViewProjectionMatrix4Enabled_Cff(context.handle));
            shaderParamSetter.SetCameraLookAtVectorEnabled(CFFReader::GetUniformSetterIsCameraLookAtVectorEnabled_Cff(context.handle));
            shaderParamSetter.SetCameraPositionEnabled(CFFReader::GetUniformSetterIsCameraPositionEnabled_Cff(context.handle));
            shaderParamSetter.SetLightActivationEnabled(CFFReader::GetUniformSetterIsLightsActivationEnabled_Cff(context.handle));
            shaderParamSetter.SetMaterialActivationEnabled(CFFReader::GetUniformSetterIsMaterialActivationEnabled_Cff(context.handle));
            shaderParamSetter.SetTextureActivationEnabled(CFFReader::GetUniformSetterIsTextureActivationEnabled_Cff(context.handle));
            shaderParamSetter.SetLightsCoordinateSpace(static_cast<Light::CoordinateSpace>(CFFReader::GetUniformSetterLightCoordinateSpace(context.handle)));
            shaderParamSetter.SetPointSpriteActivationEnabled(CFFReader::GetUniformSetterIsPointSpriteActivationEnabled_Cff(context.handle));
            shaderParamSetter.SetMorphingMeshActivationEnabled(CFFReader::GetUniformSetterIsMorphingMeshActivationEnabled_Cff(context.handle));
            shaderParamSetter.SetCanvasActivationEnabled(CFFReader::GetUniformSetterIsCanvasActivationEnabled_Cff(context.handle));

            Int uniformsCount = CFFReader::GetUniformSetterUniformsCount(context.handle);
            if (uniformsCount != 0) {
                for (Int uniformIndex = 0; uniformIndex < uniformsCount; ++uniformIndex) {
                    const AssetDataHandle& uniformHandle = CFFReader::GetUniformSetterUniformsElementAt(context.handle, uniformIndex);
                    if (!uniformHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read Uniform %d collection for GenericShaderParamSetter " AssetIdLogStr, uniformIndex, AssetIdLogArgs(context.id));
                        return false;
                    }

                    Shader::UniformType uniformType = static_cast<Shader::UniformType>(CFFReader::GetUniformType(uniformHandle));
                    Int32 uniformCount = 1;

                    if (ShaderParamSetter::IsFloatType(uniformType)) {
                        totalSize += (static_cast<UInt32>(uniformCount)) * GetUniformSize(uniformType) * (sizeof(Float));
                    }
                    else if (ShaderParamSetter::IsIntegerType(uniformType)) {
                        totalSize += (static_cast<UInt32>(uniformCount)) * GetUniformSize(uniformType) * (sizeof(Int32));
                    }
                    else {
                        // do nothing
                    }
                }

                UniformBuffer::SharedPointer list = UniformBuffer::Create(totalSize);
                if (list == 0) {
                    FEATSTD_LOG_ERROR("Not enough memory available to store uniform data for GenericShaderParamSetter " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }
                void* memoryChunk = list->GetBuffer();
                if (memoryChunk == 0) {
                    FEATSTD_LOG_ERROR("Not enough memory available to store uniform data for GenericShaderParamSetter " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                for (Int uniformIndex = 0; uniformIndex < uniformsCount; ++uniformIndex) {
                    const AssetDataHandle& uniformHandle = CFFReader::GetUniformSetterUniformsElementAt(context.handle, uniformIndex);
                    if (!uniformHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read Uniform %d collection for GenericShaderParamSetter " AssetIdLogStr, uniformIndex, AssetIdLogArgs(context.id));
                        return false;
                    }

                const Char* uniformName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetUniformName(uniformHandle));
                    Shader::UniformType uniformType = static_cast<Shader::UniformType>(CFFReader::GetUniformType(uniformHandle));
                    const void* uniformValue = CFFReader::GetUniformValue(uniformHandle);
                    Int32 uniformCount = 1;
                    if (ShaderParamSetter::IsFloatType(uniformType)) {
                        OffsetType sizeOfUniform = static_cast<OffsetType>(sizeof(Float)) * static_cast<OffsetType>(uniformCount)* static_cast<OffsetType>(GetUniformSize(uniformType));
                        MemoryPlatform::Copy(memoryChunk, uniformValue, sizeOfUniform);
                        if (!shaderParamSetter.SetUniform(uniformName, uniformType, FeatStd::Internal::PointerToPointer<Float*>(memoryChunk), uniformCount)) {
                            FEATSTD_LOG_ERROR("Failed to set Uniform %s for GenericShaderParamSetter " AssetIdLogStr, uniformName, AssetIdLogArgs(context.id));
                            result = false;
                        }
                        memoryChunk = FeatStd::Internal::PointerAdd(memoryChunk, sizeOfUniform);
                    }
                    else if (ShaderParamSetter::IsIntegerType(uniformType)) {
                        OffsetType sizeOfUniform = static_cast<OffsetType>(sizeof(Int32)) * static_cast<OffsetType>(uniformCount)* static_cast<OffsetType>(GetUniformSize(uniformType));
                        MemoryPlatform::Copy(memoryChunk, uniformValue, sizeOfUniform);
                        if (!shaderParamSetter.SetUniform(uniformName, uniformType, FeatStd::Internal::PointerToPointer<Int32*>(memoryChunk), uniformCount)) {
                            FEATSTD_LOG_ERROR("Failed to set Uniform %s for GenericShaderParamSetter " AssetIdLogStr, uniformName, AssetIdLogArgs(context.id));
                            result = false;
                        }
                        memoryChunk = FeatStd::Internal::PointerAdd(memoryChunk, sizeOfUniform);
                    }
                    else {
                        //do nothing
                    }

                }
                if (!UniformArraysProperties::SetUniformsList(shaderParamSetter, list)) {
                    FEATSTD_LOG_ERROR("Failed to associate uniform location data to GenericShaderParamSetter " AssetIdLogStr, AssetIdLogArgs(context.id));
                    result = false;
                }
            }

            return result;
        }

        SharedPointer<GenericShaderParamSetter> AssetBuilderBase<SharedPointer<GenericShaderParamSetter> >::Create(LoaderContext& /*context*/)
        {
            return GenericShaderParamSetter::Create();
        }
    }
}
