//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SceneAssetBuilder.h"
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/RenderOrder/RenderOrder.h>
#include <Candera/Engine3D/RenderOrder/DistanceToCameraOrderCriterion.h>
#include <Candera/Engine3D/RenderOrder/RankOrderCriterion.h>
#include <Candera/Engine3D/RenderOrder/BatchOrderCriterion.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/System/Diagnostics/Log.h>

#include <CanderaAssetLoader/AssetLoaderBase/CffReader/Scene3DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/RenderOrderBinCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/RenderOrderBinCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    using namespace Diagnostics;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        Scene* AssetBuilderBase<Scene*>::Create(LoaderContext& /*context*/)
        {
            return Scene::Create();
        }

        class CameraTraverser : public TreeTraverser
        {
        public:
            CameraTraverser(RenderOrder* renderOrder) : m_renderOrder(renderOrder), m_isFirstRenderOrderSet(false) {};
        private:
            RenderOrder* m_renderOrder;
            bool m_isFirstRenderOrderSet;

            virtual TraverserAction ProcessNode(Node& node) override final;
        };

        TreeTraverser::TraverserAction CameraTraverser::ProcessNode(Node& node)
        {
            Camera* camera = Dynamic_Cast<Camera*>(&node);
            if (camera != 0) {
                RenderOrder* renderOrder = m_renderOrder;
                if (!m_isFirstRenderOrderSet) {
                    m_isFirstRenderOrderSet = true;
                }
                else {
                    renderOrder = m_renderOrder->Clone();
                }
                camera->SetRenderOrder(renderOrder);
            }

            return ProceedTraversing;
        }

        bool AssetReaderBase<Scene>::ReadFirstPass(Scene& scene, LoaderContext& context)
        {
            //Disable scene rendering while scene tree is not complete, because each camera is added to the Renderer at its construction time. Will be enabled during ReadSecondPass
            scene.SetRenderingEnabled(false);
            if (!SceneNodeChildrenReader().ReadFirstPass(scene, context)) {
                return false;
            }

            const AssetDataHandle& renderOrderBinCollectionHandle = CFFReader::GetScene3DRenderOrderBins(context.handle);
            if (!renderOrderBinCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read RenderOrderBin collection for Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            Int32 renderOrderBinCount = CFFReader::GetRenderOrderBinCollectionChildrenCount(renderOrderBinCollectionHandle);
            Int32 opaqueBinSize = 0;
            Int32 transparentBinSize = 0;

            for (Int32 renderOrderBinIndex = 0; renderOrderBinIndex < renderOrderBinCount; ++renderOrderBinIndex) {
                const AssetDataHandle& renderOrderBinHandle = CFFReader::GetRenderOrderBinCollectionChildrenElementAt(renderOrderBinCollectionHandle, renderOrderBinIndex);
                if (!renderOrderBinHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read RenderOrderBin collection for Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                const Char* renderOrderBinName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(renderOrderBinHandle));

                if (StringPlatform::CompareStrings(renderOrderBinName, "Opaque") == 0) {
                    opaqueBinSize = CFFReader::GetRenderOrderBinSizeHint(renderOrderBinHandle);
                }
                else {
                    if (StringPlatform::CompareStrings(renderOrderBinName, "Transparent") == 0) {
                        transparentBinSize = CFFReader::GetRenderOrderBinSizeHint(renderOrderBinHandle);
                    }
                }
            }

            RenderOrder* renderOrder = RenderOrder::Create(opaqueBinSize, transparentBinSize);
            if (renderOrder == 0) {
                FEATSTD_LOG_WARN("RenderOrder could not be created during build of Scene" AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            for (Int32 renderOrderBinIndex = 0; renderOrderBinIndex < renderOrderBinCount; ++renderOrderBinIndex) {
                const AssetDataHandle& renderOrderBinHandle = CFFReader::GetRenderOrderBinCollectionChildrenElementAt(renderOrderBinCollectionHandle, renderOrderBinIndex);
                if (!renderOrderBinHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read RenderOrderBin collection for Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                const Char* renderOrderBinName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(renderOrderBinHandle));

                bool opaqueRenderOrderBin = (StringPlatform::CompareStrings(renderOrderBinName, "Opaque") == 0);
                bool transparentRenderOrderBin = (StringPlatform::CompareStrings(renderOrderBinName, "Transparent") == 0);

                if (!(opaqueRenderOrderBin || transparentRenderOrderBin)) {
                    if (!renderOrder->CreateBin(renderOrderBinName, renderOrderBinIndex, CFFReader::GetRenderOrderBinSizeHint(renderOrderBinHandle))) {
                        FEATSTD_LOG_WARN("RenderOrderBin could not be created during build of Scene" AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                }

                renderOrder->SetBinRank(renderOrderBinName, renderOrderBinIndex);
                renderOrder->SetBinSortingEnabled(renderOrderBinName, CFFReader::GetRenderOrderBinIsSortingEnabled(renderOrderBinHandle));
                SupportedOrderCriterion selectedCriterion = static_cast<SupportedOrderCriterion>(CFFReader::GetRenderOrderBinCriterion(renderOrderBinHandle));

                static DistanceToCameraOrderCriterion distanceToCameraCriterion[4];
                static ReverseDistanceToCameraOrderCriterion reverseDistanceToCameraCriterion[4];
                static RankOrderCriterion renderOrderRankCriterion;
                static ReverseRankOrderCriterion reverseRenderOrderRankCriterion;
                static BatchOrderCriterion batchCriterion[2];

                distanceToCameraCriterion[0].SetDistanceFunction(CameraOrderCriterion::CameraPositionToBoundingSphere);
                reverseDistanceToCameraCriterion[0].SetDistanceFunction(CameraOrderCriterion::CameraPositionToBoundingSphere);
                distanceToCameraCriterion[1].SetDistanceFunction(CameraOrderCriterion::CameraPositionToBoundingBox);
                reverseDistanceToCameraCriterion[1].SetDistanceFunction(CameraOrderCriterion::CameraPositionToBoundingBox);
                distanceToCameraCriterion[2].SetDistanceFunction(CameraOrderCriterion::ViewPlaneToBoundingSphere);
                reverseDistanceToCameraCriterion[2].SetDistanceFunction(CameraOrderCriterion::ViewPlaneToBoundingSphere);
                distanceToCameraCriterion[3].SetDistanceFunction(CameraOrderCriterion::ViewPlaneToBoundingBox);
                reverseDistanceToCameraCriterion[3].SetDistanceFunction(CameraOrderCriterion::ViewPlaneToBoundingBox);
                batchCriterion[0].SetMode(BatchOrderCriterion::ShaderAndRenderMode);
                batchCriterion[1].SetMode(BatchOrderCriterion::Appearance);

                OrderCriterion* orderCriterion = 0;

                switch (selectedCriterion) {
                    case DistanceToCamera: {
                        switch (static_cast<CameraOrderCriterion::DistanceFunction>(CFFReader::GetRenderOrderBinDistanceFunction(renderOrderBinHandle))) {
                            case CameraOrderCriterion::CameraPositionToBoundingSphere: orderCriterion = &distanceToCameraCriterion[0]; break;
                            case CameraOrderCriterion::CameraPositionToBoundingBox: orderCriterion = &distanceToCameraCriterion[1]; break;
                            case CameraOrderCriterion::ViewPlaneToBoundingSphere: orderCriterion = &distanceToCameraCriterion[2]; break;
                            case CameraOrderCriterion::ViewPlaneToBoundingBox: orderCriterion = &distanceToCameraCriterion[3]; break;
                            default: FEATSTD_LOG_ERROR("Invalid DistanceFunction for RenderOrderBin of Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                        break;
                    }
                    case ReverseDistanceToCamera: {
                        switch (static_cast<CameraOrderCriterion::DistanceFunction>(CFFReader::GetRenderOrderBinDistanceFunction(renderOrderBinHandle))) {
                            case CameraOrderCriterion::CameraPositionToBoundingSphere: orderCriterion = &reverseDistanceToCameraCriterion[0]; break;
                            case CameraOrderCriterion::CameraPositionToBoundingBox: orderCriterion = &reverseDistanceToCameraCriterion[1]; break;
                            case CameraOrderCriterion::ViewPlaneToBoundingSphere: orderCriterion = &reverseDistanceToCameraCriterion[2]; break;
                            case CameraOrderCriterion::ViewPlaneToBoundingBox: orderCriterion = &reverseDistanceToCameraCriterion[3]; break;
                            default: FEATSTD_LOG_ERROR("Invalid DistanceFunction for RenderOrderBin of Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                        break;
                    }
                    case RenderOrderRank: orderCriterion = &renderOrderRankCriterion; break;
                    case ReverseRenderOrderRank: orderCriterion = &reverseRenderOrderRankCriterion; break;
                    case BatchOrder: {
                        switch (static_cast<BatchOrderCriterion::Mode>(CFFReader::GetRenderOrderBinBatchOrder(renderOrderBinHandle))) {
                            case BatchOrderCriterion::ShaderAndRenderMode: orderCriterion = &batchCriterion[0]; break;
                            case BatchOrderCriterion::Appearance: orderCriterion = &batchCriterion[1]; break;
                            default: FEATSTD_LOG_ERROR("Invalid Batch mode for RenderOrderBin of Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                        break;
                    }
                    default: FEATSTD_LOG_ERROR("Invalid OrderCriterion mode for RenderOrderBin of Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                }
                renderOrder->SetBinOrderCriterion(renderOrderBinName, orderCriterion);
            }

            CameraTraverser cameraTraverser(renderOrder);
            cameraTraverser.Traverse(scene);

            return true;
        }

        bool AssetReaderBase<Scene>::ReadSecondPass(Scene& scene, LoaderContext& context)
        {
            //Reenable scene rendering since scene tree is now completed.
            scene.SetRenderingEnabled(true);
            return SceneNodeChildrenReader().ReadSecondPass(scene, context);
        }

        Int32 SceneNodeChildrenReader::GetChildrenCount(CffLoaderContext& context) const
        {
            return CFFReader::GetSceneChildrenCount(context.handle);
        }

        AssetDataHandle SceneNodeChildrenReader::GetChildDataHandle(CffLoaderContext& context, Int32 index) const
        {
            return CFFReader::GetSceneChildrenElementAt(context.handle, index);
        }

    }
}
