//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "NodeAssetBuilder.h"
#include <Candera/System/Diagnostics/Log.h>

#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/Light.h>
#include <Candera/Engine3D/Core/Group.h>
#include <Candera/Engine3D/Core/MorphingMesh.h>
#include <Candera/Engine3D/Core/Billboard.h>
#include <Candera/Engine3D/Core/PointSprite.h>
#include <Candera/Engine3D/Core/ReflectionCamera.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/LineList.h>
#include <Candera/Engine3D/Core/LodNode.h>
#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/Engine3D/Core/StereoCamera.h>
#include <Candera/Engine3D/Core/PlanarShadow.h>
#include <Candera/Engine3D/Core/CompositeGroup.h>
#ifdef CANDERA_3D_CANVAS_ENABLED
#include <Candera/Engine3D/Canvas/Canvas.h>
#include <Candera/Engine3D/Canvas/CanvasGroup.h>
#include <Candera/Engine3D/Canvas/CanvasSprite.h>
#include <Candera/Engine3D/Canvas/CanvasText.h>
#endif
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>

#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/Layout.h>
#endif

#include <CanderaAssetLoader/AssetLoaderBase/CffReader/NodeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnchorDefinitionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnchorCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/DynamicPropertyItemInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/SceneAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/LightAssetBuilder.h>
#ifdef CANDERA_3D_CANVAS_ENABLED
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/CanvasGroupAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/CanvasSpriteAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/CanvasTextAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/CanvasAssetBuilder.h>
#endif
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/CompositeGroupAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/GroupAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/MorphingMeshAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/BillboardAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/PointSpriteAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/ReflectionCameraAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/CameraAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/LodNodeAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/MeshAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/AppearanceCollectionAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/AppearanceAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/StereoCameraAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/PlanarShadowAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/ScriptComponentCollectionAssetReader.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;
    using namespace DynamicProperties;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool NodeChildrenReader::ReadFirstPass(Node& node, CffLoaderContext& context) const
        {
            AssetObjectType nodeType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            if (nodeType == AotCompositeNode) {
                CompositeGroup* compositeGroup = Dynamic_Cast<CompositeGroup*>(&node);
                if (compositeGroup == 0) {
                    FEATSTD_LOG_ERROR("Type described in asset differs from actual type for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                Int32 childernCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childernCount; ++childIndex) {
                    const AssetDataHandle& anchorHandle = GetChildDataHandle(context, childIndex);
                    if (!anchorHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read anchor for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    AssetObjectType anchorType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(anchorHandle));
                   if (anchorType != AotAnchor) {
                        FEATSTD_LOG_ERROR("Type described in asset differs from reqyured Anchor type for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    const AssetId& anchorNodeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(anchorHandle));
                    if (!anchorNodeAssetId.IsValid()) {
                        FEATSTD_LOG_DEBUG("Anchor node AssetId is not valid");
                    }
                    Id anchorId = AssetIdFunctions::GetNodeId(anchorNodeAssetId);
                    Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetAnchorAnchorDefinition(anchorHandle));
                    if (!result.IsValid()) {
                        FEATSTD_LOG_DEBUG("Anchor AssetId is not valid");
                    }
                    bool updateResult = CompositeGroupFunctions::UpdateAnchorPointId(
                        *compositeGroup,
                        AssetIdFunctions::GetNodeId(result),
                        anchorId);
                    Node* anchorNode = compositeGroup->GetAnchorPointNode(anchorId);
                    if ((!updateResult) || (anchorNode == 0)) {
                        FEATSTD_LOG_ERROR("Failed to set anchor for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    CffLoaderContext anchorContext = context.Clone(anchorHandle);
                    if (!ReadFirstPass(*anchorNode, anchorContext)) {
                        FEATSTD_LOG_WARN("Failed to read AnchorNode for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                }
            }
            else {
                Int32 childernCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childernCount; ++childIndex) {
                    const AssetDataHandle& childHandle = GetChildDataHandle(context, childIndex);
                    if (!childHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read child for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                    CffLoaderContext childContext = context.Clone(childHandle);
                    Node* child = AssetBuilder<Node*>::CreateAndBuildFirstPass(childContext);
                    if (child == 0) {
                        FEATSTD_LOG_WARN("Failed to create child for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                    }
                    else {
                        if (!node.AddChild(child)) {
                            FEATSTD_LOG_WARN("Failed to add child for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }
                }
            }

#ifdef CANDERA_SCRIPTING_ENABLED
            if ((nodeType != AotScene) && (nodeType != AotComposite)) {
                if (!ScriptComponentCollectionAssetReader::ReadScriptComponentCollectionData(node, context)) {
                    FEATSTD_LOG_WARN("Script component collection not completely attached to the node.");
                }
            }
#endif

            return true;
        }

        bool NodeChildrenReader::ReadSecondPass(Node& node, CffLoaderContext& context) const
        {
            AssetObjectType nodeType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            if (nodeType == AotCompositeNode) {
                CompositeGroup* compositeGroup = Dynamic_Cast<CompositeGroup*>(&node);
                if (compositeGroup == 0) {
                    FEATSTD_LOG_ERROR("Type described in asset differs from actual type for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                Int32 childernCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childernCount; ++childIndex) {
                    const AssetDataHandle& anchorHandle = GetChildDataHandle(context, childIndex);
                    if (!anchorHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read anchor at index %d for CompositeGroup " AssetIdLogStr, childIndex, AssetIdLogArgs(context.id));
                        return false;
                    }

                    const AssetId& anchorAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(anchorHandle));
                    if (!anchorAssetId.IsValid()) {
                        FEATSTD_LOG_ERROR("CompositeGroup " AssetIdLogStr " has no anchor attached at index %d", AssetIdLogArgs(context.id), childIndex);
                        return false;
                    }
                    Node* anchorNode = compositeGroup->GetAnchorPointNode(AssetIdFunctions::GetNodeId(anchorAssetId));
                    if (anchorNode == 0) {
                        FEATSTD_LOG_ERROR("Failed to retrieve anchor from CompositeGroup " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    CffLoaderContext anchorContext = context.Clone(anchorHandle);
                    if (!ReadSecondPass(*anchorNode, anchorContext)) {
                        FEATSTD_LOG_WARN("Failed to read AnchorNode for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                }
            }
            else {
                Int32 childrenCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childrenCount; ++childIndex) {
                    const AssetDataHandle& childHandle = GetChildDataHandle(context, childIndex);
                    if (!childHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read child for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                    CffLoaderContext childContext = context.Clone(childHandle);
                    Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(childHandle));
                    if (!result.IsValid()) {
                        FEATSTD_LOG_DEBUG("GetAssetId is not valid");
                    }
                    Node* child = node.GetChild(AssetIdFunctions::GetNodeId(result));
                    if (child != 0) {
                        if (!AssetBuilder<Node*>::BuildSecondPass(child, childContext)) {
                            FEATSTD_LOG_WARN("Failed to resolve child for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }
                    else {
                        FEATSTD_LOG_WARN("Failed to retrieve child of Node " AssetIdLogStr " for resolving.", AssetIdLogArgs(context.id));
                    }
                }
            }
            return true;
        }

        class NodeNodeChildrenReader: public NodeChildrenReader {
        protected:
            virtual Int32 GetChildrenCount(CffLoaderContext& context) const
            { return CFFReader::GetNode3DChildrenCount(context.handle); }

            virtual AssetDataHandle GetChildDataHandle(CffLoaderContext& context, Int32 index) const
            { return CFFReader::GetNode3DChildrenElementAt(context.handle, index); }
        };

        bool AssetReaderBase<Node>::ReadFirstPass(Node& node, LoaderContext& context)
        {
            if (!NodeNodeChildrenReader().ReadFirstPass(node, context)) {
                FEATSTD_LOG_WARN("Failed to load children for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }
            bool result = true;
            SharedPointer<Appearance> appearance(0);

            switch (CFFReader::GetNode3DResolvedAppearanceCollectionAttachmentType(context.handle)) {
            case NotSharedAttachment:
                {
                    const AssetDataHandle& appearanceCollectionHandle = CFFReader::GetNode3DResolvedAppearanceCollectionAttachment(context.handle);
                    LoaderContext appearanceContext = context.Clone(appearanceCollectionHandle);
                    appearance = AppearanceCollectionAssetBuilder::CreateAndBuildFirstPass(appearanceContext);
                    if (!AppearanceCollectionAssetBuilder::BuildSecondPass(appearance, appearanceContext)) {
                        FEATSTD_LOG_WARN("BuildSecondPass failed.");
                    }
                    break;
                }
            case SharedAttachment:
                {
                    Candera::Internal::AssetId assetId = AssetIdFunctions::GetAssetId(CFFReader::GetNode3DResolvedAppearanceCollectionAttachmentId(context.handle));
                    if (!assetId.IsValid()) {
                        FEATSTD_LOG_DEBUG("SharedAttachment AssetId is not valid");
                    }
                    appearance = context.provider->GetAppearanceByAssetId(assetId);
                    if (appearance.PointsToNull()){
                        FEATSTD_LOG_DEBUG("GetAppearanceByAssetId returns a null pointer.");
                    }
                    break;
                }
            default:
                break;
            }

            node.SetAppearance(appearance);
            node.SetRenderingEnabled(CFFReader::GetNode3DIsRenderingEnabled(context.handle));
            node.SetAlphaValue(CFFReader::GetNode3DAlphaValue(context.handle));

            Float x = 0.0F;
            Float y = 0.0F;
            Float z = 0.0F;
            CFFReader::GetNode3DCenter(context.handle, x, y, z);

            node.SetCenter(Vector3(x, y, z));
            node.SetRadius(CFFReader::GetNode3DRadius(context.handle));

            CFFReader::GetNode3DBoxLowerLeftBackVertex(context.handle, x, y, z);
            Vector3 minBounds(x, y, z);
            CFFReader::GetNode3DBoxUpperRightFrontVertex(context.handle, x, y, z);
            Vector3 maxBounds(x, y, z);
            node.SetBoundingBox(minBounds, maxBounds);
            node.SetIntersectionTestEnabled(CFFReader::GetNode3DIsIntersectionTestEnabled(context.handle));

            Candera::Internal::AssetId resultAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetNode3DRenderOrderBin(context.handle));
            const Char* renderOrderBinName = context.provider->GetNameByAssetId(resultAssetId);
            if ((renderOrderBinName == 0) || (StringPlatform::CompareStrings(renderOrderBinName, "") == 0)) {
                node.SetRenderOrderBinAssignment(0);
            } else {
                node.SetRenderOrderBinAssignment(renderOrderBinName);
            }

            node.SetRenderOrderRank(CFFReader::GetNode3DRenderOrderRank(context.handle));
            node.SetRenderBenchmark(CFFReader::GetNode3DRenderBenchmark(context.handle));

            UInt32 scopeBits = CFFReader::GetNode3DScopeBits(context.handle);
            for (UInt32 scopeIndex = 0; scopeIndex < 32; scopeIndex++) {
                result = result && node.SetScopeEnabled(scopeIndex, ((scopeBits >> scopeIndex) & 1U) == 1U);
            }

            return result;
        }

        bool AssetReaderBase<Node>::ReadSecondPass(Node& node, LoaderContext& context)
        {
            if (!NodeNodeChildrenReader().ReadSecondPass(node, context)) {
                FEATSTD_LOG_WARN("Failed to load children for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

#if defined(CANDERA_LAYOUT_ENABLED)
            Layouter* layouter = node.GetLayouter();
            Layouter* parentLayouter = (node.GetParent() != 0)?node.GetParent()->GetLayouter():0;

            if (layouter == 0) {
                layouter = DefaultLayouter::GetInstance();
            }
            if (parentLayouter == 0) {
                parentLayouter = DefaultLayouter::GetInstance();
            }

            //Move into a separate function when property iteration will be generic in CFFReader::
            Int32 layoutPropertyCount = CFFReader::GetNodeLayoutPropertiesCount(context.handle);
            for (Int32 layoutPropertyIndex = 0; layoutPropertyIndex < layoutPropertyCount; ++layoutPropertyIndex) {
                const AssetDataHandle& layoutPropertiesHandle = CFFReader::GetNodeLayoutPropertiesElementAt(context.handle, CFFReader::CFF_DYNAMIC_PROPERTY_ITEM_INFO_SIZE, layoutPropertyIndex);
                LoaderContext layoutPropertyContext = context.Clone(layoutPropertiesHandle);
                bool isPropertySet = ReadDynamicProperty(layouter->GetPropertyHierarchy(), node, layoutPropertyContext);
                if (!isPropertySet) {
                    if (!ReadDynamicProperty(parentLayouter->GetPropertyHierarchy(), node, layoutPropertyContext)) {
                        FEATSTD_LOG_WARN("Failed to load dynamic property for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                }
            }
#endif
            return true;
        }       

        bool AssetReaderBase<Node>::ReadDynamicProperty(const DynamicProperties::PropertyHierarchyNode& phn, DynamicProperties::DynamicPropertyHost& host, LoaderContext& context)
        {
            if (!context.IsValid()) {
                FEATSTD_LOG_ERROR("Asset error, missing handle for layouter properties.\n");
                return false;
            }
            const Char* propName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetDynamicPropertyItemInfoName(context.handle));
            const Char* propValue = CFFReader::GetDynamicPropertyItemInfoValue(context.handle);
            DynamicPropertyBase* searchedDynamicProperty = 0;
            for (const DynamicProperties::PropertyHierarchyNode* phnPtr = &phn; (phnPtr != 0) && (searchedDynamicProperty == 0); phnPtr = phnPtr->GetParent()) {
                DynamicPropertyBase* dynamicProp = phnPtr->GetProperties();
                for (bool found = false; ((dynamicProp != 0) && (!found)); dynamicProp = dynamicProp->GetNext()) {
                    found = (StringPlatform::CompareStrings(dynamicProp->GetPropertyName(), propName) == 0);
                    if (found) {
                        searchedDynamicProperty = dynamicProp;
                    }
                }
            }

            if (searchedDynamicProperty == 0) {
                return false;
            }

            if (!searchedDynamicProperty->ConvertFromString(host, propValue)) {
                FEATSTD_LOG_ERROR("Failed to set property %s with value %s", propName, propValue);
                return false;
            }

            return true;
        }

        Node* AssetBuilder<Node*>::CreateAndBuildFirstPass(LoaderContext& context)
        {
            FEATSTD_LOG_DEBUG("Creating node: %s", AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(context.handle)));
            Node* ret = 0;
            AssetObjectType childType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            switch (childType) {
            case AotGroup: ret =  AssetBuilder<Group*>::CreateAndBuildFirstPass(context); break;
#ifdef CANDERA_3D_CANVAS_ENABLED
            case AotCanvasGroup: ret = AssetBuilder<CanvasGroup*>::CreateAndBuildFirstPass(context); break;
            case AotCanvasSprite: ret = AssetBuilder<CanvasSprite*>::CreateAndBuildFirstPass(context); break;
            case AotCanvasText: ret = AssetBuilder<CanvasText*>::CreateAndBuildFirstPass(context); break;
            case AotCanvas: ret = AssetBuilder<Canvas*>::CreateAndBuildFirstPass(context); break;
#endif
            case AotCamera: ret =  AssetBuilder<Camera*>::CreateAndBuildFirstPass(context); break;
            case AotMorphingMesh: ret =  AssetBuilder<MorphingMesh*>::CreateAndBuildFirstPass(context); break;
            case AotBillboard: ret =  AssetBuilder<Billboard*>::CreateAndBuildFirstPass(context); break;
            case AotPointSprite: ret =  AssetBuilder<PointSprite*>::CreateAndBuildFirstPass(context); break;
            case AotLight: ret =  AssetBuilder<Light*>::CreateAndBuildFirstPass(context); break;
            case AotCompositeNode: ret =  AssetBuilder<CompositeGroup*>::CreateAndBuildFirstPass(context); break;
            case AotScene: ret =  AssetBuilder<Scene*>::CreateAndBuildFirstPass(context); break;
            case AotComposite: ret =  AssetBuilder<Scene*>::CreateAndBuildFirstPass(context); break;
            case AotReflectionCamera: ret =  AssetBuilder<ReflectionCamera*>::CreateAndBuildFirstPass(context); break;
            case AotLodNode: ret =  AssetBuilder<LodNode*>::CreateAndBuildFirstPass(context); break;
            case AotSkyBoxMesh:
            case AotMesh: ret =  AssetBuilder<Mesh*>::CreateAndBuildFirstPass(context); break;
            case AotStereoCamera: ret =  AssetBuilder<StereoCamera*>::CreateAndBuildFirstPass(context); break;
            case AotPlanarShadow: ret =  AssetBuilder<PlanarShadow*>::CreateAndBuildFirstPass(context); break;
            default: FEATSTD_DEBUG_FAIL(); break; // Shouldn't go here, all types must be covered
            }

            return ret;
        }

        bool AssetBuilder<Node*>::BuildSecondPass(Node* node, LoaderContext& context)
        {
            bool ret = false;
            AssetObjectType childType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            switch (childType) {
            case AotGroup: ret =  AssetBuilder<Group*>::BuildSecondPass(Dynamic_Cast<Group*>(node), context); break;
#ifdef CANDERA_3D_CANVAS_ENABLED
            case AotCanvas: ret = AssetBuilder<Canvas*>::BuildSecondPass(Dynamic_Cast<Canvas*>(node), context); break;
            case AotCanvasGroup: ret = AssetBuilder<CanvasGroup*>::BuildSecondPass(Dynamic_Cast<CanvasGroup*>(node), context); break;
            case AotCanvasSprite: ret = AssetBuilder<CanvasSprite*>::BuildSecondPass(Dynamic_Cast<CanvasSprite*>(node), context); break;
            case AotCanvasText: ret = AssetBuilder<CanvasText*>::BuildSecondPass(Dynamic_Cast<CanvasText*>(node), context); break;
#endif
            case AotCamera: ret =  AssetBuilder<Camera*>::BuildSecondPass(Dynamic_Cast<Camera*>(node), context); break;
            case AotMorphingMesh: ret =  AssetBuilder<MorphingMesh*>::BuildSecondPass(Dynamic_Cast<MorphingMesh*>(node), context); break;
            case AotBillboard: ret =  AssetBuilder<Billboard*>::BuildSecondPass(Dynamic_Cast<Billboard*>(node), context); break;
            case AotPointSprite: ret =  AssetBuilder<PointSprite*>::BuildSecondPass(Dynamic_Cast<PointSprite*>(node), context); break;
            case AotLight: ret =  AssetBuilder<Light*>::BuildSecondPass(Dynamic_Cast<Light*>(node), context); break;
            case AotCompositeNode: ret =  AssetBuilder<CompositeGroup*>::BuildSecondPass(Dynamic_Cast<CompositeGroup*>(node), context); break;
            case AotScene: ret =  AssetBuilder<Scene*>::BuildSecondPass(Dynamic_Cast<Scene*>(node), context); break;
            case AotComposite: ret =  AssetBuilder<Scene*>::BuildSecondPass(Dynamic_Cast<Scene*>(node), context); break;
            case AotReflectionCamera: ret =  AssetBuilder<ReflectionCamera*>::BuildSecondPass(Dynamic_Cast<ReflectionCamera*>(node), context); break;
            case AotLodNode: ret =  AssetBuilder<LodNode*>::BuildSecondPass(Dynamic_Cast<LodNode*>(node), context); break;
            case AotSkyBoxMesh:
            case AotMesh: ret =  AssetBuilder<Mesh*>::BuildSecondPass(Dynamic_Cast<Mesh*>(node), context); break;
            case AotStereoCamera: ret =  AssetBuilder<StereoCamera*>::BuildSecondPass(Dynamic_Cast<StereoCamera*>(node), context); break;
            case AotPlanarShadow: ret =  AssetBuilder<PlanarShadow*>::BuildSecondPass(Dynamic_Cast<PlanarShadow*>(node), context); break;
            default: FEATSTD_DEBUG_FAIL(); break; // Shouldn't go here, all types must be covered
            }

            return ret;
        }

        template <typename T>
        bool TDispose(Node* node)
        {
            if (node->IsTypeOf(T::GetTypeId())) {
                AssetBuilder<T*>::Dispose(Dynamic_Cast<T*>(node));
                return true;
            }

            return false;
        }

        void AssetBuilder<Node*>::Dispose(Node* node)
        {
            if (node != 0) {
                for (Node* child = node->GetFirstChild(); child != 0; child = child->GetNextSibling()) {
                    Dispose(child);
                }

                if ((!TDispose<Scene>(node)) &&
                    (!TDispose<StereoCamera>(node)) &&
                    (!TDispose<ReflectionCamera>(node)) &&
                    (!TDispose<Camera>(node)) &&
                    (!TDispose<CompositeGroup>(node)) &&
                    (!TDispose<MorphingMesh>(node)) &&
                    (!TDispose<Billboard>(node)) &&
                    (!TDispose<PointSprite>(node)) &&
                    (!TDispose<Light>(node)) &&
                    (!TDispose<LineList>(node)) &&
#ifdef CANDERA_3D_CANVAS_ENABLED
                    (!TDispose<CanvasSprite>(node)) &&
                    (!TDispose<CanvasText>(node)) &&
                    (!TDispose<Canvas>(node)) &&
                    (!TDispose<CanvasGroup>(node)) &&
#endif
                    (!TDispose<Group>(node)) &&
                    (!TDispose<LodNode>(node)) &&
                    (!TDispose<Mesh>(node)) &&
                    (!TDispose<PlanarShadow>(node))) {
                    FEATSTD_DEBUG_FAIL();    // Shouldn't go here, all types must be covered
                }
            }
        }
    }
}
