//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CompositeGroupAssetBuilder.h"

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine3D/Core/CompositeGroup.h>
#include <CanderaWidget/WidgetBase/WidgetBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/Composite3DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CompositeNodeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnchorDefinitionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnchorDefinitionCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/SceneAnimationInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CompositePropertyDefinitionCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CompositePropertyDefinitionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/DynamicItemCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/DynamicPropertyItemInfoCffReader.h>

#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProviderInternal.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/WidgetCollectionAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/AnimationBlendedPropertyListAssetReader.h>
#include <FeatStd/Util/StaticObject.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/SceneAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/CompositeGroupFinder.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/LayouterAssetBuilder.h>

namespace Candera {
    using namespace Diagnostics;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        class CompositeGroupWidgetCollectionBuilder: public WidgetCollectionAssetBuilderBase {
        public:
            CompositeGroupWidgetCollectionBuilder(CompositeGroup& compositeGroup):m_compositeGroup(compositeGroup) {}
        private:
            virtual bool AddWidget(WidgetBase* widget) override
            {
                return CompositeGroupFunctions::AddWidget(m_compositeGroup, widget);
            }

            virtual WidgetBase* GetFirstWidget() override
            {
                m_widgetIterator = m_compositeGroup.GetWidgetIterator();
                return GetNextWidget();
            }
            virtual WidgetBase* GetNextWidget() override
            {
                if (!m_widgetIterator.IsValid()) {
                    return 0;
                }

                return *m_widgetIterator++;
            }

            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
            CompositeGroup& m_compositeGroup;
            CompositeGroup::WidgetIterator m_widgetIterator;
            FEATSTD_MAKE_CLASS_UNCOPYABLE(CompositeGroupWidgetCollectionBuilder);
        };

        /** Smart helper class that stores the partent CompositeGroup in a static variable while buidling the current one.*/
        class CompositeGroupBuilderStack
        {
        public:
            CompositeGroupBuilderStack(DefaultAssetProviderInternal* assetProvider, CompositeGroup* currentCompositeGroup);
            ~CompositeGroupBuilderStack();

            CompositeGroup* GetPreviousCompositeGroupOnStack();
        private:
            DefaultAssetProviderInternal* m_assetProvider;
            CompositeGroup* m_oldCompositeGroup;
        };

        CompositeGroupBuilderStack::CompositeGroupBuilderStack(DefaultAssetProviderInternal* assetProvider, CompositeGroup* currentCompositeGroup):
            m_assetProvider(assetProvider),
            m_oldCompositeGroup(assetProvider->m_currentCompositeGroupBuild)
        {
            assetProvider->m_currentCompositeGroupBuild = currentCompositeGroup;
        }

        CompositeGroupBuilderStack::~CompositeGroupBuilderStack()
        {
            m_assetProvider->m_currentCompositeGroupBuild = m_oldCompositeGroup;
            m_assetProvider = 0;
            m_oldCompositeGroup = 0;
        }

        CompositeGroup* CompositeGroupBuilderStack::GetPreviousCompositeGroupOnStack()
        {
            return m_oldCompositeGroup;
        }

        CompositeGroup* CompositeInstanceBuilder::CreateAndBuildFirstPass(CffLoaderContext context)
        {
            //Create CompositeGroup
            CompositeGroup* result = Candera::Internal::CompositeGroupFunctions::Create(CFFReader::GetComposite3DIsCanvasComposite(context.handle)? CompositeGroup::CanvasGroupMode : CompositeGroup::GroupMode);
            if (result == 0) {
                FEATSTD_LOG_ERROR("Failed to create new CompositeGroup instance.");
                return 0;
            }

            //Stores the current CompositeGroup in a static variable of DefaultAssetProvider (used for later resolval of asset ids).
            CompositeGroupBuilderStack compositeGroupBuilderStack(context.provider->m_internal, result);

            if (compositeGroupBuilderStack.GetPreviousCompositeGroupOnStack() != 0) {
                CompositeGroupFunctions::CompositeStringIdentifier(*result).SetOwner(&CompositeGroupFunctions::CompositeStringIdentifier(*compositeGroupBuilderStack.GetPreviousCompositeGroupOnStack()));
            }

            result->SetName(AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(context.handle)));

            const AssetId& compositeGroupAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!compositeGroupAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Composite Group AssetId is not valid");
            }
            //Read children (first pass).
            if (!SceneNodeChildrenReader().ReadFirstPass(*result, context)) {
                FEATSTD_LOG_ERROR("Failed to read children of CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                Dispose(result);
                return 0;
            }

            //Define anchors.
            const AssetDataHandle& anchorCollectionHandle = CFFReader::GetComposite3DAnchors(context.handle);
            if (!anchorCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read anchor collection of CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                Dispose(result);
                return 0;
            }
            Int32 anchorCount = CFFReader::GetAnchorDefinitionCollectionChildrenCount(anchorCollectionHandle);
            for (Int32 anchorIndex = 0; anchorIndex < anchorCount; ++anchorIndex) {
                const AssetDataHandle& anchorDataHandle = CFFReader::GetAnchorDefinitionCollectionChildrenElementAt(anchorCollectionHandle, anchorIndex);
                if (!anchorDataHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read anchor no.%d of CompositeGroup " AssetIdLogStr, anchorIndex, AssetIdLogArgs(compositeGroupAssetId));
                    Dispose(result);
                    return 0;
                }
                const Char* anchorName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(anchorDataHandle));
                const AssetId& anchorAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(anchorDataHandle));
                if (!anchorAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Anchor AssetId is not valid");
                }
                const AssetId& anchorNodeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetAnchorDefinitionAnchorNode(anchorDataHandle));
                if (!anchorNodeAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Anchor Node AssetId is not valid");
                }
                Node* anchorNode = context.provider->GetNodeByAssetId(anchorNodeAssetId);
                if (anchorNode == 0) {
                    FEATSTD_LOG_ERROR("Failed to find anchor Node " AssetIdLogStr " for CompositeGroup with "
                        "AssetId: " AssetIdLogStr, AssetIdLogArgs(anchorNodeAssetId), AssetIdLogArgs(compositeGroupAssetId));
                    Dispose(result);
                    return 0;
                }

                if (!CompositeGroupFunctions::DefineAnchorPoint(*result, AssetIdFunctions::GetNodeId(anchorAssetId), anchorName, anchorNode)) {
                    FEATSTD_LOG_ERROR("Failed to define AnchorPoint " AssetIdLogStr " for CompositeGroup "
                        AssetIdLogStr, AssetIdLogArgs(anchorAssetId), AssetIdLogArgs(compositeGroupAssetId));
                    Dispose(result);
                    return 0;
                }
            }

            //Read widgets (first pass).
            CompositeGroupWidgetCollectionBuilder widgetCollectionBuilder(*result);
            const AssetDataHandle& widgetCollectionHandle = CFFReader::GetSceneWidgets(context.handle);
            if (!widgetCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read widget collection for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return 0;
            }
            CffLoaderContext widgetCollectionContext = context.Clone(widgetCollectionHandle);
            if (!widgetCollectionBuilder.CreateAndBuildFirstPass(widgetCollectionContext)) {
                FEATSTD_LOG_ERROR("Failed to load widget collection for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                Dispose(result);
                return 0;
            }

#if defined(CANDERA_LAYOUT_ENABLED)
            if (CFFReader::GetComposite3DIsLayoutEnabled(context.handle)) {
                const AssetDataHandle& layoutHandle = CFFReader::GetComposite3DLayout(context.handle);
                CffLoaderContext layouterContext = context.Clone(layoutHandle);
                result->SetLayouter(AssetBuilder<Candera::Layouter*>::CreateAndBuildFirstPass(layouterContext));
            }
#endif

            return result;
        }

        bool CompositeInstanceBuilder::BuildSecondPass(CompositeGroup* compositeGroup, CffLoaderContext context)
        {
            //Stores the current CompositeGroup in a static variable of DefaultAssetProvider (used for later resolval of asset ids).
            CompositeGroupBuilderStack compositeGroupBuilderStack(context.provider->m_internal, compositeGroup);

            const AssetId& compositeGroupAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!compositeGroupAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Composite Group AssetId is not valid");
            }
            //Read children (second pass).
            if (!SceneNodeChildrenReader().ReadSecondPass(*compositeGroup, context)) {
                FEATSTD_LOG_ERROR("Failed to read children of CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            //Read animations.
            bool result = true;
            Int32 animationCount = CFFReader::GetSceneSceneAnimationInfoCount(context.handle);
            for (Int animationIndex = 0; animationIndex < animationCount; ++animationIndex) {
                const AssetDataHandle& aiHandle = CFFReader::GetSceneSceneAnimationInfoElementAt(context.handle, animationIndex);
                if (!aiHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read AnimationPlayer data for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }
                const AssetId& animationAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetSceneAnimationInfoAnimation(aiHandle));
                if (!animationAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Animation AssetId is not valid");
                }
                Animation::AnimationPlayer::SharedPointer animation = context.provider->CreateCompositeAnimation(animationAssetId);
                if (animation == 0) {
                    FEATSTD_LOG_ERROR("Failed to load AnimationPlayer " AssetIdLogStr " for CompositeGroup " AssetIdLogStr
                        , AssetIdLogArgs(animationAssetId), AssetIdLogArgs(compositeGroupAssetId));
                    result = false;
                }
                else {
                    CffLoaderContext aiContext = context.Clone(aiHandle);
                    if (!SceneAnimationBlendedPropertyListAssetReader(animationAssetId, animation).Read(aiContext)) {
                        FEATSTD_LOG_ERROR("Failed to read properties of AnimationPlayer " AssetIdLogStr " for CompositeGroup " AssetIdLogStr
                            , AssetIdLogArgs(animationAssetId), AssetIdLogArgs(compositeGroupAssetId));
                        result = false;
                    }
                    CompositeGroupFunctions::AddAnimation(*compositeGroup, animation);
                }
            }
            if (!result) {
                FEATSTD_LOG_ERROR("Failed to load animation collection for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            //Read widgets (second pass).
            CompositeGroupWidgetCollectionBuilder widgetCollectionBuilder(*compositeGroup);
            const AssetDataHandle& widgetCollectionHandle = CFFReader::GetSceneWidgets(context.handle);
            if (!widgetCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read widget collection for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }
            CffLoaderContext widgetCollectionContext = context.Clone(widgetCollectionHandle);
            if (!widgetCollectionBuilder.BuildSecondPass(widgetCollectionContext)) {
                FEATSTD_LOG_ERROR("Failed to load widget collection for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            //Define property list.
            const AssetDataHandle& compositePropertiesCollectionHandle = CFFReader::GetComposite3DCompositeProperties(context.handle);
            if (!compositePropertiesCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read properties definition for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }
            CompositePropertyList propertyList;
            Int32 propertyCount = CFFReader::GetCompositePropertyDefinitionCollectionChildrenCount(compositePropertiesCollectionHandle);
            for (Int32 propertyIndex = 0; propertyIndex < propertyCount; ++propertyIndex) {
                const AssetDataHandle& propertyDataHandle = CFFReader::GetCompositePropertyDefinitionCollectionChildrenElementAt(compositePropertiesCollectionHandle, propertyIndex);
                if (!propertyDataHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read property no.%d definition for CompositeGroup " AssetIdLogStr, propertyIndex, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }

                Int32 pathLength = CFFReader::GetCompositePropertyDefinitionWidgetPathLength(propertyDataHandle);
                const AssetId& widgetId = AssetIdFunctions::GetAssetId(CFFReader::GetCompositePropertyDefinitionWidgetPathElementAt(propertyDataHandle, pathLength - 1));
                if (!widgetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Widget AssetId is not valid");
                }

                if (pathLength < 1) {
                    FEATSTD_LOG_ERROR("Path length to short for property definition no.%d for CompositeGroup " AssetIdLogStr, propertyIndex, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }

                WidgetBase* widget = 0;
                CompositeGroup* innerCompositeGroup = CompositeGroupFinder<Node, CompositeGroup>::GetCompositeGroup(compositeGroup, pathLength - 1, CFFReader::GetCompositePropertyDefinitionWidgetPath(propertyDataHandle));
                if (innerCompositeGroup != 0) {
                    if (widgetId.IsValid()) {
                        widget = innerCompositeGroup->GetWidget(AssetIdFunctions::GetNodeId(widgetId));
                    }
                }
                if (widget != 0) {

                    const Char* widgetPropName = CFFReader::GetCompositePropertyDefinitionWidgetPropertyName(propertyDataHandle);
                    const Char* propName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(propertyDataHandle));

                    MetaInfo::WidgetMetaInfo* widgetMetaInfo = widget->GetMetaInfo();
                    if (widgetMetaInfo != 0) {
                        MetaInfo::WidgetPropertyMetaInfo* propertyMetaInfo = widgetMetaInfo->LookupItem(widgetPropName);
                        if (propertyMetaInfo != 0) {
                            CompositePropertyList::Property property = {propName, widgetPropName, widget, propertyMetaInfo};
                            result = result && propertyList.m_list.Add(property);
                        }
                        else {
                            FEATSTD_LOG_ERROR("Failed to find Widget " AssetIdLogStr " PropertyMetaInfo for %s property", AssetIdLogArgs(widgetId), widgetPropName);
                        }
                    }
                    else {
                        FEATSTD_LOG_ERROR("Failed to find Widget " AssetIdLogStr " MetaInfo to access one of its properties", AssetIdLogArgs(widgetId));
                    }
                }
                else {
                    FEATSTD_LOG_ERROR("Failed to find Widget " AssetIdLogStr " to initialize one of its properties in CompositeGroup "
                        AssetIdLogStr, AssetIdLogArgs(widgetId), AssetIdLogArgs(compositeGroupAssetId));
                }
            }
            if (result) {
                if (!CompositePropertyList::SetPropertyList(compositeGroup, propertyList)) {
                    FEATSTD_LOG_ERROR("Failed to initialize property list of CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    result = false;
                }
            }

#if defined(CANDERA_LAYOUT_ENABLED)
            if (CFFReader::GetComposite3DIsLayoutEnabled(context.handle)) {
                const AssetDataHandle& layoutHandle = CFFReader::GetComposite3DLayout(context.handle);
                CffLoaderContext layouterContext = context.Clone(layoutHandle);
                if (!AssetBuilder<Layouter*>::BuildSecondPass(compositeGroup->GetLayouter(), layouterContext)) {
                    FEATSTD_LOG_ERROR("Failed to initialize layout properties of CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    result = false;
                }
            }
#endif

            return result;
        }

        void CompositeInstanceBuilder::Dispose(CompositeGroup* compositeInstance)
        {
            AssetBuilder<Node*>::Dispose(compositeInstance);
        }

        CompositeGroup* AssetBuilderBase<CompositeGroup*>::Create(const LoaderContext& context)
        {
            return context.provider->CreateCompositeGroupByAssetId(AssetIdFunctions::GetAssetId(CFFReader::GetCompositeNodeCompositeInterface(context.handle)));
        }

        void AssetBuilderBase<CompositeGroup*>::Dispose(const CompositeGroup* compositeGroup)
        {
            for (CompositeGroup::WidgetIterator iterator = compositeGroup->GetWidgetIterator(); iterator.IsValid(); ++iterator) {
                AssetBuilder<WidgetBase*>::Dispose(*iterator);
            }
        }

        bool AssetReaderBase<CompositeGroup>::ReadSecondPass(CompositeGroup& compositeGroup, const LoaderContext& context)
        {
            const AssetId& compositeGroupAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!compositeGroupAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Composite Group AssetId is not valid");
            }

            const Char* compositeIdentifier = CFFReader::GetCompositeNodeThisNode(context.handle);
            if (compositeIdentifier == 0) {
                FEATSTD_LOG_ERROR("Failed to read identifier of CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            if (context.handle.IsPersistent() == false) {
                SizeType compositeIdentifierLength = StringPlatform::Length(compositeIdentifier);
                Char* compositeLongName = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, compositeIdentifierLength + 1);
                if (compositeLongName == 0) {
                    FEATSTD_LOG_ERROR("Failed to allocate memory for identifier of CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }
                StringPlatform::Copy(compositeLongName, compositeIdentifier);
                compositeLongName[compositeIdentifierLength] = '\0';
                CompositeGroupFunctions::CompositeStringIdentifier(compositeGroup).SetId(compositeLongName, StringIdentifier::Disposer::Dispose);

                 // set owner id for widgets
                for (CompositeGroup::WidgetIterator widgetIt = compositeGroup.GetWidgetIterator(); widgetIt.IsValid(); ++widgetIt) {
                    (*widgetIt)->SetOwnerId(&CompositeGroupFunctions::CompositeStringIdentifier(compositeGroup));
                }

                // set owner id for animations
                for (CompositeGroup::AnimationIterator animationIt = compositeGroup.GetAnimationIterator(); animationIt.IsValid(); ++animationIt) {
                    (*animationIt)->SetOwnerId(&CompositeGroupFunctions::CompositeStringIdentifier(compositeGroup));
                }
            }
            else {
                CompositeGroupFunctions::CompositeStringIdentifier(compositeGroup).SetId(compositeIdentifier, 0);
            }

            const CompositePropertyList& propertyList = CompositePropertyList::GetPropertyList(&compositeGroup);
            const AssetDataHandle& nodePropHandle = CFFReader::GetCompositeNodeCompositeProperties(context.handle);
            if (!nodePropHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read properties data for CompositeGroup " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }
            Int32 propertyCount = CFFReader::GetDynamicItemDynamicPropertiesCount(nodePropHandle);
            for (Int32 propertyIndex = 0; propertyIndex < propertyCount; ++propertyIndex) {
                const AssetDataHandle& dynPropItemInfoHandle = CFFReader::GetDynamicItemDynamicPropertiesElementAt(nodePropHandle, CFFReader::CFF_DYNAMIC_PROPERTY_ITEM_INFO_SIZE, propertyIndex);
                if (!dynPropItemInfoHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read property no.%d data for CompositeGroup " AssetIdLogStr, propertyIndex, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }
                const Char* propertyName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetDynamicPropertyItemInfoName(dynPropItemInfoHandle));
                const Char* propValue = CFFReader::GetDynamicPropertyItemInfoValue(dynPropItemInfoHandle);

                const CompositePropertyList::Property* property = propertyList.GetProperty(propertyName);
                if (property != 0) {
                    if (!property->metaInfo->Set(property->widget, propValue)) {
                        FEATSTD_LOG_WARN("Vaue of property %s not set on CompositeGroup " AssetIdLogStr, propertyName, AssetIdLogArgs(compositeGroupAssetId));
                    }
                }
                else {
                    FEATSTD_LOG_WARN("Property %s definition not found for CompositeGroup " AssetIdLogStr, propertyName, AssetIdLogArgs(compositeGroupAssetId));
                }
            }

            return true;
        }

        CompositePropertyList& CompositePropertyList::DefaultPropertyList()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(CompositePropertyList, staticInstance);
            return staticInstance;
        }

        const CompositePropertyList& CompositePropertyList::GetPropertyList(const CompositeGroup* compositeGroup)
        {
            return compositeGroup->GetValue(CdaDynamicPropertyInstance(PropertyList));
        }

        bool CompositePropertyList::SetPropertyList(CompositeGroup* compositeGroup, const CompositePropertyList& propertyList)
        {
            return compositeGroup->SetValue(CdaDynamicPropertyInstance(PropertyList), propertyList);
        }

        bool CompositePropertyList::ClearPropertyList(CompositeGroup* compositeGroup)
        {
            return compositeGroup->ClearValue(CdaDynamicPropertyInstance(PropertyList));
        }

        const CompositePropertyList::Property* CompositePropertyList::GetProperty(const Char* name) const
        {
            for (Vector<CompositePropertyList::Property>::ConstIterator it = m_list.ConstBegin(); it != m_list.ConstEnd(); ++it) {
                const CompositePropertyList::Property& property = *it;
                if (property.name == name) {
                    return &property;
                }
            }

            return 0;
        }

        bool CompositePropertyList::operator==(const CompositePropertyList& other) const
        {
            return ((m_list.Size() == 0) && (other.m_list.Size() == 0));
        }
    }
}
