//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AppearanceAssetBuilder.h"
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AppearanceCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TextureCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProviderInternal.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/MaterialAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/TextureAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/ShaderAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/RenderModeAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/ShaderParamSetterAssetBuilder.h>

namespace Candera {
    using namespace MemoryManagement;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<Appearance>::ReadFirstPass(Appearance& appearance, LoaderContext& context)
        {
            AssetId shaderId = AssetIdFunctions::GetAssetId(CFFReader::GetAppearanceResolvedShaderProgram(context.handle));
            if (!shaderId.IsValid()) {
                FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no Shader attached!", AssetIdLogArgs(context.id));
            }
            else {
                if ((!CreateDependency(context, appearance, shaderId, SetShader))) {
                    FEATSTD_LOG_WARN("Failed to associate Shader " AssetIdLogStr " to Appearance "
                        AssetIdLogStr, AssetIdLogArgs(shaderId), AssetIdLogArgs(context.id));
                }
            }

            FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: only getter, no side effect")
            switch (CFFReader::GetAppearanceResolvedMaterialAttachmentType(context.handle)) {
            case NotSharedAttachment:
                {
                    LoaderContext materialContext = context.Clone(CFFReader::GetAppearanceResolvedMaterialAttachment(context.handle));
                    appearance.SetMaterial(AssetBuilder<SharedPointer<Material> >::CreateInOnePass(materialContext));
                }
                break;
            case SharedAttachment:
                {
                    const AssetId& materialAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetAppearanceResolvedMaterialAttachmentId(context.handle));
                    if (!materialAssetId.IsValid()) {
                        FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no Material attached!", AssetIdLogArgs(context.id));
                    }
                    else {
                        const Material::SharedPointer& material = context.provider->GetMaterialByAssetId(materialAssetId);
                        if (material == 0) {
                            FEATSTD_LOG_WARN("Failed to associate Material " AssetIdLogStr " to Appearance "
                                AssetIdLogStr, AssetIdLogArgs(materialAssetId), AssetIdLogArgs(context.id));
                        }
                        else {
                            appearance.SetMaterial(material);
                        }
                    }
                    break;
                }
            default:
                FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no Material attached!", AssetIdLogArgs(context.id));
                break;
            }

            switch (CFFReader::GetAppearanceResolvedRenderModeAttachmentType(context.handle)) {
            case NotSharedAttachment:
                {
                    LoaderContext renderModeContext = context.Clone(CFFReader::GetAppearanceResolvedRenderModeAttachment(context.handle));
                    appearance.SetRenderMode(AssetBuilder<SharedPointer<RenderMode> >::CreateInOnePass(renderModeContext));
                    break;
                }
            case SharedAttachment:
                {
                    const AssetId& renderModeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetAppearanceResolvedRenderModeAttachmentId(context.handle));
                    if (!renderModeAssetId.IsValid()) {
                        FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no RenderMode attached!", AssetIdLogArgs(context.id));
                    }
                    else {
                        const RenderMode::SharedPointer& renderMode = context.provider->GetRenderModeByAssetId(renderModeAssetId);
                        if (renderMode == 0) {
                            FEATSTD_LOG_WARN("Failed to associate RenderMode " AssetIdLogStr " to Appearance "
                                AssetIdLogStr, AssetIdLogArgs(renderModeAssetId), AssetIdLogArgs(context.id));
                        }
                        else {
                            appearance.SetRenderMode(renderMode);
                        }
                    }
                    break;
                }
            default:
                FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no RenderMode attached!", AssetIdLogArgs(context.id));
                break;
            }

            switch (CFFReader::GetAppearanceResolvedUniformSetterForAssetAttachmentType(context.handle)) {
            case NotSharedAttachment:
                {
                    LoaderContext shaderParamSetterContext = context.Clone(CFFReader::GetAppearanceResolvedUniformSetterForAssetAttachment(context.handle));
                    SharedPointer<GenericShaderParamSetter> shaderParamSetter = AssetBuilder<SharedPointer<GenericShaderParamSetter> >::CreateInOnePass(shaderParamSetterContext);
                    appearance.SetShaderParamSetter(shaderParamSetter);
                    break;
                }
            case SharedAttachment:
                {
                    const AssetId& shaderParamSetterAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetAppearanceResolvedUniformSetterForAssetAttachmentId(context.handle));
                    if (!shaderParamSetterAssetId.IsValid()) {
                        FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no ShaderParamSetter attached!", AssetIdLogArgs(context.id));
                    }
                    else {
                        const AbstractShaderParamSetter::SharedPointer& shaderParamSetter = context.provider->GetShaderParamSetterByAssetId(shaderParamSetterAssetId);
                        if (shaderParamSetter == 0) {
                            FEATSTD_LOG_WARN("Failed to associate ShaderParamSetter " AssetIdLogStr " to Appearance "
                                AssetIdLogStr, AssetIdLogArgs(shaderParamSetterAssetId), AssetIdLogArgs(context.id));
                        }
                        else {
                            appearance.SetShaderParamSetter(shaderParamSetter);
                        }
                    }
                    break;
                }
            default:
                FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no ShaderParamSetter attached!", AssetIdLogArgs(context.id));
                break;
            }

            const AssetDataHandle& textureCollectionHandle = CFFReader::GetAppearanceTextures(context.handle);
            if (textureCollectionHandle.IsValid()) {
                Int32 textureCount = CFFReader::GetTextureCollectionChildrenAttachmentCount(textureCollectionHandle);
                for (Int32 textureIndex = 0; textureIndex < textureCount; textureIndex++) {
                    SharedPointer<Texture> texture(0);
                    switch (CFFReader::GetTextureCollectionChildrenAttachmentTypeOfElementAt(textureCollectionHandle, textureIndex)) {
                    case NotSharedAttachment:
                        {
                            LoaderContext textureContext = context.Clone(CFFReader::GetTextureCollectionChildrenAttachmentElementAt(textureCollectionHandle, textureIndex));
                            texture = AssetBuilder<SharedPointer<Texture> >::CreateInOnePass(textureContext);
                            break;
                        }
                    case SharedAttachment:
                        {
                            const AssetId& textureAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTextureCollectionChildrenAttachmentIdOfElementAt(textureCollectionHandle, textureIndex));
                            if (!textureAssetId.IsValid()) {
                                FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no Texture attached with index %d!", AssetIdLogArgs(context.id), textureIndex);
                            }
                            else {
                                texture = context.provider->GetTextureByAssetId(textureAssetId);
                            }
                            break;
                        }
                    default:
                        FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no Texture attached with index %d!", AssetIdLogArgs(context.id), textureIndex);
                        break;
                    }

                    if (texture.PointsToNull()) {
                        FEATSTD_LOG_WARN("Failed to associate Texture to Appearance " AssetIdLogStr, AssetIdLogArgs(context.id));
                    }
                    else {
                        if (!appearance.SetTexture(texture, static_cast<UInt>(textureIndex))) {
                            FEATSTD_LOG_WARN("Failed to associate Texture to Appearance " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }
                }
            }
            else {
                FEATSTD_LOG_INFO("Appearance " AssetIdLogStr " has no Texture attached!", AssetIdLogArgs(context.id));
            }

            return true;
        }


        /******************************************************************************
         *  SetShader
         ******************************************************************************/
        bool AssetReaderBase<Appearance>::SetShader(Appearance& appearance, AssetId shaderId, DependencyList* dependencyList, const DefaultAssetProvider* provider)
        {
            MemoryManagement::SharedPointer<Shader> shader = provider->m_internal->m_shaderProvider.Get(shaderId, dependencyList);

            if (shader != 0) {
                appearance.SetShader(shader);
            }

            return (shader != 0);
        }

        SharedPointer<Appearance> AssetBuilderBase<SharedPointer<Appearance> >::Create(LoaderContext& /*context*/)
        {
            return Appearance::Create();
        }

        SharedPointer<MultiPassAppearance> AssetBuilderBase<SharedPointer<MultiPassAppearance> >::Create(LoaderContext& /*context*/)
        {
            return MultiPassAppearance::Create();
        }
    }
}
