//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationPropertySetterFactory3D.h"
#include <Candera/Engine3D/AnimationPropertySetters/NodePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/LightPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/CameraFovPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/CameraViewportPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/CameraScissorRectanglePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/CanvasVertexRectanglePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/GenericLodCriterionValuePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/ScaleNodePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/RotateNodePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/TranslateNodePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/AlphaNodePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/RenderingEnabledNodePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/MaterialColorPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/MaterialSpecularPowerPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/MorphWeightPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/ColorLightPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/UniformPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/FloatUniformPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/IntUniformPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/BoolUniformPropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/ScaleNodeRelativePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/RotateNodeRelativePropertySetter.h>
#include <Candera/Engine3D/AnimationPropertySetters/TranslateNodeRelativePropertySetter.h>
#include <Candera/Engine3D/Core/GenericValueLodCriterion.h>
#include <Candera/Engine3D/Core/Camera.h>
#ifdef CANDERA_3D_CANVAS_ENABLED
#include <Candera/Engine3D/Canvas/CanvasRenderable.h>
#endif
#include <Candera/Engine3D/Core/Light.h>
#include <Candera/Engine3D/Core/LodNode.h>
#include <Candera/Engine3D/Core/LodRenderStrategy.h>
#include <Candera/Engine3D/Core/MorphingMesh.h>
#include <Candera/Engine3D/Core/Material.h>
#include <Candera/Engine3D/ShaderParamSetters/ShaderParamSetter.h>
#include <Candera/EngineBase/Animation/AnimationPropertySetter.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <CanderaWidget/WidgetBase/WidgetBase.h>

namespace Candera {
    using namespace Candera::Internal;
    using namespace Animation;
    using namespace MemoryManagement;

    namespace Internal {

        AssetProvider* AnimationPropertySetterFactory3D::m_assetProvider = 0;

        // Node
        const Int32 c_Node_Low         = 100;
        const Int32 c_Node_High        = 999;
        const Int32 c_ScaleXyz         = c_Node_Low + 1;
        const Int32 c_ScaleX           = c_Node_Low + 2;
        const Int32 c_ScaleY           = c_Node_Low + 3;
        const Int32 c_ScaleZ           = c_Node_Low + 4;
        const Int32 c_RotateXyz        = c_Node_Low + 101;
        const Int32 c_RotateX          = c_Node_Low + 102;
        const Int32 c_RotateY          = c_Node_Low + 103;
        const Int32 c_RotateZ          = c_Node_Low + 104;
        const Int32 c_TranslateXyz     = c_Node_Low + 201;
        const Int32 c_TranslateX       = c_Node_Low + 202;
        const Int32 c_TranslateY       = c_Node_Low + 203;
        const Int32 c_TranslateZ       = c_Node_Low + 204;
        const Int32 c_Alpha            = c_Node_Low + 301;
        const Int32 c_RenderingEnabled = c_Node_Low + 401;

        // Material
        const Int32 c_Material_Low     = 1000;
        const Int32 c_Material_High    = 1999;
        const Int32 c_MatAmbientRgb    = c_Material_Low + 1;
        const Int32 c_MatAmbientR      = c_Material_Low + 2;
        const Int32 c_MatAmbientG      = c_Material_Low + 3;
        const Int32 c_MatAmbientB      = c_Material_Low + 4;
        const Int32 c_MatDiffuseRgb    = c_Material_Low + 100;
        const Int32 c_MatDiffuseR      = c_Material_Low + 101;
        const Int32 c_MatDiffuseG      = c_Material_Low + 102;
        const Int32 c_MatDiffuseB      = c_Material_Low + 103;
        const Int32 c_MatDiffuseA      = c_Material_Low + 104;
        const Int32 c_MatEmissiveRgb   = c_Material_Low + 200;
        const Int32 c_MatEmissiveR     = c_Material_Low + 201;
        const Int32 c_MatEmissiveG     = c_Material_Low + 202;
        const Int32 c_MatEmissiveB     = c_Material_Low + 203;
        const Int32 c_MatSpecularRgb   = c_Material_Low + 300;
        const Int32 c_MatSpecularR     = c_Material_Low + 301;
        const Int32 c_MatSpecularG     = c_Material_Low + 302;
        const Int32 c_MatSpecularB     = c_Material_Low + 303;
        const Int32 c_MatSpecularPower = c_Material_Low + 400;

        // Light
        const Int32 c_Light_Low        = 11000;
        const Int32 c_Light_High       = 14999;
        const Int32 c_LightAmbientRgb  = c_Light_Low + 1;
        const Int32 c_LightAmbientR    = c_Light_Low + 2;
        const Int32 c_LightAmbientG    = c_Light_Low + 3;
        const Int32 c_LightAmbientB    = c_Light_Low + 4;
        const Int32 c_LightDiffuseRgb  = c_Light_Low + 1000;
        const Int32 c_LightDiffuseR    = c_Light_Low + 1001;
        const Int32 c_LightDiffuseG    = c_Light_Low + 1002;
        const Int32 c_LightDiffuseB    = c_Light_Low + 1003;
        const Int32 c_LightSpecularRgb = c_Light_Low + 2000;
        const Int32 c_LightSpecularR   = c_Light_Low + 2001;
        const Int32 c_LightSpecularG   = c_Light_Low + 2002;
        const Int32 c_LightSpecularB   = c_Light_Low + 2003;

        // Camera
        const Int32 c_Camera_Low                    = 15000;
        const Int32 c_Camera_High                   = 15999;
        const Int32 c_CameraFov                     = c_Camera_Low + 1;
        const Int32 c_CameraViewport                = c_Camera_Low + 2;
        const Int32 c_CameraViewportLeft            = c_Camera_Low + 3;
        const Int32 c_CameraViewportTop             = c_Camera_Low + 4;
        const Int32 c_CameraViewportWidth           = c_Camera_Low + 5;
        const Int32 c_CameraViewportHeight          = c_Camera_Low + 6;
        const Int32 c_CameraScissorRectangle        = c_Camera_Low + 7;
        const Int32 c_CameraScissorRectangleLeft    = c_Camera_Low + 8;
        const Int32 c_CameraScissorRectangleTop     = c_Camera_Low + 9;
        const Int32 c_CameraScissorRectangleWidth   = c_Camera_Low + 10;
        const Int32 c_CameraScissorRectangleHeight  = c_Camera_Low + 11;

        // Level of detail
        const Int32 c_LodNode_Low      = 16000;
        const Int32 c_LodNode_High     = 16999;
        const Int32 c_LodCriterion     = c_Light_Low + 1;

        // Uniform
        const Int32 c_Uniform_Low      = 17000;
        const Int32 c_Uniform_High     = 17999;
        const Int32 c_Uniform          = c_Uniform_Low + 1;

        // Morphing
        const Int32 c_Morph_Low        = 18000;
        const Int32 c_Morph_High       = 18999;
        const Int32 c_MorphWeight      = c_Morph_Low + 1;

        // CanvasRenderable
        const Int32 c_Canvas_Low = 19000;
        const Int32 c_Canvas_High = 19999;
        const Int32 c_CanvasVertexRectangle = c_Canvas_Low + 1;
        const Int32 c_CanvasVertexRectangleLeft = c_Canvas_Low + 2;
        const Int32 c_CanvasVertexRectangleTop = c_Canvas_Low + 3;
        const Int32 c_CanvasVertexRectangleWidth = c_Canvas_Low + 4;
        const Int32 c_CanvasVertexRectangleHeight = c_Canvas_Low + 5;

        static bool CompileTime_CheckRanges()
        {
            FEATSTD_COMPILETIME_ASSERT(c_RenderingEnabled < c_Node_High);
            FEATSTD_COMPILETIME_ASSERT(c_MatSpecularPower < c_Material_High);
            FEATSTD_COMPILETIME_ASSERT(c_LightSpecularB < c_Light_High);
            FEATSTD_COMPILETIME_ASSERT(c_CameraFov < c_Camera_High);
            FEATSTD_COMPILETIME_ASSERT(c_LodCriterion < c_LodNode_High);
            FEATSTD_COMPILETIME_ASSERT(c_Uniform < c_Uniform_High);
            FEATSTD_COMPILETIME_ASSERT(c_MorphWeight < c_Morph_High);
            return true;
        }
        static bool rangesChecked = CompileTime_CheckRanges();

        static const PropertySetterAssociation propertySettersMapper[] = {
            { aps3dAlpha, 1, apcDefault, c_Alpha },
            { aps3dFov, 1, apcDefault, c_CameraFov },
            { aps3dCameraViewport, 4, apcInvalid, c_CameraViewport },
            { aps3dCameraViewport, 1, apcLeft, c_CameraViewportLeft },
            { aps3dCameraViewport, 1, apcTop, c_CameraViewportTop },
            { aps3dCameraViewport, 1, apcWidth, c_CameraViewportWidth },
            { aps3dCameraViewport, 1, apcHeight, c_CameraViewportHeight },
            { aps3dCameraScissorRectangle, 4, apcInvalid, c_CameraScissorRectangle },
            { aps3dCameraScissorRectangle, 1, apcLeft, c_CameraScissorRectangleLeft },
            { aps3dCameraScissorRectangle, 1, apcTop, c_CameraScissorRectangleTop },
            { aps3dCameraScissorRectangle, 1, apcWidth, c_CameraScissorRectangleWidth },
            { aps3dCameraScissorRectangle, 1, apcHeight, c_CameraScissorRectangleHeight },
            { aps3dCanvasRenderableVertexRectangle, 4, apcInvalid, c_CanvasVertexRectangle },
            { aps3dCanvasRenderableVertexRectangle, 1, apcLeft, c_CanvasVertexRectangleLeft },
            { aps3dCanvasRenderableVertexRectangle, 1, apcTop, c_CanvasVertexRectangleTop },
            { aps3dCanvasRenderableVertexRectangle, 1, apcWidth, c_CanvasVertexRectangleWidth },
            { aps3dCanvasRenderableVertexRectangle, 1, apcHeight, c_CanvasVertexRectangleHeight },
            { aps3dLodCriterion, 1, apcDefault, c_LodCriterion },
            { aps3dLightAmbient, 3, apcInvalid, c_LightAmbientRgb },
            { aps3dLightAmbient, 1, apcRed, c_LightAmbientR },
            { aps3dLightAmbient, 1, apcGreen, c_LightAmbientG },
            { aps3dLightAmbient, 1, apcBlue, c_LightAmbientB },
            { aps3dLightDiffuse, 3, apcInvalid, c_LightDiffuseRgb },
            { aps3dLightDiffuse, 1, apcRed, c_LightDiffuseR },
            { aps3dLightDiffuse, 1, apcGreen, c_LightDiffuseG },
            { aps3dLightDiffuse, 1, apcBlue, c_LightDiffuseB },
            { aps3dLightSpecular, 3, apcInvalid, c_LightSpecularRgb },
            { aps3dLightSpecular, 1, apcRed, c_LightSpecularR },
            { aps3dLightSpecular, 1, apcGreen, c_LightSpecularG },
            { aps3dLightSpecular, 1, apcBlue, c_LightSpecularB },
            { aps3dMatAmbient, 3, apcInvalid, c_MatAmbientRgb },
            { aps3dMatAmbient, 1, apcRed, c_MatAmbientR },
            { aps3dMatAmbient, 1, apcGreen, c_MatAmbientG },
            { aps3dMatAmbient, 1, apcBlue, c_MatAmbientB },
            { aps3dMatDiffuse, 3, apcInvalid, c_MatDiffuseRgb },
            { aps3dMatDiffuse, 1, apcRed, c_MatDiffuseR },
            { aps3dMatDiffuse, 1, apcGreen, c_MatDiffuseG },
            { aps3dMatDiffuse, 1, apcBlue, c_MatDiffuseB },
            { aps3dMatDiffuse, 1, apcAlpha, c_MatDiffuseA },
            { aps3dMatEmissive, 3, apcInvalid, c_MatEmissiveRgb },
            { aps3dMatEmissive, 1, apcRed, c_MatEmissiveR },
            { aps3dMatEmissive, 1, apcGreen, c_MatEmissiveG },
            { aps3dMatEmissive, 1, apcBlue, c_MatEmissiveB },
            { aps3dMatSpecular, 3, apcInvalid, c_MatSpecularRgb },
            { aps3dMatSpecular, 1, apcRed, c_MatSpecularR },
            { aps3dMatSpecular, 1, apcGreen, c_MatSpecularG },
            { aps3dMatSpecular, 1, apcBlue, c_MatSpecularB },
            { aps3dMatSpecularPower, 1, apcDefault, c_MatSpecularPower },
            { aps3dRenderingEnabled, 1, apcDefault, c_RenderingEnabled },
            { aps3dRotate, 3, apcInvalid, c_RotateXyz },
            { aps3dRotate, 1, apcX, c_RotateX },
            { aps3dRotate, 1, apcY, c_RotateY },
            { aps3dRotate, 1, apcZ, c_RotateZ },
            { aps3dScale, 3, apcInvalid, c_ScaleXyz },
            { aps3dScale, 1, apcX, c_ScaleX },
            { aps3dScale, 1, apcY, c_ScaleY },
            { aps3dScale, 1, apcZ, c_ScaleZ },
            { aps3dTranslate, 3, apcInvalid, c_TranslateXyz },
            { aps3dTranslate, 1, apcX, c_TranslateX },
            { aps3dTranslate, 1, apcY, c_TranslateY },
            { aps3dTranslate, 1, apcZ, c_TranslateZ },
            { aps3dUniform, 0, apcDefault, c_Uniform },
            { aps3dMorphWeight, 1, apcDefault, c_MorphWeight}
        };

        static const Int32 numMapperEntries = sizeof(propertySettersMapper) / sizeof(PropertySetterAssociation);

        template<typename BaseTypeSPtr, typename DerivedType>
        BaseTypeSPtr CreateBasePropertySetter()
        {
            return DerivedType::Create();
        }

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateCorrespondingSetter(AnimationPropertySetterId propertyId, const Char* propertyName, const Vector<AnimatedPropertyChannels>& apcList, AssetId objectId, PropertySetterType setterType)
        {
            Animation::AnimationPropertySetter::SharedPointer nullPtr;

            if (m_assetProvider == 0) {
                FEATSTD_LOG_ERROR("No AssetProvider associated to the AnimationPropertySetterFactory!");
                return nullPtr;
            }

            Int32 setterId = FindApsClassId(propertyId, apcList);
            return CreateAnimationPropertySetterByClassId(setterId, propertyName, objectId, setterType);
        }

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateAnimationPropertySetterByClassId(Int32 setterId, const Char* propertyName, AssetId objectId, PropertySetterType setterType)
        {
            if (setterId == -1) {
                return Animation::AnimationPropertySetter::SharedPointer();
            }

            Animation::AnimationPropertySetter::SharedPointer propertySetter;

            if ((setterId > c_Node_Low) && (setterId < c_Node_High)) {
                Node* node = m_assetProvider->GetNodeByAssetId(objectId);
                if (node != 0) {
                    propertySetter = CreateNodePropertySetter(setterId, node, setterType);
                }
                else{
                    FEATSTD_LOG_DEBUG("Node not found!");
                }
            }
            else if ((setterId > c_Material_Low) && (setterId < c_Material_High)) {
                SharedPointer<Material> material = m_assetProvider->GetMaterialByAssetId(objectId);
                if (material != 0) {
                    propertySetter = CreateMaterialPropertySetter(setterId, material, setterType);
                }
                else {
                    FEATSTD_LOG_DEBUG("Material not found!");
                }
            }
            else if ((setterId > c_Light_Low) && (setterId < c_Light_High)) {
                Node* result = m_assetProvider->GetNodeByAssetId(objectId);
                Light* light = Dynamic_Cast<Light*>(result);
                if (light != 0) {
                    propertySetter = CreateLightPropertySetter(setterId, light);
                }
                else {
                    FEATSTD_LOG_DEBUG("Light not found!");
                }
            }
            else if ((setterId > c_Camera_Low) && (setterId < c_Camera_High)) {
                Node* result = m_assetProvider->GetNodeByAssetId(objectId);
                Camera* camera = Dynamic_Cast<Camera*>(result);
                if (camera != 0) {
                    propertySetter = CreateCameraPropertySetter(setterId, camera);
                }
                else {
                    FEATSTD_LOG_DEBUG("Camera not found!");
                }
            }
            else if ((setterId > c_LodNode_Low) && (setterId < c_LodNode_High)) {
                Node* result = m_assetProvider->GetNodeByAssetId(objectId);
                LodNode* lodNode = Dynamic_Cast<LodNode*>(result);
                if (lodNode != 0) {
                    GenericLodCriterionValuePropertySetter::SharedPointer lodCriterionValuePropertySetter = GenericLodCriterionValuePropertySetter::Create();
                    if ((lodCriterionValuePropertySetter != 0) && (lodNode->GetLodRenderStrategy() != 0) && (lodNode->GetLodRenderStrategy()->GetLodCriterion() != 0)) {
                        LodCriterion* criterion = const_cast<LodCriterion*>(lodNode->GetLodRenderStrategy()->GetLodCriterion());
                        lodCriterionValuePropertySetter->SetGenericValueLodCriterion(Dynamic_Cast<GenericValueLodCriterion*>(criterion));
                    }
                    propertySetter = lodCriterionValuePropertySetter;
                }
                else {
                    FEATSTD_LOG_DEBUG("LodNode not found!");
                }
            }
            else if ((setterId > c_Uniform_Low) && (setterId < c_Uniform_High)) {
                MemoryManagement::SharedPointer<AbstractShaderParamSetter> result = m_assetProvider->GetShaderParamSetterByAssetId(objectId);
                if (result.PointsToNull()){
                    FEATSTD_LOG_DEBUG("GetShaderParamSetterByAssetId returns a null pointer.");
                }
                SharedPointer<ShaderParamSetter> shaderParamSetter = Dynamic_Cast<SharedPointer<ShaderParamSetter> >(result);
                propertySetter = CreateShaderParamSetterPropertySetter(propertyName, shaderParamSetter, setterType);
            }
            else if ((setterId > c_Morph_Low) && (setterId < c_Morph_High)) {
                Node* result = m_assetProvider->GetNodeByAssetId(objectId);
                MorphingMesh* morhpingMesh = Dynamic_Cast<MorphingMesh*>(result);
                if (morhpingMesh == 0) {
                    FEATSTD_LOG_DEBUG("Morphing Mesh not found!");
                }
                propertySetter = CreateMorphWeightPropertySetter(propertyName, morhpingMesh, setterType);
            }
            else if ((setterId > c_Canvas_Low) && (setterId < c_Canvas_High)) {
#ifdef CANDERA_3D_CANVAS_ENABLED
                Node* result = m_assetProvider->GetNodeByAssetId(objectId);
                CanvasRenderable* canvasRenderable = Dynamic_Cast<CanvasRenderable*>(result);
                if (0 != canvasRenderable) {
                    propertySetter = CreateCanvasRenderablePropertySetter(setterId, canvasRenderable);
                }
                else {
                    FEATSTD_LOG_DEBUG("Canvas Renderable not found!");
                }
#endif
            }
            else {
                //do nothing
            }

            return propertySetter;
        }

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateNodePropertySetter(Int32 setterId, Node* node, PropertySetterType setterType)
        {
            Animation::AnimationPropertySetter::SharedPointer newPropertySetter;
            switch (setterType) {
                case pstAbsolute: {
                    NodePropertySetter::SharedPointer nodePropertySetter;
                    switch (setterId) {
                        case c_ScaleXyz:
                            nodePropertySetter = ScaleNodePropertySetter::Create();
                            break;
                        case c_ScaleX:
                            nodePropertySetter = ScaleXNodePropertySetter::Create();
                            break;
                        case c_ScaleY:
                            nodePropertySetter = ScaleYNodePropertySetter::Create();
                            break;
                        case c_ScaleZ:
                            nodePropertySetter = ScaleZNodePropertySetter::Create();
                            break;
                        case c_RotateXyz:
                            nodePropertySetter = RotateNodePropertySetter::Create();
                            break;
                        case c_RotateX:
                            nodePropertySetter = RotateXNodePropertySetter::Create();
                            break;
                        case c_RotateY:
                            nodePropertySetter = RotateYNodePropertySetter::Create();
                            break;
                        case c_RotateZ:
                            nodePropertySetter = RotateZNodePropertySetter::Create();
                            break;
                        case c_TranslateXyz:
                            nodePropertySetter = TranslateNodePropertySetter::Create();
                            break;
                        case c_TranslateX:
                            nodePropertySetter = TranslateXNodePropertySetter::Create();
                            break;
                        case c_TranslateY:
                            nodePropertySetter = TranslateYNodePropertySetter::Create();
                            break;
                        case c_TranslateZ:
                            nodePropertySetter = TranslateZNodePropertySetter::Create();
                            break;
                        case c_Alpha:
                            nodePropertySetter = AlphaNodePropertySetter::Create();
                            break;
                        case c_RenderingEnabled:
                            nodePropertySetter = RenderingEnabledNodePropertySetter::Create();
                            break;
                        default:
                            break;
                    } //switch
                    if (nodePropertySetter != 0) {
                        nodePropertySetter->SetNode(node);
                        newPropertySetter = nodePropertySetter;
                        }
                    break;
                }
                case pstRelative: {
                    NodeRelativePropertySetter::SharedPointer nodeRelativePropertySetter;
                    switch (setterId) {
                        case c_ScaleXyz:
                            nodeRelativePropertySetter = ScaleNodeRelativePropertySetter::Create();
                            break;
                        case c_ScaleX:
                            nodeRelativePropertySetter = ScaleXNodeRelativePropertySetter::Create();
                            break;
                        case c_ScaleY:
                            nodeRelativePropertySetter = ScaleYNodeRelativePropertySetter::Create();
                            break;
                        case c_ScaleZ:
                            nodeRelativePropertySetter = ScaleZNodeRelativePropertySetter::Create();
                            break;
                        case c_RotateXyz:
                            nodeRelativePropertySetter = RotateNodeRelativePropertySetter::Create();
                            break;
                        case c_RotateX:
                            nodeRelativePropertySetter = RotateXNodeRelativePropertySetter::Create();
                            break;
                        case c_RotateY:
                            nodeRelativePropertySetter = RotateYNodeRelativePropertySetter::Create();
                            break;
                        case c_RotateZ:
                            nodeRelativePropertySetter = RotateZNodeRelativePropertySetter::Create();
                            break;
                        case c_TranslateXyz:
                            nodeRelativePropertySetter = TranslateNodeRelativePropertySetter::Create();
                            break;
                        case c_TranslateX:
                            nodeRelativePropertySetter = TranslateXNodeRelativePropertySetter::Create();
                            break;
                        case c_TranslateY:
                            nodeRelativePropertySetter = TranslateYNodeRelativePropertySetter::Create();
                            break;
                        case c_TranslateZ:
                            nodeRelativePropertySetter = TranslateZNodeRelativePropertySetter::Create();
                            break;
                        default:
                            break;
                    } //switch
                    if (nodeRelativePropertySetter != 0) {
                        nodeRelativePropertySetter->SetNode(node);
                        newPropertySetter = nodeRelativePropertySetter;
                        }
                    break;
                }
                default:
                    break;
            }

            return newPropertySetter;
        }

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateLightPropertySetter(Int32 setterId, Light* light)
        {
            LightPropertySetter::SharedPointer newPropertySetter;
            switch (setterId) {
                case c_LightAmbientRgb:
                    newPropertySetter = AmbientLightPropertySetter::Create();
                    break;
                case c_LightAmbientR:
                    newPropertySetter = AmbientRLightPropertySetter::Create();
                    break;
                case c_LightAmbientG:
                    newPropertySetter = AmbientGLightPropertySetter::Create();
                    break;
                case c_LightAmbientB:
                    newPropertySetter = AmbientBLightPropertySetter::Create();
                    break;
                case c_LightDiffuseRgb:
                    newPropertySetter = DiffuseLightPropertySetter::Create();
                    break;
                case c_LightDiffuseR:
                    newPropertySetter = DiffuseRLightPropertySetter::Create();
                    break;
                case c_LightDiffuseG:
                    newPropertySetter = DiffuseGLightPropertySetter::Create();
                    break;
                case c_LightDiffuseB:
                    newPropertySetter = DiffuseBLightPropertySetter::Create();
                    break;
                case c_LightSpecularRgb:
                    newPropertySetter = SpecularLightPropertySetter::Create();
                    break;
                case c_LightSpecularR:
                    newPropertySetter = SpecularRLightPropertySetter::Create();
                    break;
                case c_LightSpecularG:
                    newPropertySetter = SpecularGLightPropertySetter::Create();
                    break;
                case c_LightSpecularB:
                    newPropertySetter = SpecularBLightPropertySetter::Create();
                    break;

                default:
                    break;
            } //switch
            if (newPropertySetter != 0) {
                newPropertySetter->SetLight(light);
            }
            return newPropertySetter;
        }

        static void SetupCameraPropertySetter(Animation::AnimationPropertySetter::SharedPointer& propertySetter, Camera* camera, BaseCameraPropertySetter::SharedPointer(*createFctPtr)())
        {
            BaseCameraPropertySetter::SharedPointer cameraPropertySetter = createFctPtr();
            if ((!cameraPropertySetter.PointsToNull()) && ((0 != camera))) {
                cameraPropertySetter->SetCamera(camera);
                propertySetter = cameraPropertySetter;
            }
        }

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateCameraPropertySetter(Int32 setterId, Camera* camera)
        {
            Animation::AnimationPropertySetter::SharedPointer newPropertySetter;
            switch (setterId) {
            case c_CameraFov: {
                                  CameraFovPropertySetter::SharedPointer cameraFovPropertySetter = CameraFovPropertySetter::Create();
                                  if (cameraFovPropertySetter != 0) {
                                      cameraFovPropertySetter->SetCamera(camera);
                                  }
                                  newPropertySetter = cameraFovPropertySetter;
                                  break;
            }
            case c_CameraViewport:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraViewportPropertySetter>);
                break;
            case c_CameraViewportLeft:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraViewportLeftPropertySetter>);
                break;
            case c_CameraViewportTop:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraViewportTopPropertySetter>);
                break;
            case c_CameraViewportWidth:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraViewportWidthPropertySetter>);
                break;
            case c_CameraViewportHeight:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraViewportHeightPropertySetter>);
                break;
            case c_CameraScissorRectangle:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraScissorRectanglePropertySetter>);
                break;
            case c_CameraScissorRectangleLeft:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraScissorRectangleLeftPropertySetter>);
                break;
            case c_CameraScissorRectangleTop:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraScissorRectangleTopPropertySetter>);
                break;
            case c_CameraScissorRectangleWidth:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraScissorRectangleWidthPropertySetter>);
                break;
            case c_CameraScissorRectangleHeight:
                SetupCameraPropertySetter(newPropertySetter, camera, &CreateBasePropertySetter<BaseCameraPropertySetter::SharedPointer, CameraScissorRectangleHeightPropertySetter>);
                break;
            default:
                break;
            }
            return newPropertySetter;
        }

        Int32 AnimationPropertySetterFactory3D::FindApsClassId(AnimationPropertySetterId propertyId, const Vector<AnimatedPropertyChannels> &apcList)
        {
            Int32 foundIndex = -1;
            for (Int i = 0; i < numMapperEntries; i++) {
                if (propertyId == propertySettersMapper[i].propertyId) {
                    if ((static_cast<Int32>(apcList.Size()) == propertySettersMapper[i].channelCount) && (apcList.Size() == 1)) {
                        if (apcList[0] == propertySettersMapper[i].channel0Id) {
                            foundIndex = i;
                            break;
                        }
                    }
                    else if (((static_cast<Int32>(apcList.Size()) == propertySettersMapper[i].channelCount) && (apcList.Size() > 1)) || (propertySettersMapper[i].channelCount == 0)){
                        foundIndex = i;
                        break;
                    }
                    else {
                        //do nothing
                    }
                }
            }
            if (foundIndex == -1) {
                FEATSTD_LOG_INFO("Invalid PropertySetterId %d", propertyId);
                return -1;
            }

            return propertySettersMapper[foundIndex].setterClassId;
        }

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateMorphWeightPropertySetter(const Char* propertyName, MorphingMesh* node, PropertySetterType /*setterType*/)
        {
            if ((propertyName == 0) || (node == 0)) {
                return Animation::AnimationPropertySetter::SharedPointer();
            }

            Int32 index = -1;

            const Char* substrIndex = propertyName + 11; // Skip "MorphWeight" substr
            if (StringPlatform::ComparePartial(substrIndex, "[0]", 3) == 0) {
                index = 0;
            } else if (StringPlatform::ComparePartial(substrIndex, "[1]", 3) == 0) {
                index = 1;
            } else if (StringPlatform::ComparePartial(substrIndex, "[2]", 3) == 0) {
                index = 2;
            } else if (StringPlatform::ComparePartial(substrIndex, "[3]", 3) == 0) {
                index = 3;
            } else if (StringPlatform::ComparePartial(substrIndex, "[4]", 3) == 0) {
                index = 4;
            } else if (StringPlatform::ComparePartial(substrIndex, "[5]", 3) == 0) {
                index = 5;
            } else if (StringPlatform::ComparePartial(substrIndex, "[6]", 3) == 0) {
                index = 6;
            } else if (StringPlatform::ComparePartial(substrIndex, "[7]", 3) == 0) {
                index = 7;
            } else {
                //do nothing
            }

            if (index == -1) {
                return Animation::AnimationPropertySetter::SharedPointer();
            }

            MorphWeightPropertySetter::SharedPointer newPropertySetter = MorphWeightPropertySetter::Create();

            if (newPropertySetter != 0) {
                newPropertySetter->SetMorphingMesh(node);
                newPropertySetter->SetWeightIndex(index);
            }

            return newPropertySetter;
        }

        static void SetupCanvasRectanglePropertySetter(Animation::AnimationPropertySetter::SharedPointer& propertySetter, CanvasRenderable* canvasRenderable, BaseCanvasRenderablePropertySetter::SharedPointer(*createFctPtr)())
        {
            BaseCanvasRenderablePropertySetter::SharedPointer canvasRenderablePropertySetter = createFctPtr();
            if ((!canvasRenderablePropertySetter.PointsToNull()) && ((0 != canvasRenderable))) {
                canvasRenderablePropertySetter->SetCanvasRenderable(canvasRenderable);
                propertySetter = canvasRenderablePropertySetter;
            }
        }

#ifdef CANDERA_3D_CANVAS_ENABLED
        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateCanvasRenderablePropertySetter(Int32 setterId, CanvasRenderable* canvasRenderable)
        {
            Animation::AnimationPropertySetter::SharedPointer newPropertySetter;
            switch (setterId) {
            case c_CanvasVertexRectangle:
                SetupCanvasRectanglePropertySetter(newPropertySetter, canvasRenderable, &CreateBasePropertySetter<BaseCanvasRenderablePropertySetter::SharedPointer, CanvasVertexRectanglePropertySetter>);
                break;
            case c_CanvasVertexRectangleLeft:
                SetupCanvasRectanglePropertySetter(newPropertySetter, canvasRenderable, &CreateBasePropertySetter<BaseCanvasRenderablePropertySetter::SharedPointer, CanvasVertexRectangleLeftPropertySetter>);
                break;
            case c_CanvasVertexRectangleTop:
                SetupCanvasRectanglePropertySetter(newPropertySetter, canvasRenderable, &CreateBasePropertySetter<BaseCanvasRenderablePropertySetter::SharedPointer, CanvasVertexRectangleTopPropertySetter>);
                break;
            case c_CanvasVertexRectangleWidth:
                SetupCanvasRectanglePropertySetter(newPropertySetter, canvasRenderable, &CreateBasePropertySetter<BaseCanvasRenderablePropertySetter::SharedPointer, CanvasVertexRectangleWidthPropertySetter>);
                break;
            case c_CanvasVertexRectangleHeight:
                SetupCanvasRectanglePropertySetter(newPropertySetter, canvasRenderable, &CreateBasePropertySetter<BaseCanvasRenderablePropertySetter::SharedPointer, CanvasVertexRectangleHeightPropertySetter>);
                break;
            default:
                break;
            }
            return newPropertySetter;
        }
#endif

        class EmptyNode : public Node {
        public:
            static EmptyNode* Create() { return EmptyNode::Create(); }
        protected:
            virtual void DisposeSelf() { FEATSTD_DELETE(this); }
            virtual void Render() {}
            virtual Node* Clone() const { return 0; }
        };

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateMaterialPropertySetter(Int32 setterId, SharedPointer<Material> material, PropertySetterType /*setterType*/)
        {
            MaterialPropertySetter::SharedPointer materialPropertySetter;

            switch (setterId) {
                case c_MatAmbientRgb:
                    materialPropertySetter = MaterialAmbientPropertySetter::Create();
                    break;
                case c_MatAmbientR:
                    materialPropertySetter = MaterialAmbientRPropertySetter::Create();
                    break;
                case c_MatAmbientG:
                    materialPropertySetter = MaterialAmbientGPropertySetter::Create();
                    break;
                case c_MatAmbientB:
                    materialPropertySetter = MaterialAmbientBPropertySetter::Create();
                    break;
                case c_MatDiffuseRgb:
                    materialPropertySetter = MaterialDiffusePropertySetter::Create();
                    break;
                case c_MatDiffuseR:
                    materialPropertySetter = MaterialDiffuseRPropertySetter::Create();
                    break;
                case c_MatDiffuseG:
                    materialPropertySetter = MaterialDiffuseGPropertySetter::Create();
                    break;
                case c_MatDiffuseB:
                    materialPropertySetter = MaterialDiffuseBPropertySetter::Create();
                    break;
                case c_MatDiffuseA:
                    materialPropertySetter = MaterialDiffuseAPropertySetter::Create();
                    break;
                case c_MatEmissiveRgb:
                    materialPropertySetter = MaterialEmissivePropertySetter::Create();
                    break;
                case c_MatEmissiveR:
                    materialPropertySetter = MaterialEmissiveRPropertySetter::Create();
                    break;
                case c_MatEmissiveG:
                    materialPropertySetter = MaterialEmissiveGPropertySetter::Create();
                    break;
                case c_MatEmissiveB:
                    materialPropertySetter = MaterialEmissiveBPropertySetter::Create();
                    break;
                case c_MatSpecularRgb:
                    materialPropertySetter = MaterialSpecularPropertySetter::Create();
                    break;
                case c_MatSpecularR:
                    materialPropertySetter = MaterialSpecularRPropertySetter::Create();
                    break;
                case c_MatSpecularG:
                    materialPropertySetter = MaterialSpecularGPropertySetter::Create();
                    break;
                case c_MatSpecularB:
                    materialPropertySetter = MaterialSpecularBPropertySetter::Create();
                    break;
                case c_MatSpecularPower:
                    materialPropertySetter = MaterialSpecularPowerPropertySetter::Create();
                    break;
                default:
                    break;
            }
            if (materialPropertySetter != 0) {
                materialPropertySetter->SetMaterial(material);
            }

            return materialPropertySetter;
        }

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory3D::CreateShaderParamSetterPropertySetter(const Char* uniformName, SharedPointer<ShaderParamSetter> shaderParamSetter, PropertySetterType/* setterType*/)
        {
            UniformPropertySetter::SharedPointer uniformPropertySetter;

            if (shaderParamSetter != 0) {
                Shader::UniformType uniformType;
                if (shaderParamSetter->GetUniformType(uniformName, uniformType)) {
                    switch (uniformType) {
                        case Shader::Float: uniformPropertySetter = FloatUniformPropertySetter::Create(); break;
                        case Shader::FloatVec2: uniformPropertySetter = FloatVec2UniformPropertySetter::Create(); break;
                        case Shader::FloatVec3: uniformPropertySetter = FloatVec3UniformPropertySetter::Create(); break;
                        case Shader::FloatVec4: uniformPropertySetter = FloatVec4UniformPropertySetter::Create(); break;

                        case Shader::Integer: uniformPropertySetter = IntUniformPropertySetter::Create(); break;
                        case Shader::IntegerVec2: uniformPropertySetter = IntVec2UniformPropertySetter::Create(); break;
                        case Shader::IntegerVec3: uniformPropertySetter = IntVec3UniformPropertySetter::Create(); break;
                        case Shader::IntegerVec4: uniformPropertySetter = IntVec4UniformPropertySetter::Create(); break;

                        case Shader::Bool: uniformPropertySetter = BoolUniformPropertySetter::Create(); break;
                        case Shader::BoolVec2: uniformPropertySetter = BoolVec2UniformPropertySetter::Create(); break;
                        case Shader::BoolVec3: uniformPropertySetter = BoolVec3UniformPropertySetter::Create(); break;
                        case Shader::BoolVec4: uniformPropertySetter = BoolVec4UniformPropertySetter::Create(); break;

                        case Shader::FloatMat2: uniformPropertySetter = FloatMat2UniformPropertySetter::Create(); break;
                        case Shader::FloatMat3: uniformPropertySetter = FloatMat3UniformPropertySetter::Create(); break;
                        case Shader::FloatMat4: uniformPropertySetter = FloatMat4UniformPropertySetter::Create(); break;

                        case Shader::Sampler2D: uniformPropertySetter = Sampler2DUniformPropertySetter::Create(); break;
                        case Shader::SamplerCube:  uniformPropertySetter = SamplerCubeUniformPropertySetter::Create(); break;

                        default:
                            break;
                    }
                }
            }

            if (uniformPropertySetter != 0) {
                uniformPropertySetter->SetShaderParamSetter(MemoryManagement::SharedPointer<ShaderParamSetter>(shaderParamSetter));
                uniformPropertySetter->SetUniformIndex(0);
                uniformPropertySetter->SetUniformName(uniformName);
            }

            return uniformPropertySetter;
        }
    }
}

