//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Scene2DContextAssetBuilder.h"
#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/Scene2DAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/WidgetBaseAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/SceneCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;
    using namespace MetaInfo;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<Scene2DContext>::ReadSecondPass(Scene2DContext& sceneContext, LoaderContext& context)
        {
            Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetSceneTheme(context.handle));
            if (!result.IsValid()) {
                FEATSTD_LOG_DEBUG("Scene Theme AssetId is not valid");
            }
            TemporaryTheme temporaryTheme(*(context.provider), (result));

            Candera::Scene2D* scene = sceneContext.GetScene();
            if ((scene == 0) || (!AssetReader<Scene2D>::ReadSecondPass(*scene, context))) {
                FEATSTD_LOG_WARN("Failed to load scene tree for Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            return true;
        }

        Scene2DContext* AssetBuilderBase<Scene2DContext*>::Create(LoaderContext& context)
        {
            context.id = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!context.id.IsValid()) {
                FEATSTD_LOG_DEBUG("AssetId is not valid");
            }
            Scene2DContext* sceneContext = ASSETLOADER_TRANSIENT_NEW(Scene2DContext);
            if (sceneContext == 0) {
                FEATSTD_LOG_WARN("Failed to allocate memory for SceneContext.");
                return 0;
            }

            Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetSceneTheme(context.handle));
            if (!result.IsValid()) {
                FEATSTD_LOG_DEBUG("Scene Theme AssetId is not valid");
            }
            TemporaryTheme temporaryTheme(*(context.provider), (result));

            if (context.reloadList == 0) {
                context.reloadList = &sceneContext->m_reloadList;
            }
            context.iterativeLoad = DefaultAssetProvider::GetInstance().IsIterativeLoadingEnabled();

            Node2D* node = AssetBuilder<Node2D*>::CreateAndBuildFirstPass(context);
            if (node == 0) {
                FEATSTD_LOG_WARN("Failed to create scene tree for Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                ASSETLOADER_DELETE(sceneContext);
                return 0;
            }

            Scene2D* scene = Dynamic_Cast<Scene2D*>(node);
            if (scene == 0) {
                FEATSTD_LOG_ERROR("Type described in asset differs from actual type for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                AssetBuilder<Node2D*>::Dispose(node);
                ASSETLOADER_DELETE(sceneContext);
                return 0;
            }

            sceneContext->SetScene(scene);

            return sceneContext;
        }

        void Scene2DContextAssetBuilder::Dispose(Scene2DContext* sceneContext)
        {
            AssetReaderBase<SceneContextBase>::Dispose(sceneContext);
            Node2DAssetBuilder::Dispose(sceneContext->GetScene());
            ASSETLOADER_DELETE(sceneContext);
        }

        bool Scene2DContextAssetBuilder::IsValid(const Scene2DContext* sceneContext, LoaderContext& context)
        {
            Id currentThemeId = context.provider->GetCurrentThemeId();
            if (currentThemeId != 0) {
                return (sceneContext->GetLoadedThemeId() == currentThemeId);
            }

            return (sceneContext->GetLoadedThemeId() == sceneContext->GetDefaultThemeId());
        }
    }
}
