//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Node2DAssetBuilder.h"
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/Layout.h>
#endif
#include <Candera/Engine2D/Core/CompositeGroup2D.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/Node2DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/Anchor2DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/DynamicPropertyItemInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/ScriptComponentCollectionAssetReader.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/Group2DAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/RenderNodeAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/Camera2DAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/TextNode2DAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/Scene2DAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/CompositeGroup2DAssetBuilder.h>

namespace Candera {
    using namespace DynamicProperties;
    using namespace Diagnostics;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool Node2DChildrenReader::ReadFirstPass(Node2D& node, CffLoaderContext& context) const
        {
            AssetObjectType nodeType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            if (nodeType == AotCompositeNode2D) {
                CompositeGroup2D* compositeGroup = Dynamic_Cast<CompositeGroup2D*>(&node);
                if (compositeGroup == 0) {
                    FEATSTD_LOG_ERROR("Type described in asset differs from actual type for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                Int32 childernCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childernCount; ++childIndex) {
                    const AssetDataHandle& anchorHandle = GetChildDataHandle(context, childIndex);
                    if (!anchorHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read anchor for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    AssetObjectType anchorType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(anchorHandle));
                    if (anchorType != AotAnchor2D) {
                        FEATSTD_LOG_ERROR("Type described in asset differs from reqyured Anchor type for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    const AssetId& anchorNodeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(anchorHandle));
                    if (!anchorNodeAssetId.IsValid()) {
                        FEATSTD_LOG_DEBUG("Anchestor Node AssetId is not valid");
                    }
                    Id anchorId = AssetIdFunctions::GetNodeId(anchorNodeAssetId);
                    Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetAnchor2DAnchorDefinition(anchorHandle));
                    if (!result.IsValid()) {
                        FEATSTD_LOG_DEBUG("Anchor AssetId is not valid");
                    }
                    bool updateResult = CompositeGroup2DFunctions::UpdateAnchorPointId(
                        *compositeGroup,
                        AssetIdFunctions::GetNodeId(result),
                        anchorId);
                    Node2D* anchorNode = compositeGroup->GetAnchorPointNode(anchorId);
                    if ((!updateResult) || (anchorNode == 0)) {
                        FEATSTD_LOG_ERROR("Failed to set anchor for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    CffLoaderContext anchorContext = context.Clone(anchorHandle);
                    if (!ReadFirstPass(*anchorNode, anchorContext)) {
                        FEATSTD_LOG_WARN("Failed to read AnchorNode2D for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                }
            }
            else {
                Int32 childernCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childernCount; ++childIndex) {
                    const AssetDataHandle& childHandle = GetChildDataHandle(context, childIndex);
                    if (!childHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read child for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                    CffLoaderContext childContext = context.Clone(childHandle);
                    Node2D* child = AssetBuilder<Node2D*>::CreateAndBuildFirstPass(childContext);
                    if (child == 0) {
                        FEATSTD_LOG_WARN("Failed to create child for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                    }
                    else {
                        if (!node.AddChild(child)) {
                            FEATSTD_LOG_WARN("Failed to add child for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }
                }
            }

#ifdef CANDERA_SCRIPTING_ENABLED
            if ((nodeType != AotScene2D) && (nodeType != AotComposite2D)) {
                if (!ScriptComponentCollectionAssetReader::ReadScriptComponentCollectionData(node, context)) {
                    FEATSTD_LOG_WARN("Script component collection not completely attached to the node.");
                }
            }
#endif

            return true;
        }

        bool Node2DChildrenReader::ReadSecondPass(Node2D& node, CffLoaderContext& context) const
        {
            AssetObjectType nodeType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            if (nodeType == AotCompositeNode2D) {
                CompositeGroup2D* compositeGroup = Dynamic_Cast<CompositeGroup2D*>(&node);
                if (compositeGroup == 0) {
                    FEATSTD_LOG_ERROR("Type described in asset differs from actual type for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                Int32 childernCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childernCount; ++childIndex) {
                    const AssetDataHandle& anchorHandle = GetChildDataHandle(context, childIndex);
                    if (!anchorHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read anchor at index %d for CompositeGroup2D " AssetIdLogStr, childIndex, AssetIdLogArgs(context.id));
                        return false;
                    }

                    const AssetId& anchorAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(anchorHandle));
                    if (!anchorAssetId.IsValid()) {
                        FEATSTD_LOG_ERROR("CompositeGroup2D " AssetIdLogStr " has no anchor attached at index %d", AssetIdLogArgs(context.id), childIndex);
                        return false;
                    }
                    Node2D* anchorNode = compositeGroup->GetAnchorPointNode(AssetIdFunctions::GetNodeId(anchorAssetId));
                    if (anchorNode == 0) {
                        FEATSTD_LOG_ERROR("Failed to retrieve anchor from CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }

                    CffLoaderContext anchorContext = context.Clone(anchorHandle);
                    if (!ReadSecondPass(*anchorNode, anchorContext)) {
                        FEATSTD_LOG_WARN("Failed to read AnchorNode2D for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                }
            }
            else {
                Int32 childrenCount = GetChildrenCount(context);
                for (Int32 childIndex = 0; childIndex < childrenCount; ++childIndex) {
                    const AssetDataHandle& childHandle = GetChildDataHandle(context, childIndex);
                    if (!childHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Failed to read child for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                    CffLoaderContext childContext = context.Clone(childHandle);
                    Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(childHandle));
                    if (!result.IsValid()) {
                        FEATSTD_LOG_WARN("AssetId is not valid");
                    }
                    Node2D* child = node.GetChild(AssetIdFunctions::GetNodeId(result));
                    if (child != 0) {
                        if (!AssetBuilder<Node2D*>::BuildSecondPass(child, childContext)) {
                            FEATSTD_LOG_WARN("Failed to resolve child for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }
                    }
                    else {
                        FEATSTD_LOG_WARN("Failed to retrieve child of Node2D " AssetIdLogStr " for resolving.", AssetIdLogArgs(context.id));
                    }
                }
            }
            return true;
        }

        class NodeNode2DChildrenReader: public Node2DChildrenReader {
        protected:
            virtual Int32 GetChildrenCount(CffLoaderContext& context) const
            { return CFFReader::GetNode2DChildrenCount(context.handle); }

            virtual AssetDataHandle GetChildDataHandle(CffLoaderContext& context, Int32 index) const
            { return CFFReader::GetNode2DChildrenElementAt(context.handle, index); }
        };

        bool AssetReaderBase<Node2D>::ReadFirstPass(Node2D& node, LoaderContext& context)
        {
            node.SetRenderingEnabled(CFFReader::GetNode2DIsRenderingEnabled(context.handle));
            node.SetRenderOrderRank(static_cast<Int16>(CFFReader::GetNode2DRenderOrderRank(context.handle)));
            node.SetScopeMask(ScopeMask(CFFReader::GetNode2DScopeBits(context.handle)));

            if (!NodeNode2DChildrenReader().ReadFirstPass(node, context)) {
                FEATSTD_LOG_WARN("Failed to load children for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            return true;
        }

        bool AssetReaderBase<Node2D>::ReadSecondPass(Node2D& node, LoaderContext& context)
        {
            if (!NodeNode2DChildrenReader().ReadSecondPass(node, context)) {
                FEATSTD_LOG_WARN("Failed to load children for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

#if defined(CANDERA_LAYOUT_ENABLED)
            Layouter* layouter = node.GetLayouter();
            Layouter* parentLayouter = (node.GetParent() != 0) ? node.GetParent()->GetLayouter() : 0;

            if (layouter == 0) {
                layouter = DefaultLayouter::GetInstance();
            }
            if (parentLayouter == 0) {
                parentLayouter = DefaultLayouter::GetInstance();
            }

            //Move into a separate function when property iteration will be generic in CFFReader::
            Int32 layoutPropertyCount = CFFReader::GetNodeLayoutPropertiesCount(context.handle);
            for (Int32 layoutPropertyIndex = 0; layoutPropertyIndex < layoutPropertyCount; ++layoutPropertyIndex) {
                const AssetDataHandle& layoutPropertiesHandle = CFFReader::GetNodeLayoutPropertiesElementAt(context.handle, CFFReader::CFF_DYNAMIC_PROPERTY_ITEM_INFO_SIZE, layoutPropertyIndex);
                LoaderContext layoutPropertyContext = context.Clone(layoutPropertiesHandle);
                bool isPropertySet = ReadDynamicProperty(layouter->GetPropertyHierarchy(), node, layoutPropertyContext);
                if (!isPropertySet) {
                    if (!ReadDynamicProperty(parentLayouter->GetPropertyHierarchy(), node, layoutPropertyContext)) {
                        FEATSTD_LOG_WARN("Failed to load dynamic property for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                        return false;
                    }
                }
            }

#endif
            //Move into a separate function when property iteration will be generic in CFFReader::
            Int32 dynamicPropertyCount = CFFReader::GetNode2DDynamicPropertiesCount(context.handle);
            for (Int32 dynamicPropertyIndex = 0; dynamicPropertyIndex < dynamicPropertyCount; ++dynamicPropertyIndex) {
                const AssetDataHandle& dynamicPropertiesHandle = CFFReader::GetNode2DDynamicPropertiesElementAt(context.handle, CFFReader::CFF_DYNAMIC_PROPERTY_ITEM_INFO_SIZE, dynamicPropertyIndex);
                LoaderContext dynamicPropertyContext = context.Clone(dynamicPropertiesHandle);
                if (!ReadDynamicProperty(node.GetPropertyHierarchy(), node, dynamicPropertyContext)) {
                    FEATSTD_LOG_WARN("Failed to load dynamic property for Node2D " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }
            }

            return true;
        }

        bool AssetReaderBase<Node2D>::ReadDynamicProperty(const DynamicProperties::PropertyHierarchyNode& phn, DynamicProperties::DynamicPropertyHost& host, LoaderContext& context)
        {
            if (!context.IsValid()) {
                FEATSTD_LOG_ERROR("Asset error, missing handle for layouter properties.");
                return false;
            }
            const Char* propName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetDynamicPropertyItemInfoName(context.handle));
            const Char* propValue = CFFReader::GetDynamicPropertyItemInfoValue(context.handle);
            DynamicPropertyBase* searchedDynamicProperty = 0;
            for (const DynamicProperties::PropertyHierarchyNode* phnPtr = &phn; (phnPtr != 0) && (searchedDynamicProperty == 0); phnPtr = phnPtr->GetParent()) {
                DynamicPropertyBase* dynamicProp = phnPtr->GetProperties();
                for (bool found = false; ((dynamicProp != 0) && (!found)); dynamicProp = dynamicProp->GetNext()){
                    found = (StringPlatform::CompareStrings(dynamicProp->GetPropertyName(), propName) == 0);
                    if (found) {
                        searchedDynamicProperty = dynamicProp;
                    }
                }
            }

            if (searchedDynamicProperty == 0) {
                return false;
            }

            if (!searchedDynamicProperty->ConvertFromString(host, propValue)) {
                FEATSTD_LOG_ERROR("Failed to set property %s with value %s", propName, propValue);
                return false;
            }

            return true;
        }

        Node2D* Node2DAssetBuilder::CreateAndBuildFirstPass(LoaderContext& context)
        {
            FEATSTD_LOG_DEBUG("Creating node: %s", AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(context.handle)));
            Node2D* ret = 0;
            AssetObjectType childType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            switch (childType) {
            case AotGroup2D: ret = AssetBuilder<Group2D*>::CreateAndBuildFirstPass(context); break;
            case AotRenderNode: ret = AssetBuilder<RenderNode*>::CreateAndBuildFirstPass(context); break;
            case AotTextNode2D: ret = AssetBuilder<TextNode2D*>::CreateAndBuildFirstPass(context); break;
            case AotCamera2D: ret =  AssetBuilder<Camera2D*>::CreateAndBuildFirstPass(context); break;
            case AotCompositeNode2D: ret = AssetBuilder<CompositeGroup2D*>::CreateAndBuildFirstPass(context); break;
            case AotComposite2D: ret =   AssetBuilder<Scene2D*>::CreateAndBuildFirstPass(context); break;
            case AotScene2D: ret =  AssetBuilder<Scene2D*>::CreateAndBuildFirstPass(context); break;
            default: FEATSTD_DEBUG_FAIL(); break; // Shouldn't go here, all types must be covered
            }

            return ret;
        }

        bool Node2DAssetBuilder::BuildSecondPass(Node2D* node, LoaderContext& context)
        {
            bool ret = false;
            AssetObjectType childType = static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(context.handle));
            switch (childType) {
            case AotGroup2D: ret =  AssetBuilder<Group2D*>::BuildSecondPass(Dynamic_Cast<Group2D*>(node), context); break;
            case AotRenderNode: ret = AssetBuilder<RenderNode*>::BuildSecondPass(Dynamic_Cast<RenderNode*>(node), context); break;
            case AotTextNode2D: ret = AssetBuilder<TextNode2D*>::BuildSecondPass(Dynamic_Cast<TextNode2D*>(node), context); break;
            case AotCamera2D: ret =  AssetBuilder<Camera2D*>::BuildSecondPass(Dynamic_Cast<Camera2D*>(node), context); break;
            case AotCompositeNode2D: ret =  AssetBuilder<CompositeGroup2D*>::BuildSecondPass(Dynamic_Cast<CompositeGroup2D*>(node), context); break;
            case AotScene2D: ret =  AssetBuilder<Scene2D*>::BuildSecondPass(Dynamic_Cast<Scene2D*>(node), context); break;
            case AotComposite2D: ret =  AssetBuilder<Scene2D*>::BuildSecondPass(Dynamic_Cast<Scene2D*>(node), context); break;
            default: FEATSTD_DEBUG_FAIL(); break; // Shouldn't go here, all types must be covered
            }

            return ret;
        }

        void Node2DAssetBuilder::Dispose(Node2D* node)
        {
            if (node != 0) {
                Node2D* child = node->GetFirstChild();
                while (child != 0) {
                    Node2DAssetBuilder::Dispose(child);
                    child = child->GetNextSibling();
                }

                if (node->IsTypeOf(Scene2D::GetTypeId())) {
                    Scene2D* scene = Dynamic_Cast<Scene2D*>(node);
                    Scene2DAssetBuilder::Dispose(scene);
                }
                else if (node->IsTypeOf(CompositeGroup2D::GetTypeId())) {
                    CompositeGroup2D* compositeGroup = Dynamic_Cast<CompositeGroup2D*>(node);
                    AssetBuilder<CompositeGroup2D*>::Dispose(compositeGroup);
                }
                else if (node->IsTypeOf(Group2D::GetTypeId())) {
                    Group2D* group = Dynamic_Cast<Group2D*>(node);
                    Group2DAssetBuilder::Dispose(group);
                }
                else if (node->IsTypeOf(TextNode2D::GetTypeId())) {
                    TextNode2D* group = Dynamic_Cast<TextNode2D*>(node);
                    TextNode2DAssetBuilder::Dispose(group);
                }
                else if (node->IsTypeOf(RenderNode::GetTypeId())) {
                    RenderNode* renderNode = Dynamic_Cast<RenderNode*>(node);
                    RenderNodeAssetBuilder::Dispose(renderNode);
                }
                else if (node->IsTypeOf(Camera2D::GetTypeId())) {
                    Camera2D* camera = Dynamic_Cast<Camera2D*>(node);
                    Camera2DAssetBuilder::Dispose(camera);
                }
                else {
                    FEATSTD_DEBUG_FAIL();   // Shouldn't go here, all types must be covered
                }
            }
        }
    }
}
