//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CompositeGroup2DAssetBuilder.h"

#include <FeatStd/Util/StaticObject.h>

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine2D/Core/CompositeGroup2D.h>
#include <CanderaWidget/WidgetBase/WidgetBase.h>

#include <CanderaAssetLoader/AssetLoaderBase/CffReader/Composite2DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CompositeNode2DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnchorDefinitionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnchorDefinitionCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/SceneAnimationInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CompositePropertyDefinitionCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CompositePropertyDefinitionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/DynamicItemCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/DynamicPropertyItemInfoCffReader.h>

#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProviderInternal.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/WidgetCollectionAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/AnimationBlendedPropertyListAssetReader.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/Scene2DAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/CompositeGroupFinder.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/LayouterAssetBuilder.h>

namespace Candera {
    using namespace Diagnostics;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        class CompositeGroup2DWidgetCollectionBuilder: public WidgetCollectionAssetBuilderBase {
        public:
            CompositeGroup2DWidgetCollectionBuilder(CompositeGroup2D& compositeGroup):m_compositeGroup(compositeGroup) {}
        private:
            virtual bool AddWidget(WidgetBase* widget) override
            {
                return CompositeGroup2DFunctions::AddWidget(m_compositeGroup, widget);
            }

            virtual WidgetBase* GetFirstWidget() override
            {
                m_widgetIterator = m_compositeGroup.GetWidgetIterator();
                return GetNextWidget();
            }
            virtual WidgetBase* GetNextWidget() override
            {
                if (!m_widgetIterator.IsValid()) {
                    return 0;
                }

                return *m_widgetIterator++;
            }

            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
            CompositeGroup2D& m_compositeGroup;
            CompositeGroup2D::WidgetIterator m_widgetIterator;
            FEATSTD_MAKE_CLASS_UNCOPYABLE(CompositeGroup2DWidgetCollectionBuilder);
        };

        /** Smart helper class that stores the partent CompositeGroup in a static variable while buidling the current one.*/
        class CompositeGroup2DBuilderStack
        {
        public:
            CompositeGroup2DBuilderStack(DefaultAssetProviderInternal* assetProvider, CompositeGroup2D* currentCompositeGroup);
            ~CompositeGroup2DBuilderStack();

            CompositeGroup2D* GetPreviousCompositeGroupOnStack();
        private:
            DefaultAssetProviderInternal* m_assetProvider;
            CompositeGroup2D* m_oldCompositeGroup;
        };

        CompositeGroup2DBuilderStack::CompositeGroup2DBuilderStack(DefaultAssetProviderInternal* assetProvider, CompositeGroup2D* currentCompositeGroup):
            m_assetProvider(assetProvider),
            m_oldCompositeGroup(assetProvider->m_currentCompositeGroup2DBuild)
        {
            assetProvider->m_currentCompositeGroup2DBuild = currentCompositeGroup;
        }

        CompositeGroup2DBuilderStack::~CompositeGroup2DBuilderStack()
        {
            m_assetProvider->m_currentCompositeGroup2DBuild = m_oldCompositeGroup;
            m_assetProvider = 0;
            m_oldCompositeGroup = 0;
        }

        CompositeGroup2D* CompositeGroup2DBuilderStack::GetPreviousCompositeGroupOnStack()
        {
            return m_oldCompositeGroup;
        }

        CompositeGroup2D* Composite2DInstanceBuilder::CreateAndBuildFirstPass(CffLoaderContext context)
        {
            //Create CompositeGroup
            CompositeGroup2D* result = Candera::Internal::CompositeGroup2DFunctions::Create();
            if (result == 0) {
                FEATSTD_LOG_ERROR("Failed to create new CompositeGroup2D instance.");
                return 0;
            }

            //Stores the current CompositeGroup in a static variable of DefaultAssetProvider (used for later resolval of asset ids).
            CompositeGroup2DBuilderStack compositeGroupBuilderStack(context.provider->m_internal, result);

            if (compositeGroupBuilderStack.GetPreviousCompositeGroupOnStack() != 0) {
                CompositeGroup2DFunctions::CompositeStringIdentifier(*result).SetOwner(&CompositeGroup2DFunctions::CompositeStringIdentifier(*compositeGroupBuilderStack.GetPreviousCompositeGroupOnStack()));
            }

            result->SetName(AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(context.handle)));

            const AssetId& compositeGroupAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!compositeGroupAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Composite Group Asset Id is not valid");
            }
            //Read children (first pass).
            if (!SceneNode2DChildrenReader().ReadFirstPass(*result, context)) {
                FEATSTD_LOG_ERROR("Failed to read children of CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                Dispose(result);
                return 0;
            }

            //Define anchors.
            const AssetDataHandle& anchorCollectionHandle = CFFReader::GetComposite2DAnchors(context.handle);
            if (!anchorCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read anchor collection of CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                Dispose(result);
                return 0;
            }

            Int32 anchorCount = CFFReader::GetAnchorDefinitionCollectionChildrenCount(anchorCollectionHandle);
            for (Int32 anchorIndex = 0; anchorIndex < anchorCount; ++anchorIndex) {
                const AssetDataHandle& anchorDataHandle = CFFReader::GetAnchorDefinitionCollectionChildrenElementAt(anchorCollectionHandle, anchorIndex);
                if (!anchorDataHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read anchor no.%d of CompositeGroup2D " AssetIdLogStr, anchorIndex, AssetIdLogArgs(compositeGroupAssetId));
                    Dispose(result);
                    return 0;
                }
                const Char* anchorName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(anchorDataHandle));
                const AssetId& anchorAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(anchorDataHandle));
                if (!anchorAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Anchor AssetId is not valid");
                }
                const AssetId& anchorNodeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetAnchorDefinitionAnchorNode(anchorDataHandle));
                if (!anchorNodeAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Anchor Node AssetId is not valid");
                }
                Node2D* anchorNode = context.provider->GetNode2DByAssetId(anchorNodeAssetId);
                if (anchorNode == 0) {
                    FEATSTD_LOG_ERROR("Failed to find anchor Node2D " AssetIdLogStr " for CompositeGroup2D with "
                        "AssetId: " AssetIdLogStr, AssetIdLogArgs(anchorNodeAssetId), AssetIdLogArgs(compositeGroupAssetId));
                    Dispose(result);
                    return 0;
                }

                if (!CompositeGroup2DFunctions::DefineAnchorPoint(*result, AssetIdFunctions::GetNodeId(anchorAssetId), anchorName, anchorNode)) {
                    FEATSTD_LOG_ERROR("Failed to define AnchorPoint " AssetIdLogStr " for CompositeGroup2D "
                        AssetIdLogStr, AssetIdLogArgs(anchorAssetId), AssetIdLogArgs(compositeGroupAssetId));
                    Dispose(result);
                    return 0;
                }
            }

            //Read widgets (first pass).
            CompositeGroup2DWidgetCollectionBuilder widgetCollectionBuilder(*result);
            const AssetDataHandle& widgetCollectionHandle = CFFReader::GetSceneWidgets(context.handle);
            if (!widgetCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read widget collection for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return 0;
            }
            CffLoaderContext widgetCollectionContext = context.Clone(widgetCollectionHandle);
            if (!widgetCollectionBuilder.CreateAndBuildFirstPass(widgetCollectionContext)) {
                FEATSTD_LOG_ERROR("Failed to load widget collection for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                Dispose(result);
                return 0;
            }

#if defined(CANDERA_LAYOUT_ENABLED)
            if (CFFReader::GetComposite2DIsLayoutEnabled(context.handle)) {
                const AssetDataHandle& layoutHandle = CFFReader::GetComposite2DLayout(context.handle);
                CffLoaderContext layouterContext = context.Clone(layoutHandle);
                result->SetLayouter(AssetBuilder<Candera::Layouter*>::CreateAndBuildFirstPass(layouterContext));
            }
#endif

            return result;
        }

        bool Composite2DInstanceBuilder::BuildSecondPass(CompositeGroup2D* compositeGroup, CffLoaderContext context)
        {
            //Stores the current CompositeGroup in a static variable of DefaultAssetProvider (used for later resolval of asset ids).
            CompositeGroup2DBuilderStack compositeGroupBuilderStack(context.provider->m_internal, compositeGroup);

            const AssetId& compositeGroupAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!compositeGroupAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Composite AssetId is not valid");
            }
            //Read children (second pass).
            if (!SceneNode2DChildrenReader().ReadSecondPass(*compositeGroup, context)) {
                FEATSTD_LOG_ERROR("Failed to read children of CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            bool result = true;
            //Read animations.
            Int32 animationCount = CFFReader::GetSceneSceneAnimationInfoCount(context.handle);
            for (Int animationIndex = 0; animationIndex < animationCount; ++animationIndex) {
                const AssetDataHandle& aiHandle = CFFReader::GetSceneSceneAnimationInfoElementAt(context.handle, animationIndex);
                if (!aiHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read AnimationPlayer data for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }
                const AssetId& animationAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetSceneAnimationInfoAnimation(aiHandle));
                if (!animationAssetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Animation AssetId is not valid");
                }
                Animation::AnimationPlayer::SharedPointer animation = context.provider->CreateCompositeAnimation(animationAssetId);
                if (animation == 0) {
                    FEATSTD_LOG_ERROR("Failed to load AnimationPlayer " AssetIdLogStr " for CompositeGroup2D " AssetIdLogStr
                        , AssetIdLogArgs(animationAssetId), AssetIdLogArgs(compositeGroupAssetId));
                    result = false;
                }
                else {
                    CffLoaderContext aiContext = context.Clone(aiHandle);
                    if (!SceneAnimationBlendedPropertyListAssetReader(animationAssetId, animation).Read(aiContext)) {
                        FEATSTD_LOG_ERROR("Failed to read properties of AnimationPlayer " AssetIdLogStr " for CompositeGroup2D " AssetIdLogStr
                            , AssetIdLogArgs(animationAssetId), AssetIdLogArgs(compositeGroupAssetId));
                        result = false;
                    }
                    CompositeGroup2DFunctions::AddAnimation(*compositeGroup, animation);
                }
            }

            //Read widgets (second pass).
            CompositeGroup2DWidgetCollectionBuilder widgetCollectionBuilder(*compositeGroup);
            const AssetDataHandle& widgetCollectionHandle = CFFReader::GetSceneWidgets(context.handle);
            if (!widgetCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read widget collection for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }
            CffLoaderContext widgetCollectionContext = context.Clone(widgetCollectionHandle);
            if (!widgetCollectionBuilder.BuildSecondPass(widgetCollectionContext)) {
                FEATSTD_LOG_ERROR("Failed to load widget collection for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            if (!result) {
                FEATSTD_LOG_ERROR("Failed to load animation collection for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            //Define property list.
            const AssetDataHandle& compositePropertiesCollectionHandle = CFFReader::GetComposite2DCompositeProperties(context.handle);
            if (!compositePropertiesCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read properties definition for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }
            Composite2DPropertyList propertyList;
            Int32 propertyCount = CFFReader::GetCompositePropertyDefinitionCollectionChildrenCount(compositePropertiesCollectionHandle);
            for (Int32 propertyIndex = 0; propertyIndex < propertyCount; ++propertyIndex) {
                const AssetDataHandle& propertyDataHandle = CFFReader::GetCompositePropertyDefinitionCollectionChildrenElementAt(compositePropertiesCollectionHandle, propertyIndex);
                if (!propertyDataHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read property no.%d definition for CompositeGroup2D " AssetIdLogStr, propertyIndex, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }

                Int32 pathLength = CFFReader::GetCompositePropertyDefinitionWidgetPathLength(propertyDataHandle);
                const AssetId& widgetId = AssetIdFunctions::GetAssetId(CFFReader::GetCompositePropertyDefinitionWidgetPathElementAt(propertyDataHandle, pathLength - 1));
                if (!widgetId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Widget AssetId is not valid");
                }
                if (pathLength < 1) {
                    FEATSTD_LOG_ERROR("Path length to short for property definition no.%d for CompositeGroup " AssetIdLogStr, propertyIndex, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }

                WidgetBase* widget = 0;
                CompositeGroup2D* innerCompositeGroup = CompositeGroupFinder<Node2D, CompositeGroup2D>::GetCompositeGroup(compositeGroup, pathLength - 1, CFFReader::GetCompositePropertyDefinitionWidgetPath(propertyDataHandle));
                if (innerCompositeGroup != 0) {
                    if (widgetId.IsValid()) {
                        widget = innerCompositeGroup->GetWidget(AssetIdFunctions::GetNodeId(widgetId));
                    }
                }
                if (widget != 0) {
                    const Char* widgetPropName = CFFReader::GetCompositePropertyDefinitionWidgetPropertyName(propertyDataHandle);
                    const Char* propName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(propertyDataHandle));

                    MetaInfo::WidgetMetaInfo* widgetMetaInfo = widget->GetMetaInfo();
                    if (widgetMetaInfo != 0) {
                        MetaInfo::WidgetPropertyMetaInfo* propertyMetaInfo = widgetMetaInfo->LookupItem(widgetPropName);
                        if (propertyMetaInfo != 0) {
                            Composite2DPropertyList::Property property = {propName, widgetPropName, widget, propertyMetaInfo};
                            result = result && propertyList.m_list.Add(property);
                        }
                        else {
                            FEATSTD_LOG_ERROR("Failed to find Widget " AssetIdLogStr " PropertyMetaInfo for %s property", AssetIdLogArgs(widgetId), widgetPropName);
                        }
                    }
                    else {
                        FEATSTD_LOG_ERROR("Failed to find Widget " AssetIdLogStr " MetaInfo to access one of its properties", AssetIdLogArgs(widgetId));
                    }
                }
                else {
                    FEATSTD_LOG_ERROR("Failed to find Widget " AssetIdLogStr " to initialize one of its properties in CompositeGroup2D "
                        AssetIdLogStr, AssetIdLogArgs(widgetId), AssetIdLogArgs(compositeGroupAssetId));
                }
            }
            if (result) {
                if (!Composite2DPropertyList::SetPropertyList(compositeGroup, propertyList)) {
                    FEATSTD_LOG_ERROR("Failed to initialize property list of CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    result = false;
                }
            }

#if defined(CANDERA_LAYOUT_ENABLED)
            if (CFFReader::GetComposite2DIsLayoutEnabled(context.handle)) {
                const AssetDataHandle& layoutHandle = CFFReader::GetComposite2DLayout(context.handle);
                CffLoaderContext layouterContext = context.Clone(layoutHandle);
                if (!AssetBuilder<Layouter*>::BuildSecondPass(compositeGroup->GetLayouter(), layouterContext)) {
                    FEATSTD_LOG_ERROR("Failed to initialize layout properties of CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    result = false;
                }
            }
#endif

            return result;
        }

        void Composite2DInstanceBuilder::Dispose(CompositeGroup2D* compositeInstance)
        {
            AssetBuilder<Node2D*>::Dispose(compositeInstance);
        }

        CompositeGroup2D* AssetBuilderBase<CompositeGroup2D*>::Create(const LoaderContext& context)
        {
            return context.provider->CreateCompositeGroup2DByAssetId(AssetIdFunctions::GetAssetId(CFFReader::GetCompositeNode2DCompositeInterface(context.handle)));
        }

        void AssetBuilderBase<CompositeGroup2D*>::Dispose(const CompositeGroup2D* compositeGroup)
        {
            for (CompositeGroup2D::WidgetIterator iterator = compositeGroup->GetWidgetIterator(); iterator.IsValid(); ++iterator) {
                AssetBuilder<WidgetBase*>::Dispose(*iterator);
            }
        }

        bool AssetReaderBase<CompositeGroup2D>::ReadSecondPass(CompositeGroup2D& compositeGroup, const LoaderContext& context)
        {
            const AssetId& compositeGroupAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!compositeGroupAssetId.IsValid()) {
                FEATSTD_LOG_DEBUG("Composite Group AssetId is not valid");
            }
            const Char* compositeIdentifier = CFFReader::GetCompositeNode2DThisNode(context.handle);
            if (compositeIdentifier == 0) {
                FEATSTD_LOG_ERROR("Failed to read identifier of CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }

            if (context.handle.IsPersistent() == false) {
                SizeType compositeIdentifierLength = StringPlatform::Length(compositeIdentifier);
                Char* compositeLongName = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, compositeIdentifierLength + 1);
                if (compositeLongName == 0) {
                    FEATSTD_LOG_ERROR("Failed to allocate memory for identifier of CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }
                StringPlatform::Copy(compositeLongName, compositeIdentifier);
                compositeLongName[compositeIdentifierLength] = '\0';
                CompositeGroup2DFunctions::CompositeStringIdentifier(compositeGroup).SetId(compositeLongName, StringIdentifier::Disposer::Dispose);

                // set owner id for widgets
                for (CompositeGroup2D::WidgetIterator widgetIt = compositeGroup.GetWidgetIterator(); widgetIt.IsValid(); ++widgetIt) {
                    (*widgetIt)->SetOwnerId(&CompositeGroup2DFunctions::CompositeStringIdentifier(compositeGroup));
                }

                // set owner id for animations
                for (CompositeGroup2D::AnimationIterator animationIt = compositeGroup.GetAnimationIterator(); animationIt.IsValid(); ++animationIt) {
                    (*animationIt)->SetOwnerId(&CompositeGroup2DFunctions::CompositeStringIdentifier(compositeGroup));
                }
            }
            else {
                CompositeGroup2DFunctions::CompositeStringIdentifier(compositeGroup).SetId(compositeIdentifier, 0);
            }

            const Composite2DPropertyList& propertyList = Composite2DPropertyList::GetPropertyList(&compositeGroup);
            const AssetDataHandle& nodePropHandle = CFFReader::GetCompositeNode2DCompositeProperties(context.handle);
            if (!nodePropHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read properties data for CompositeGroup2D " AssetIdLogStr, AssetIdLogArgs(compositeGroupAssetId));
                return false;
            }
            Int32 propertyCount = CFFReader::GetDynamicItemDynamicPropertiesCount(nodePropHandle);
            for (Int32 propertyIndex = 0; propertyIndex < propertyCount; ++propertyIndex) {
                const AssetDataHandle& dynPropItemInfoHandle = CFFReader::GetDynamicItemDynamicPropertiesElementAt(nodePropHandle, CFFReader::CFF_DYNAMIC_PROPERTY_ITEM_INFO_SIZE, propertyIndex);
                if (!dynPropItemInfoHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read property no.%d data for CompositeGroup2D " AssetIdLogStr, propertyIndex, AssetIdLogArgs(compositeGroupAssetId));
                    return false;
                }
                const Char* propertyName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetDynamicPropertyItemInfoName(dynPropItemInfoHandle));
                const Char* propValue = CFFReader::GetDynamicPropertyItemInfoValue(dynPropItemInfoHandle);

                const Composite2DPropertyList::Property* property = propertyList.GetProperty(propertyName);
                if (property != 0) {
                    if (!property->metaInfo->Set(property->widget, propValue)) {
                        FEATSTD_LOG_WARN("Vaue of property %s not set on CompositeGroup2D " AssetIdLogStr, propertyName, AssetIdLogArgs(compositeGroupAssetId));
                    }
                }
                else {
                    FEATSTD_LOG_WARN("Property %s definition not found for CompositeGroup2D " AssetIdLogStr, propertyName, AssetIdLogArgs(compositeGroupAssetId));
                }
            }

            return true;
        }

        Composite2DPropertyList& Composite2DPropertyList::DefaultPropertyList()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(Composite2DPropertyList, staticInstance);
            return staticInstance;
        }

        const Composite2DPropertyList& Composite2DPropertyList::GetPropertyList(const CompositeGroup2D* compositeGroup)
        {
            return compositeGroup->GetValue(CdaDynamicPropertyInstance(PropertyList));
        }

        bool Composite2DPropertyList::SetPropertyList(CompositeGroup2D* compositeGroup, const Composite2DPropertyList& propertyList)
        {
            return compositeGroup->SetValue(CdaDynamicPropertyInstance(PropertyList), propertyList);
        }

        bool Composite2DPropertyList::ClearPropertyList(CompositeGroup2D* compositeGroup)
        {
            return compositeGroup->ClearValue(CdaDynamicPropertyInstance(PropertyList));
        }

        const Composite2DPropertyList::Property* Composite2DPropertyList::GetProperty(const Char* name) const
        {
            for (Vector<Composite2DPropertyList::Property>::ConstIterator it = m_list.ConstBegin(); it != m_list.ConstEnd(); ++it) {
                const Composite2DPropertyList::Property& property = *it;
                if (property.name == name) {
                    return &property;
                }
            }

            return 0;
        }

        bool Composite2DPropertyList::operator==(const Composite2DPropertyList& other) const
        {
            return ((m_list.Size() == 0) && (other.m_list.Size() == 0));
        }
    }
}
