//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Camera2DAssetBuilder.h"
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/Camera2DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    namespace Internal {
        using namespace Diagnostics;

        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<Camera2D>::ReadFirstPass(Camera2D& camera2D, LoaderContext& context)
        {
            const AssetId& grpahicDeviceUnitAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCamera2DRenderTarget(context.handle));
            if (!grpahicDeviceUnitAssetId.IsValid()) {
                FEATSTD_LOG_INFO("There is no GraphicDeviceUnit associated to Camera2D " AssetIdLogStr, AssetIdLogArgs(context.id));
            }
            else {
                GraphicDeviceUnit* grapahicDeviceUnit = context.provider->GetGraphicDeviceUnitByAssetId(grpahicDeviceUnitAssetId);

                if (grapahicDeviceUnit != 0) {
                    RenderTarget2D* renderTarget = grapahicDeviceUnit->ToRenderTarget2D();
                    if (renderTarget != 0) {
                        camera2D.SetRenderTarget(renderTarget);
                    }
                    else {
                        FEATSTD_LOG_WARN("GraphicDeviceUnit " AssetIdLogStr " cannot be used as a RenderTarget2D for Camera2D "
                            AssetIdLogStr, AssetIdLogArgs(grpahicDeviceUnitAssetId), AssetIdLogArgs(context.id));
                    }
                }
                else {
                    FEATSTD_LOG_WARN("GraphicDeviceUnit " AssetIdLogStr " could not be created during building Camera2D"
                        AssetIdLogStr, AssetIdLogArgs(grpahicDeviceUnitAssetId), AssetIdLogArgs(context.id));
                }
            }

            Float f1 = 0.0F;
            Float f2 = 0.0F;
            Float f3 = 0.0F;
            Float f4 = 0.0F;

            CFFReader::GetCamera2DViewport(context.handle, f1, f2, f3, f4);
            camera2D.SetViewport(Rectangle(f1, f2, f3, f4));
            CFFReader::GetCamera2DClearColor(context.handle, f1, f2, f3, f4);
            camera2D.SetClearColor(Color(f1, f2, f3, f4));
            camera2D.SetClearColorEnabled(CFFReader::GetCamera2DIsColorClearEnabled(context.handle));
            CFFReader::GetCamera2DScissorRectangle(context.handle, f1, f2, f3, f4);
            camera2D.SetScissorRectangle(Rectangle(f1, f2, f3, f4));
            camera2D.SetScissoringEnabled(CFFReader::GetCamera2DIsScissoringEnabled(context.handle));
            camera2D.SetSequenceNumber(CFFReader::GetCamera2DSequenceNumber(context.handle));
            camera2D.SetSwapEnabled(CFFReader::GetCamera2DIsSwapEnabled(context.handle));
            camera2D.SetCameraEffectiveAlphaEnabled(CFFReader::GetCamera2DApplyCameraAlpha(context.handle));

            return true;
        }

        Camera2D* Camera2DAssetBuilder::Create(LoaderContext& /*context*/)
        {
            return Camera2D::Create();
        }
    }
}
