//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined CANDERA_TEXTRENDERING_TYPES_H
#define CANDERA_TEXTRENDERING_TYPES_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/Alignment.h>


namespace Candera {
    namespace TextRendering {

        /** @addtogroup CanderaTextEngine
         *  @{
         */

        // Types of characters.
        typedef UInt8  Utf8;    ///< Type of an item in an Utf8 encoded string.
        typedef UInt16 Utf16;   ///< Type of an item in an Utf16 encoded string.
        typedef UInt32 Utf32;   ///< Type of an item in an Utf32 encoded string.

        // Generic code point.
        /**
         * @brief Type used for encoding code points.
         * Code points are defined in the widest code space
         * supported by the text engine, which is Unicode.
         */
        typedef Utf32 CodePoint;

        // Generic glyph index.
        /**
         * @brief Type used for encoding glyph indexes.
         * Glyph indexes are a means for the the font engine
         * to swiftly identify glyphs. This can be either a
         * code point or something defined by the font engine.
         */
        typedef UInt32 GlyphIndex;

        // Type for measuring text.
        /**
         * @brief Type used to specify the number of characters within
         * a text buffer. When negative, the text is considered to be
         * '\0' terminated.
         */
        typedef Int32 TextLength;

        /**
         * @brief Type used to specify the position of a character within
         * a text buffer.
         */
        typedef Int32 TextPosition;
        typedef Int32 TextLine;

        // Types for handling bitmaps.
        typedef const UInt8* PixelBuffer; ///< Default type to use for pixel buffers.

        // Types for handling one dimension of bitmaps.
        typedef UInt16 PixelSize;         ///< Type for describing sizes in pixels. It is always positive.
        typedef Int16 PixelPosition;      ///< Type for describing positions in pixels. There are no semantics to the space.

        /// Type for describing sizes of 2D objects in pixels.
        struct PixelSize2D {
            PixelSize width;
            PixelSize height;
        };

        /// Type for describing positions of 2D objects in pixels.
        struct PixelPosition2D {
            PixelPosition x;
            PixelPosition y;
        };
        inline bool operator == (PixelPosition2D const a, PixelPosition2D const b) { return ((a.x == b.x) && (a.y == b.y)); }
        inline bool operator != (PixelPosition2D const a, PixelPosition2D const b) { return (!(a == b)); }

        typedef Int16 PixelPitch2D; ///< Type for describing the distance between lines in a 2D discrete object.

        typedef struct { Int m_value; } FontIdentifier; ///< Type for identifying the font within a style.
        inline bool operator == (FontIdentifier a, FontIdentifier b) { return a.m_value == b.m_value; }
        inline bool operator != (FontIdentifier a, FontIdentifier b) { return a.m_value != b.m_value; }

        typedef struct{ UInt8 m_value; } FaceIdentifier; ///< Type for identifying the font face.
        inline bool operator == (FaceIdentifier a, FaceIdentifier b) { return a.m_value == b.m_value; }
        inline bool operator != (FaceIdentifier a, FaceIdentifier b) { return a.m_value != b.m_value; }

        struct GPosScale {
            UInt16 designUnits;
            Int32 s00;
            Int32 s01;
            Int32 s10;
            Int32 s11;
            Int32 xPpm;
            Int32 yPpm;
        };

        namespace BidiBaseLevel {
            /**
            *  @brief BidiBaseLevel enumeration defines the base level which shall be used on the text.
            *  This is a shaping option and can be set by modifying the class ShapingOptions.
            */
            enum Enum {
                Implicit,            ///< Text defines whether it has the text flow direction LTR or RTL.
                LeftToRight,         ///< Text flow direction will be LTR.
                RightToLeft,         ///< Text flow direction will be RTL.       
                Culture,             ///< Text flow direction will be culture dependent. 
                Deprecated = 0xFF    ///< Text flow direction will be the default old behavior - this enum value shall be ignored.
            };
        }

        /** @} */ // end of CanderaTextEngine

    }    // namespace TextRendering
}    // namespace Candera

#endif // CANDERA_TEXTRENDERING_TYPES_H
