//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TruncationToGlyphIteratorContext_h
#define Candera_TextEngine_TruncationToGlyphIteratorContext_h


#include <FeatStd/Base.h>
#include <FeatStd/Container/Vector.h>
#include <Candera/Environment.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/TextRect.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/PreprocessingContext.h>
#include <Candera/TextEngine/TextRenderContexts/PreprocessedGlyphData.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/Internal/TextProcessProperties.h>

namespace Candera {
    namespace TextRendering {


        /** @addtogroup CanderaTextEngine
        *  @{
        */

        /**
        *  @brief TruncationToGlyphIteratorContext generates the glyph container for truncation.
        *  It currently supports nothing special treatment it only receives the glyphs from TextEngine
        *  and stores them including the area they take.
        */
        class TruncationToGlyphIteratorContext : public PreprocessingContext {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(TruncationToGlyphIteratorContext);
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION()
            {
                return TruncationToGlyphIteratorContext::SharedPointer(TEXTENGINE_TRANSIENT_NEW(TruncationToGlyphIteratorContext)());
            }

            /**
            * Creates a default shared pointer (null pointer) for dynamic property
            * @return a default shared pointer (null pointer) for dynamic properties
            */
            static const Candera::TextRendering::TruncationToGlyphIteratorContext::SharedPointer& GetDefault();

            /**
            * Creates the context for truncation
            */
            TruncationToGlyphIteratorContext();

            /**
            * Destructs the context for truncation
            */
            ~TruncationToGlyphIteratorContext();

            /**
            * Overrides and reinitializes the context so it does not have to be recreated.
            * Saves unnecessary copy/ctor calls.
            * @param style of the truncation text
            * @param textCache is the glyph cache
            */
            void Reinit(const Candera::TextRendering::SharedStyle::SharedPointer& style,
                                                          TextRendering::Internal::TextProcessProperties::TextCache* textCache);
            /**
            * Overrides and reinitializes the context so it does not have to be recreated.
            * Saves unnecessary copy/ctor calls.
            * @param style of the truncation text
            * @param referenceContext is the glyph cache
            */
            void Reinit(const Candera::TextRendering::SharedStyle::SharedPointer& style,
                        const TextRendering::TextRenderContext* referenceContext);
           
            /**
            * The layout bounding rectangle of the text.
            * @return The layout bounding rectangle of the text.
            */
            Candera::TextRendering::TextRect& GetTruncationLayoutRect();

            /**
            * The text bounding rectangle of the text.
            * @return The text bounding rectangle of the text.
            */
            Candera::TextRendering::TextRect& GetTruncationTextRect();
            /**
            * Count of how many glyphs are needed for truncation
            * @return The amount of glyphs in the glyph container
            */
            TextPosition GetGlyphCount() const;
            /**
            * Compares two truncation contexts. Used for dynamic property.
            * @return true if the other glyph iterator is equal to this one
            */
            bool operator==(TruncationToGlyphIteratorContext const& comp) const;
        protected:
            /**
            * Generates the bounding rectangle of an incoming glyph
            */
            TextRect GenerateGlyphRect(PixelPosition const& x, PixelPosition const& y, GlyphBitmap const & glyph) const;

            /**
            * Sets/union of the text rectangle
            */
            void SetTextRectangle(PixelPosition x, PixelPosition y, PixelPosition width, PixelPosition height);
            /**
            * simple 1 to 1 copy into the truncation glyph container, based on the information generated by the TextEngine
            */
            virtual void Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph) override;

        private:
            FEATSTD_SHARED_POINTER_DECLARATION();
            FEATSTD_MAKE_CLASS_UNCOPYABLE(TruncationToGlyphIteratorContext);
            virtual ReferenceIterator* GetReferenceIterator() const override;
            FeatStd::Internal::Vector<PreprocessedGlyphData> m_truncationVector;
            Candera::TextRendering::TextRect m_layoutRectangle;
            Candera::TextRendering::TextRect m_textRectangle;
            Candera::TextRendering::PixelPosition m_ascender;
            Candera::TextRendering::PixelPosition m_descender;
            TextRenderContext * m_textCache;


            void Cleanup();
        };


        /** @} */ // end of CanderaTextEngine

    }// namespace TextRendering
}// namespace Candera
#endif // Candera_TextEngine_TruncationToGlyphIteratorContext_h
