//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlyphDataContainer.h"
#include <Candera/TextEngine/Internal/PreprocessingContextVectorIterator.h>
#include <Candera/TextEngine/PreprocessingContext.h>

namespace Candera {
    namespace TextRendering {


        GlyphDataContainer::GlyphDataContainer(TextPosition const reserveSize) :
            m_textVector()
        {
            Reserve(reserveSize);
        }


        void GlyphDataContainer::AddNew()
        {
            static_cast<void>(m_textVector.Add(PreprocessedGlyphData::GetInvalidGlyphData()));
            m_textVector[m_textVector.Size() - 1].SetMemoryPosition(static_cast<TextPosition>(m_textVector.Size() - 1));
        }

        PreprocessedGlyphData const& GlyphDataContainer::operator[](TextPosition const textPos) const
        {
            FEATSTD_DEBUG_ASSERT(textPos < Size());
            return m_textVector[textPos];
        }


        PreprocessedGlyphData& GlyphDataContainer::operator[](TextPosition const textPos)
        {
            FEATSTD_DEBUG_ASSERT(textPos <= Size());
            if (textPos == Size()) {
                AddNew();
            }
            return m_textVector[textPos];
        }

        void GlyphDataContainer::InvalidateGlyphRange(TextPosition start, TextPosition excludedEndpos)
        {
            FEATSTD_DEBUG_ASSERT(start <= excludedEndpos);
            if (start < 0) {
                start = 0;
            }
            if (excludedEndpos > Size()) {
                excludedEndpos = Size();
            }
            for (; start < excludedEndpos; start++) {
                InvalidateGlyph(start);
            }
        }

        void GlyphDataContainer::MoveInXDirection(TextPosition start, TextPosition excludedEndpos, PixelPosition x)
        {
            MoveInDirection(start, excludedEndpos, x, 0);
        }


        void GlyphDataContainer::MoveInDirection(TextPosition start, TextPosition excludedEndpos, PixelPosition x, PixelPosition y)
        {
            FEATSTD_DEBUG_ASSERT(start <= excludedEndpos);
            if (start < 0) {
                start = 0;
            }
            if (excludedEndpos > Size()) {
                excludedEndpos = Size();
            }
            for (; start < excludedEndpos; start++) {
                m_textVector[start].IncPosition(x, y);
            }
        }


        void GlyphDataContainer::MoveInDirection(PixelPosition x, PixelPosition y)
        {
            MoveInDirection(0, Size(), x, y);
        }

        void GlyphDataContainer::InvalidateGlyph(TextPosition const pos)
        {
            if (pos < Size()) {
                m_textVector[pos].InvalidateGlyph();
            }
        }

        TextPosition GlyphDataContainer::Size() const
        {
            return static_cast<TextPosition>(m_textVector.Size());
        }


        GlyphDataContainer::GlyphContainer::ConstIterator GlyphDataContainer::ConstBegin() const
        {
            return m_textVector.ConstBegin();
        }

        GlyphDataContainer::GlyphContainer::ConstIterator GlyphDataContainer::ConstEnd() const
        {
            return m_textVector.ConstEnd();
        }


        void GlyphDataContainer::Reserve(TextPosition const reserveSize)
        {
            if (reserveSize >= 0) {
                static_cast<void>(m_textVector.Reserve(reserveSize));
            }
        }
        class ReceiveIteratorContext : PreprocessingContext {
        public:

            static Iterator GetIterator2(GlyphDataContainer const& container)
            {
                return Iterator(GetReferenceIterator2(container));
            }

            static ReferenceIterator* GetReferenceIterator2(GlyphDataContainer const& container)
            {
                return Candera::TextRendering::Internal::PreprocessingContextVectorIterator<FeatStd::Internal::Vector<PreprocessedGlyphData>, Candera::TextRendering::PreprocessingContext::ReferenceIterator>
                    (container.ConstBegin(), container.ConstEnd()).Clone();
            }

        private:
            FEATSTD_MAKE_CLASS_STATIC(ReceiveIteratorContext);
            virtual ReferenceIterator* GetReferenceIterator() const override { return 0; }

            virtual void Measure(PixelPosition, PixelPosition, const GlyphBitmap&) override {}

        };


        PreprocessingContext::Iterator GlyphDataContainer::GetIterator() const
        {
            return ReceiveIteratorContext::GetIterator2(*this);
        }


        void GlyphDataContainer::Visit(TextPosition start, TextPosition excludedEndpos, GlyphDataContainerVisitor * visitor)
        {
            if (visitor == 0) {
                return;
            }
            if (start < 0) {
                start = 0;
            }
            if (excludedEndpos > Size()) {
                excludedEndpos = Size();
            }
            for (; start < excludedEndpos; start++) {
                visitor->Visit(m_textVector[start]);
            }
        }

        void GlyphDataContainer::Visit(GlyphDataContainerVisitor * visitor)
        {
            Visit(0, Size(), visitor);
        }

        Candera::TextRendering::GlyphDataContainer::PixelSpan1D GlyphDataContainer::GetXSpan(TextPosition const posStart, TextPosition const posEnd, bool includeAdvance) const
        {
            if ((posStart > posEnd) || (posStart < 0)) {
                return PixelSpan1D(-1, -1);
            }
            PixelPosition endAddValue = includeAdvance?m_textVector[posEnd].GetXAdvance():m_textVector[posEnd].GetWidth();
            return PixelSpan1D(m_textVector[posStart].GetPosition().x, m_textVector[posEnd].GetPosition().x + endAddValue);

        }

        void GlyphDataContainer::AppendInvalidGlyphs(TextPosition const invalidCount)
        {
            TextPosition alreadyInvalidGlyphs = GetTrailingInvalidGlyphs();
            TextPosition invalidToAdd = invalidCount - alreadyInvalidGlyphs;

            for (TextPosition i = 0; i < invalidToAdd; i++) {
                AddNew();
            }
        }


        /**
        * Checks if the current glyph (index) is at the end of a line and within the truncation area (left to right flow)
        */
        bool GlyphDataContainer::IsGlyphInXTrailing(TextPosition const index, PixelPosition maxX, bool emptyGlyphIsTrailing)const
        {
            return IsGlyphInXTrailing(m_textVector[index], maxX, emptyGlyphIsTrailing);
        }
        /**
        * Checks if the current glyph (index) is at the end of a line and within the truncation area (left to right flow)
        */
        bool GlyphDataContainer::IsGlyphInXTrailing(PreprocessedGlyphData const& glyph, PixelPosition maxX, bool emptyGlyphIsTrailing)const
        {
            // empty glyph: e.g. Space will be ignored
            if ((emptyGlyphIsTrailing) && (glyph.GetWidth() == 0)) {
                return true;
            }
            // glyph is (partly) out of the bounding box
            return (glyph.GetPosition().x + glyph.GetXAdvance()) > maxX;
        }

        /**
        * Checks if the current glyph (index) is at the beginning of a line and within the truncation area (right to left flow)
        */
        bool GlyphDataContainer::IsGlyphInXLeading(TextPosition const index, PixelPosition maxX, bool emptyGlyphIsTrailing)const
        {
            return IsGlyphInXLeading(m_textVector[index], maxX, emptyGlyphIsTrailing);
        }

        /**
        * Checks if the current glyph (index) is at the beginning of a line and within the truncation area (right to left flow)
        */
        bool GlyphDataContainer::IsGlyphInXLeading(PreprocessedGlyphData const& glyph, PixelPosition minX, bool emptyGlyphIsTrailing)const
        {
            // empty glyph: e.g. Space will be ignored
            if ((emptyGlyphIsTrailing) && (glyph.GetWidth() == 0)) {
                return true;
            }
            // glyph is (partly) out of the bounding box
            return glyph.GetPosition().x < minX;
        }

        void GlyphDataContainer::Reset()
        {
            m_textVector.Clear();
        }

        TextPosition GlyphDataContainer::GetTrailingInvalidGlyphs()
        {
            TextPosition cnt = 0;
            for (TextPosition i = Size() - 1; i >= 0; i--) {
                if (m_textVector[i].IsInvalidGlyph()) {
                    cnt++;
                }
                else {
                    break;
                }
            }
            return cnt;
        }

    }
}
