//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################




#if !defined(Candera_TextRenderContext_h)
#define Candera_TextRenderContext_h

#include <Candera/TextEngine/GlyphBitmap.h>
#include <Candera/TextEngine/Font.h>

namespace Candera {

namespace TextRendering {

/** @addtogroup CanderaTextEngine
 *  @{
 */

class TextRect;
class Font;

/**
 *  @brief Defines an abstract render context and target for text rendering.
 *  
 *  The render context defines the way glyph bitmaps are used for rendering, once
 *  generated and layouted by the TextRenderer.
 *  Additionally render context provides a GlyphCacheAccess to the TextRenderer.
 */
class TextRenderContext {
public:
    /**
     * @brief This enumeration specifies the order in which glyphs are
     * processed during rendering.
     */
    enum GlyphOrder {
        /**
          * Glyphs are provided in the order of the original buffer.
          * Kerning is disabled. Measurements of these glyphs will
          * provide aproximations for the size the render order.
          */
        OriginalGlyphOrder, 
        /**
          * Chunks of glyphs with the same direction are provided in order
          * of the original buffer. Glyphs within the chunk are provided in the
          * intended order. This allows correct kerning, and measurements match
          * the text in render order.
          */
        OriginalChunkOrder, 
        /**
          * Glyphs are provided in the order they are meant to be rendered.
          * This order is left to right, top to bottom, in the space of the screen.
          * For text that doesn't have this rendering order, this does not coincide
          * with the order of the glyphs within the buffer.
          */
        RenderOrder
    };

    /**
    *   Constructor
    */
    TextRenderContext(): _baseLineOffset(0) {}

    /**
     *  Destructor
     */
    virtual ~TextRenderContext();

    /**
     *  Candera text rendering will use this clipping rectangle to avoid blitting
     *  of invisible (ie. completely outside clipping area) glyphs.
     *  @return Current clipping rectangle.
     */
    virtual const TextRect& GetClipRect() const = 0;

    /**
     *  Blit the given glyph bitmap to the specified position in the render target.
     *  @param x Left coordinate of the glyph bitmap.
     *  @param y Top coordinate of the glyph bitmap.
     *  @param glyph The glyph bitmap to be blitted.
     */
    virtual void Blit(Int16 x, Int16 y, const GlyphBitmap &glyph) = 0;
    /**
    *  Handles glyph which is out of the bounding area.
    *  @param x Left coordinate of the glyph bitmap.
    *  @param y Top coordinate of the glyph bitmap.
    *  @param glyph The glyph bitmap which is out of bounds.
    */
    virtual void SkippedBlit(Int16 x, Int16 y, const GlyphBitmap &glyph)
    {
        FEATSTD_UNUSED3(x, y, glyph);
    }

    /**
     *  Returns whether the the pixel information in the glyph bitmap needs
     *  to be updated. Contexts used for measuring text usually don't care about
     *  the pixel data and may return false.
     *  @return True if pixel information is relevant.
     */
    virtual bool IsRenderingEnabled() const { return true; }

    /**
     *  Returns if context requires cached access to glyphs.
     *  @return True if context requires cached access to glyphs.
     */
    virtual bool IsCachingEnabled() const { return true; }

    /**
     *  Gets GlyphCacheAccess used to allocate memory in the cache.
     *  @return pointer to an object used to allocate memory in the cache;
     *           0 to enable use of the default cache.
     */
    virtual GlyphCacheAccess* GetGlyphCacheAccess() const { return 0; }

    /**
     *  Returns the order in which glyphs shall be provided to this context.
     *  @return The order in which the glyphs are returned.
     */
    virtual GlyphOrder GetGlyphOrder() const { return RenderOrder; }

    /**
     *  Returns offset of the font baseline.
     *  @return offset of the font baseline.
     */
    Int16 BaselineOffset() const { return _baseLineOffset; }

private:
    Int16 _baseLineOffset;
};

 /** @} */ // end of CanderaTextEngine

}// namespace TextRendering

}// namespace Candera

#endif// Candera_TextRenderContext_h
