//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_TextRect_h)
#define Candera_TextRect_h

#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/TextSize.h>
#include <Candera/TextEngine/TextCoordinate.h>

namespace Candera {

    namespace TextRendering {

/** @addtogroup CanderaTextEngine
 *  @{
 */

        class TextCoordinate;
        class TextSize;

        /** 
         *  @brief Defines a scalar 2D rectangle
         */
        class TextRect {
            CANDERA_LINT_DECLARE_INITIALIZER_FUNCTION(Candera::TextRendering::TextRect::SetEmpty)
            CANDERA_LINT_DECLARE_INITIALIZER_FUNCTION(Candera::TextRendering::TextRect::Set)
            public:
                /**
                 *  Constructor
                 */
                TextRect() : m_left(0), m_top(0), m_right(0), m_bottom(0) { SetEmpty(); }

                /**
                 *  Constructor with given parameters.
                 *  @param left   Left side of the TextRectangle.
                 *  @param top    Top side of the TextRectangle.
                 *  @param right  Right side of the TextRectangle.
                 *  @param bottom Bottom side of the TextRectangle.
                 */
                TextRect(PixelPosition left, PixelPosition top, PixelPosition right, PixelPosition bottom) : m_left(0), m_top(0), m_right(0), m_bottom(0) { Set(left, top, right, bottom); }

                /**
                 *  Constructor with given parameters.
                 *  @param pos  The text coordinate position of the TextRectangle.
                 *  @param size The size of the TextRectangle.
                 */
                TextRect(const TextCoordinate &pos, const TextSize &size) : m_left(0), m_top(0), m_right(0), m_bottom(0) { Set(pos, size); }

                /**
                 *  Retrieves the left coordinate of this TextRectangle.
                 *  @return The left coordinate of this TextRectangle.
                 */
                PixelPosition GetLeft() const { return m_left; }

                /**
                 *  Sets the left coordinate of this TextRectangle.
                 *  @param left The left coordinate of this TextRectangle.
                 */
                void SetLeft(PixelPosition left) { m_left = left; }

                /**
                 *  Retrieves the top coordinate of this TextRectangle.
                 *  @return The top coordinate of this TextRectangle.
                 */
                PixelPosition GetTop() const { return m_top; }

                /**
                 *  Sets the top coordinate of this TextRectangle.
                 *  @param val The top coordinate of this TextRectangle.
                 */
                void SetTop(PixelPosition val) { m_top = val; }
                
                /**
                 *  Retrieves the right coordinate of this TextRectangle.
                 *  @return The right coordinate of this TextRectangle.
                 */
                PixelPosition GetRight() const { return m_right; }

                /**
                 *  Sets the right coordinate of this TextRectangle.
                 *  @param right The right coordinate of this TextRectangle.
                 */
                void SetRight(PixelPosition right) { m_right = right; }

                /**
                 *  Retrieves the bottom coordinate of this TextRectangle.
                 *  @return The bottom coordinate of this TextRectangle.
                 */
                PixelPosition GetBottom() const { return m_bottom; }

                /**
                 *  Sets the bottom coordinate of this TextRectangle.
                 *  @param val The bottom coordinate of this TextRectangle.
                 */
                void SetBottom(PixelPosition val) { m_bottom = val; }
                
                /**
                 *  Retrieves the width of this TextRectangle.
                 *  @return The width of this TextRectangle.
                 */
                PixelPosition GetWidth() const { return (m_right - m_left) + 1; }

                /**
                 *  Sets the width of this TextRectangle.
                 *  @param val The width of this TextRectangle.
                 */
                void SetWidth(PixelPosition val) { m_right = (m_left + val) - 1; }

                /**
                 *  Retrieves the height of this TextRectangle.
                 *  @return The height of this TextRectangle.
                 */
                PixelPosition GetHeight() const { return (m_bottom - m_top) + 1; }

                /**
                 *  Sets the height of this TextRectangle.
                 *  @param val The height of this TextRectangle.
                 */
                void SetHeight(PixelPosition val) { m_bottom = (m_top + val) - 1; }

                /**
                 *  Sets all the parameters.
                 *  @param left   Left side of the TextRectangle.
                 *  @param top    Top side of the TextRectangle.
                 *  @param right  Right side of the TextRectangle.
                 *  @param bottom Bottom side of the TextRectangle.
                 */
                void Set(PixelPosition left, PixelPosition top, PixelPosition right, PixelPosition bottom);

                /**
                 *  Sets all the parameters.
                 *  @param pos  The text coordinate position of the TextRectangle.
                 *  @param size The size of the TextRectangle.
                 */
                void Set(const TextCoordinate &pos, const TextSize &size);

                /**
                 * Get position.
                 * 
                 * @return The text coordinate position of the TextRectangle.
                 */
                TextCoordinate GetPosition() const { return TextCoordinate(GetLeft(), GetTop()); }

                /**
                 * Get size.
                 * 
                 * @return the size of the TextRectangle.
                 */
                TextSize GetSize() const { return TextSize(GetWidth(), GetHeight()); }

                /**
                 *  Retrieves whether this TextRectangle is empty or not.
                 *  @return True if it's empty, otherwise false
                 */
                bool IsEmpty() const;

                /**
                 *  Retrieves whether this TextRectangle has strictly positive
                 *  extent or not.
                 *  @return True if it's positive, otherwise false
                 */
                bool IsPositive() const;

                /**
                 *  Retrieves whether this TextRectangle has any negative
                 *  extent or not.
                 *  @return True if it's negative, otherwise false
                 */
                bool IsNegative() const;

                /**
                 *  Sets this TextRectangle object as empty.
                 */
                void SetEmpty();

                /**
                 *  Retrieves whether this TextRectangle contains the given coordinates or not.
                 *  @param x The x coordinate that is checked.
                 *  @param y The y coordinate that is checked.
                 *  @return True, if it contains the given coordinates, otherwise false.
                 */
                bool Contains(PixelPosition x, PixelPosition y) const;

                /**
                 *  Compares two rectangles.
                 *  @return True if the rectangles are the same, 
                 *      false otherwise.
                 */
                bool operator == (const TextRect& src) const;
                /**
                 *  Compares two rectangles.
                 *  @return False if the rectangles are the same, 
                 *      true otherwise.
                 */
                bool operator != (const TextRect& src) const;

                /**
                 *  Returns the intersection of two rectangles.
                 *  If the rectangles don't intersect, the return value has
                 *  negative extent (i.e. IsNegative returns true.).
                 *  @param src The other rectangle.
                 *  @return The rectangle containing the intersection.
                 */
                TextRect Intersect(const TextRect &src) const;

                /**
                 *  Returns the smallest rectangle containing the union of two
                 *  rectangles.
                 *  @param src The other rectangle.
                 *  @return The rectangle containing the union.
                 */
                TextRect Union(const TextRect &src) const;

                /**
                 *  Retrieve the minimum rectangle. This rectangle has negative
                 *  extent. It is the unit element of TextRect::Union and the
                 *  zero element of TextRect::Intersect.
                 *  @ return The minimum rectangle.
                 */
                static const TextRect& GetMin();
                /**
                 *  Retrieve the maximum rectangle. This rectangle has positive
                 *  extent. It is the unit element of TextRect::Intersect and the
                 *  zero element of TextRect::Union.
                 *  @ return The maximum rectangle.
                 */
                static const TextRect& GetMax();
                /**
                 *  Retrieve a rectangle created by with default constructor.
                 *  This is an addressable constant empty rectangle.
                 *  @ return The default rectangle.
                 */
                static const TextRect& GetDefault();

            private:
                PixelPosition m_left;
                PixelPosition m_top;
                PixelPosition m_right;
                PixelPosition m_bottom;
        };

        /**
         *  Returns the intersection of two rectangles.
         *  If the rectangle don't intersect, an empty rectangle is returned.
         *  @param l The first rectangle.
         *  @param r The second rectangle.
         *  @return The rectangle containing the intersection.
         */
        TextRect Intersect(const TextRect &l, const TextRect &r);

 /** @} */ // end of CanderaTextEngine

    }    // namespace TextRendering

}    // namespace Candera

#endif    // Candera_TextRect_h
