//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextRect.h"
#include <Candera/Environment.h>
#include <FeatStd/Util/FeatLimits.h>

namespace Candera {
    namespace TextRendering {
        // ----------------------------------------------------------------------------
        const TextRect& TextRect::GetMin()
        {
            static const TextRect s_rect(
                FeatStd::Internal::Limits<PixelPosition>::Max(), FeatStd::Internal::Limits<PixelPosition>::Max(), 
                FeatStd::Internal::Limits<PixelPosition>::Min(), FeatStd::Internal::Limits<PixelPosition>::Min());
            return s_rect;
        }

        // ----------------------------------------------------------------------------
        const TextRect& TextRect::GetMax()
        {
            static const TextRect s_rect(
                FeatStd::Internal::Limits<PixelPosition>::Min(), FeatStd::Internal::Limits<PixelPosition>::Min(), 
                FeatStd::Internal::Limits<PixelPosition>::Max(), FeatStd::Internal::Limits<PixelPosition>::Max());
            return s_rect;
        }

        // ----------------------------------------------------------------------------
        const TextRect& TextRect::GetDefault()
        {
            static const TextRect s_rect;
            return s_rect;
        }

        // ----------------------------------------------------------------------------
        void TextRect::Set(Int16 left, Int16 top, Int16 right, Int16 bottom)
        {
            m_left = left;
            m_top = top;
            m_right = right;
            m_bottom = bottom;
        }

        // ----------------------------------------------------------------------------
        void TextRect::Set(const TextCoordinate &pos, const TextSize &size)
        {
            m_left = pos.GetX();
            m_top = pos.GetY();
            m_right = (m_left + size.GetWidth()) - 1;
            m_bottom = (m_top + size.GetHeight()) - 1;
        }

        // ----------------------------------------------------------------------------
        bool TextRect::IsEmpty() const
        {
            return (m_right < m_left) || (m_bottom < m_top);
        }

        // ----------------------------------------------------------------------------
        bool TextRect::IsPositive() const
        {
            return (m_right >= m_left) && (m_bottom >= m_top);
        }

        // ----------------------------------------------------------------------------
        bool TextRect::IsNegative() const
        {
            // Check for overflow when determining if negative.
            return
                ((m_right < FeatStd::Internal::Limits<PixelPosition>::Max()) &&
                    ((m_right + 1) < m_left)) ||
                ((m_bottom < FeatStd::Internal::Limits<PixelPosition>::Max()) &&
                    ((m_bottom + 1) < m_top));
        }

        // ----------------------------------------------------------------------------
        void TextRect::SetEmpty()
        {
            m_left = 0;
            m_top = 0;
            m_right = -1;
            m_bottom = -1;
        }

        // ----------------------------------------------------------------------------
        bool TextRect::Contains(Int16 x, Int16 y) const
        {
            return (m_left   <= x) &&
                   (m_top    <= y) &&
                   (m_right  >= x) &&
                   (m_bottom >= y);
        }

        // ----------------------------------------------------------------------------
        bool TextRect::operator == (const TextRect& src) const
        {
            bool areEqual = 
                (src.GetTop() == GetTop()) &&
                (src.GetLeft() == GetLeft()) &&
                (src.GetBottom() == GetBottom()) &&
                (src.GetRight() == GetRight());
            return areEqual;
        }

        // ----------------------------------------------------------------------------
        bool TextRect::operator != (const TextRect& src) const
        {
            return !operator ==(src);
        }

        // ----------------------------------------------------------------------------
        TextRect TextRect::Intersect(const TextRect &src) const
        {
            TextRect dst(*this);
            if (dst.GetLeft() < src.GetLeft()){
                dst.SetLeft(src.GetLeft());
            }
            if (dst.GetRight() > src.GetRight()){
                dst.SetRight(src.GetRight());
            }
            if (dst.GetTop() < src.GetTop()){
                dst.SetTop(src.GetTop());
            }
            if (dst.GetBottom() > src.GetBottom()){
                dst.SetBottom(src.GetBottom());
            }
            return dst;
        }

        // ----------------------------------------------------------------------------
        TextRect TextRect::Union(const TextRect &src) const
        {
            TextRect dst(*this);
            if (dst.GetLeft() > src.GetLeft()){
                dst.SetLeft(src.GetLeft());
            }
            if (dst.GetRight() < src.GetRight()){
                dst.SetRight(src.GetRight());
            }
            if (dst.GetTop() > src.GetTop()){
                dst.SetTop(src.GetTop());
            }
            if (dst.GetBottom() < src.GetBottom()){
                dst.SetBottom(src.GetBottom());
            }
            return dst;
        }


        // ----------------------------------------------------------------------------
        static inline Int16 Maximum(Int16 l, Int16 r)
        {
            return (l < r) ? r : l;
        }

        // ----------------------------------------------------------------------------
        static inline Int16 Minimum(Int16 l, Int16 r)
        {
            return (l < r) ? l : r;
        }

        // ----------------------------------------------------------------------------
        TextRect Intersect(const TextRect &l, const TextRect &r)
        {
            TextRect i(l);

            if (!i.IsEmpty()) {
                if (r.IsEmpty()) {
                    i = r;
                }
                else {
                    i.Set(Maximum(i.GetLeft(),   r.GetLeft()),
                          Maximum(i.GetTop(),    r.GetTop()),
                          Minimum(i.GetRight(),  r.GetRight()),
                          Minimum(i.GetBottom(), r.GetBottom()));

                    if ((i.GetLeft() > i.GetRight()) || (i.GetTop() > i.GetBottom())) {
                        i.SetEmpty();    // Rectangles don't intersect
                    }
                }
            }
            return i;
        }   // Intersect

    }    // namespace TextRendering
}    // namespace Candera

