//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_TextProperties_h)
#define Candera_TextProperties_h

#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/PreprocessingContext.h>

namespace Candera {
    namespace TextRendering {

/** @addtogroup CanderaTextEngine
 *  @{
 */

        /** 
         *  @brief Defines the means to retrieve the text for rendering.
         *  
         *  The TChar interface defines access to the default text encoding.
         *  See TChar and TCharEncodingScheme and TCharIsUtfEncoded for information
         *  on the supported encodings and current statically defined text encoding.
         */
        class TextProperties {
            public:

                enum ProcessingLevel
                {
                    Raw,    ///< Raw text
                    Shaped  ///< Shaped text
                };

                typedef PreprocessingContext::Iterator GlyphIterator;

                /**
                 *  Constructor. The text is bound within the object. The buffer
                 *  should not be changed or destroyed during the lifetime of this object.
                 *  TChar objects are processed starting from the first until the first 0
                 *  is encountered, but not more than "length", when length is positive.
                 *  @param text Pointer to a buffer holding the text.
                 *  @param length Number of characters to process. -1 to process until EOS.
                 */
                TextProperties(const TChar* text, TextLength length = -1) : m_processingLevel(Raw), m_text(text), m_length(length) {}
                TextProperties(const GlyphIterator& iterator) : m_processingLevel(Shaped), m_text(0), m_length(0), m_glyphIterator(iterator) {}

                /**
                 *  Retrieve the text bound to this object.
                 *  @return Pointer to a buffer holding the text.
                 */
                const TChar* GetTChar() const { return (m_processingLevel == Raw) ? m_text : 0; }

                /**
                 *  Retrieve the length set when binding the text.
                 *  @return length Number of characters bound. -1 to process until EOS.
                 */
                TextLength GetTCharLength() const { return (m_processingLevel == Raw) ? m_length : 0; }

                GlyphIterator GetGlyphIterator() const { return (m_processingLevel == Shaped) ? m_glyphIterator : GlyphIterator(); }

                ProcessingLevel GetProcessingLevel() const { return m_processingLevel; }

            private:
                ProcessingLevel m_processingLevel;

                const TChar* m_text;
                TextLength m_length;

                GlyphIterator m_glyphIterator;
        };

 /** @} */ // end of CanderaTextEngine

    }    // namespace TextRendering

}    // namespace Candera

#endif    // Candera_TextProperties_h
