//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_TextMeasureContext_h)
#define Candera_TextMeasureContext_h

#include <Candera/TextEngine/TextRenderContext.h>
#include <Candera/TextEngine/MeasuringOptions.h>
#include <Candera/TextEngine/TextRect.h>

namespace Candera { namespace TextRendering {

/** @addtogroup CanderaTextEngine
 *  @{
 */

/**
 *  @brief TextMeasureContext defines the base class used for measuring text
 *  by faking a render call.
 */
class TextMeasureContext : public TextRenderContext{
public:
    //CANDERA_DEPRECATED_3_1_1(
    //    "Take reference context into consideration.",
    //    TextMeasureContext());

    /**
     * Constructor.
     * @param reference A context from which the measure context takes its
     *      caching information.
     */
    explicit TextMeasureContext(const TextRenderContext* reference);

    /**
     * Set the reference text render context. This is the context from which
     * the measure context takes its caching information.
     * @param reference A context from which the measure context takes its
     *      caching information.
     */
    void SetReferenceTextRenderContext(const TextRenderContext* reference) { m_reference = reference; }
    const TextRenderContext* GetReferenceTextRenderContext() const { return m_reference; }

    /**
     * Set the next TextMeasureContext in the chain.
     * TextMeasureContexts are chained to measure different characteristics of
     * a text by only calling TextRenderer::Render once.
     * @param context Pointer to the next context in the list. The caller
     *                should insure the linked object is alive when
     *                TextRenderer::Render is called.
     */
    void SetNextContext(TextMeasureContext* context) { m_next = context; }
    TextMeasureContext* GetNextContext() { return m_next; }

protected:
    /**
     * This is called each time Blit is called on TextRenderContext during
     * TextRendering::Render, for each context in the measuring chain.
     *  @param x Left coordinate of the glyph bitmap.
     *  @param y Top coordinate of the glyph bitmap.
     *  @param glyph The glyph bitmap to be measured.
     */
    virtual void Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph) = 0;
    /**
    * This is called each time SkippedBlit is called on TextRenderContext during
    * TextRendering::Render, for each context in the measuring chain.
    *  @param x Left coordinate of the glyph bitmap.
    *  @param y Top coordinate of the glyph bitmap.
    *  @param glyph The glyph bitmap to be measured.
    */
    virtual void SkippedMeasure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph)
    {
        FEATSTD_UNUSED3(x, y, glyph);
    }

private:
    virtual const TextRect& GetClipRect() const override; 
    virtual void Blit(Int16 x, Int16 y, const GlyphBitmap &glyph) override final;
    virtual void SkippedBlit(Int16 x, Int16 y, const GlyphBitmap &glyph) override final;
    virtual bool IsRenderingEnabled() const override final;
    virtual bool IsCachingEnabled() const override final;
    virtual GlyphCacheAccess* GetGlyphCacheAccess() const override final;

    const TextRenderContext* m_reference;
    TextMeasureContext* m_next;
};

 /** @} */ // end of CanderaTextEngine

}// namespace TextRendering
}// namespace Candera

#endif// Candera_TextMeasureContext_h
