//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_TextLayoutStrategy_h)
#define Candera_TextLayoutStrategy_h

#include <Candera/TextEngine/Types.h>

namespace Candera {

namespace TextRendering {

/** @addtogroup CanderaTextEngine
 *  @{
 */

/**
 *  @brief Defines an abstract class that can be used to modify the text layout
 *  algorithm.
 */
class TextLayoutStrategy {
public:
    /**
     * @brief Action taken by the text renderer after an event is processed.
     */
    enum Action{
        Continue,   ///< Continue processing the text.
        Release,    ///< Stop accessing the strategy.
        Break,      ///< Stop processing the text.
        Skip        ///< Skip processing the text.
    };

    /**
     * @brief Pass describes the type of operation the text renderer is going
     * to do in the next pass of the text.
     * 
     * Line breaks are guaranteed to fall in the same place on each pass.
     */
    enum Pass{
        Measurement, ///< Preliminary measurement pass.
        Render       ///< Stop processing the text.
    };

    /**
     * @brief Type of stop at a line break event.
     */
    enum LineBreakType{
        MandatoryBreak,     ///< A mandatory break, like a paragraph end.
        AutomaticBreak,     ///< Break by word wrap at an allowed position.
        ForcedBreak,        ///< Break by word wrap in the middle of a word.
        TruncatedLineBreak  ///< Break with no word wrap enabled but in the middle of text (currently not detected by TextRenderer itself)
    };

    /**
     *  This event is triggered before the text has began processing.
     *  During a render the text might be processed multiple time because
     *  of required measurements.
     *  Each time a new processing starts the strategy should be the same,
     *  otherwise the result is undefined. Events after this one are all
     *  considered to be part of the same pass.
     *  @param pass Type of the current pass.
     *  @return Action taken by the text renderer before beginning a pass over
     *      the text. This can be used to stop processing after a pass
     *      if the result of subsequent passes is not important.
     */
    virtual Action OnInit(Pass pass) = 0;

    /**
     *  This event is triggered after a line has been processed.
     *  @param textCursor The position within the text, at the beginning of the
     *      next line.
     *  @param type Type of the line break.
     *  @return Action taken by the text renderer after the line has ended.
     *      This can be used to stop text processing after a number of lines
     *      have been processed.
     */
    virtual Action OnLineBreak(const Char* textCursor, LineBreakType type) = 0;

    /**
     *  This event is triggered before blitting each glyph bitmap.
     *  @param textPosition The position of the glyph in the original text.
     *  @return Action taken by the text renderer for the glyph at the textPosition.
     *      This can be used to skip the further blitting of the glyph bitmap if needed.
     */
    virtual Action OnGlyphProcessing(TextPosition /*textPosition*/) { return Continue; };

};

 /** @} */ // end of CanderaTextEngine

}// namespace TextRendering

}// namespace Candera

#endif// Candera_TextLayoutStrategy_h
