//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextEngineDataTypes.h"
#include <Candera/TextEngine/Font.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <CanderaPlatform/OS/SerializationPlatform.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>

namespace Candera {

    namespace Internal {
        /**
        * Simple generic class for wrapping small buffers that can most of the time fit in a given size on the stack.
        * If the buffer does not fit, it will be temporarily allocated on heap.
        */
        template<typename T, UInt32 SizeOnStack>
        class TemporaryBuffer
        {
        public:
            TemporaryBuffer(UInt32 actualSize) :
                m_dynamic(0),
                m_data(0)
            {
                if (actualSize > SizeOnStack) {
                    m_dynamic = FEATSTD_NEW_ARRAY(T, actualSize);
                    m_data = m_dynamic;
                }
                else {
                    m_data = m_static;
                }
            }

            ~TemporaryBuffer()
            {
                if (m_dynamic != 0) {
                    FEATSTD_DELETE_ARRAY(m_dynamic);
                    m_dynamic = 0;
                }

                m_data = 0;
            }

            T* GetData() { return m_data; }
        private:
            T m_static[SizeOnStack];
            T* m_dynamic;
            T* m_data;

            TemporaryBuffer();
            FEATSTD_MAKE_CLASS_UNCOPYABLE(TemporaryBuffer);
        };
    }

    namespace MetaInfo {
        bool DataType<TextRendering::Font>::ConvertToString(const TextRendering::Font &val, Char *buf, UInt bufLen) {
            if (val.GetFaceName() == 0) {
                *buf = '\0';
            } else {
                Int pos;
                if (SerializationPlatform::FromBinary(buf, bufLen, val.GetFaceName(), &pos) != 1) {
                    return false;
                }
                buf[pos] = ';';
                buf += pos + 1;
                bufLen -= static_cast<UInt>(pos + 1);
                if (SerializationPlatform::FromBinary(buf, bufLen, val.GetFontSize().GetHeight(), &pos) != 1) {
                    return false;
                }
                if ((val.GetFaceIndex() == 0) && (val.GetFontSize().GetScale() == Vector2(1.F, 1.F))){
                    return true;
                }
                buf[pos] = ';';
                buf += pos + 1;
                bufLen -= static_cast<UInt>(pos + 1);
                if (SerializationPlatform::FromBinary(buf, bufLen, val.GetFaceIndex(), &pos) != 1) {
                    return false;
                }
                if (val.GetFontSize().GetScale() == Vector2(1.F, 1.F)) {
                    return true;
                }
                buf[pos] = ';';
                buf += pos + 1;
                bufLen -= static_cast<UInt>(pos + 1);
                Float size[] = { val.GetFontSize().GetScale().GetX(), val.GetFontSize().GetScale().GetY() };
                if (SerializationPlatform::FromBinary(buf, bufLen, ";", size, size + 2) != 2) {
                    return false;
                }
            }
            return true;
        }

        bool DataType<TextRendering::Font>::ConvertFromString(TextRendering::Font &val, const Char *buf)
        {
            if (buf == 0) {
                return false;
            }

            //find the split point between the font name and the rest of the information.
            UInt32 splitPoint = 0;
            while ((buf[splitPoint] != ';') && (buf[splitPoint] != '\0')) {
                splitPoint++;
            }

            //if font height is missing, fail
            if (buf[splitPoint] == '\0') {
                return false;
            }

            //split and convert the rest of the information into height and optional face index.
            Int fontInfo[2] = {
                0,  //height.
                0   //optional face index.
            };
            Int pos;
            if (SerializationPlatform::ToBinary(buf + splitPoint + 1, ";", &fontInfo[0], &fontInfo[2], &pos) == 0) {
                return false;
            }
            Float fontScale[2] = {
                1.F,
                1.F
            };
            if (buf[splitPoint + 1 + pos] != '\0') {
                FEATSTD_UNUSED(SerializationPlatform::ToBinary(buf + splitPoint + 1 + pos + 1, ";", &fontScale[0], &fontScale[2], &pos));
            }

            //initialize font name
            Candera::Internal::TemporaryBuffer<Char, 64> temporaryName(splitPoint + 1);
            Char* name = temporaryName.GetData();
            if (name == 0) {
                return false;
            }
            StringPlatform::CopyPartial(name, buf, splitPoint);
            name[splitPoint] = '\0';

            bool result = true;
            if (StringPlatform::CompareStrings(name, "") != 0) {
                result = val.Setup(name, TextRendering::FontSize(Int16(fontInfo[0]), fontScale[0], fontScale[1]), UInt8(fontInfo[1]));
            }

            return result;
        }
    }   // namespace MetaInfo
}   // namespace Candera

