//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_Style_h)
#define Candera_Style_h

#include <FeatStd/Platform/String.h>
#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/Common/NotifyPropertyChangedInterface.h>
#include <Candera/TextEngine/Font.h>
#include <Candera/TextEngine/Metrics.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Container/Map.h>
#include <FeatStd/Event/EventSource.h>


namespace Candera {

    namespace Globalization {
        class CultureChangeListener;
    }
    namespace TextRendering {

        namespace Internal {
            class StyleTools;
        }

        class SharedStyle;

        /** @addtogroup CanderaTextEngine
         *  @{
         */

        /**
         * @brief Defines a text style.
         *
         * A text Style contains:
         * - A list of StyleEntry definitions:
         *     - Priority.
         *     - Font (name and size).
         *     - Lower and upper code point bounds.
         * - A pointer to a base Style.
         * - A default Font (name and size).
         *
         * The Style is used to define rules for assigning Font definitions to specific
         *  code points or code point ranges. The TextRenderer queries the Font from
         *  the given Style, for each code point that it renders.
         *
         * Style generates NotifyPropertyChangedEvents as soon as a property has changed. It can also receive such an event (e.g. from its base style).
         * A new event will be propagated to the attached event listeners, when the style receives such a NotifyPropertyChangedEvent.
         */
        class Style: public CanderaObject, public NotifyPropertyChangedInterface {
            friend class Internal::StyleTools;
            public:
                /**
                 * Construct a Style.
                 */
                Style();

                virtual ~Style();

                /* Construct a Style from a Font.
                 *
                 * The font is set as the default font and, if no other rules are set,
                 *  will be returned by the GetFont method for any code point input.
                 *
                 * @param font Font used as default font.
                 */
                explicit Style(const Font& font);

                /**
                 * Retrieve the composite priority of the font for the input code point.
                 *
                 * The composite priority contains both the priorities of the fonts defined by this
                 *  style, and the ones defined in the base style. For example, if this style
                 *  defines 5 fonts, priorities 0..4, and a base style it is set with 10 fonts,
                 *  priorities 0..9, than this style composite priorities would be 10..14, higher
                 *  than the ones of the base style.
                 *
                 * This method searches into this and the base styles for the font with a composite priority
                 *  lower than a given one, but whose style definition contains the input code point.
                 * The return value is the composite priority of the font that was defined to be used
                 *  for the given code point, and it can be used to:
                 * - Get the associated Font through GetFont(),
                 * - Iterate to the next composite priority by recalling GetCompositePriority with the
                 *  returned value as maximumCompositePriority.
                 *
                 * If no Font is found for the given code point, -1 is returned.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @param codePoint Input code point.
                 * @param maximumCompositePriority Input limit for the returned composite priority.
                 * @return The composite priority of the Font that matches the code point and it is lower
                 *  then the maximumCompositePriority.
                 */
                Int GetCompositePriority(Utf32 codePoint, Int maximumCompositePriority) const;

                /**
                * Retrieve the composite priority of the font for the input code point.
                *
                * The composite priority contains both the priorities of the fonts defined by this
                *  style, and the ones defined in the base style. For example, if this style
                *  defines 5 fonts, priorities 0..4, and a base style it is set with 10 fonts,
                *  priorities 0..9, than this style composite priorities would be 10..14, higher
                *  than the ones of the base style.
                *
                * This method searches into this and the base styles for the font with a composite priority
                *  lower than a given one, but whose style definition contains the input code point.
                * The return value is the composite priority of the font that was defined to be used
                *  for the given code point, and it can be used to:
                * - Get the associated Font through GetFont(),
                * - Iterate to the next composite priority by recalling GetCompositePriority with the
                *  returned value as maximumCompositePriority.
                *
                * If no Font is found for the given code point, -1 is returned.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @param codePoint Input code point.
                * @param maximumCompositePriority Input limit for the returned composite priority.
                * @return The composite priority of the Font that matches the code point and it is lower
                *  then the maximumCompositePriority.
                */
                Int GetCompositePriority(const Char* locale, Utf32 codePoint, Int maximumCompositePriority) const;

                /**
                 * Retrieve the maximum composite priority of the Fonts defined in this and the base styles.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @return The maximum composite priority.
                 */
                Int GetMaximumCompositePriority() const;

                /**
                * Retrieve the maximum composite priority of the Fonts defined in this and the base styles.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @return The maximum composite priority.
                */
                Int GetMaximumCompositePriority(const Char* locale) const;

                /**
                 * Retrieve the font for the given composite priority.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * If the composite priority is -1, than the default font is returned.
                 * @param compositePriority Composite priority of the requested Font.
                 * @return Font with the given composite priority.
                 */
                const Font& GetFontByCompositePriority(Int compositePriority) const;

                /**
                * Retrieve the font for the given composite priority.
                *
                * If the composite priority is -1, than the default font is returned.
                * @param locale is used to operate on a style entry list other than the active one.
                * @param compositePriority Composite priority of the requested Font.
                * @return Font with the given composite priority.
                */
                const Font& GetFontByCompositePriority(const Char* locale, Int compositePriority) const;

                /**
                 * Retrieve the font for the given code point.
                 *
                 * The Font is searched from the maximum composite priority to the lowest
                 *  until a code point range that contains the input code point is found. The
                 *  first matching Font is returned. If no range is found, the default font
                 *  is returned.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @param codePoint Input code point.
                 * @return Font for the given code point.
                 */
                const Font& GetFontByCodePoint(Utf32 codePoint) const;

                /**
                * Retrieve the font for the given code point.
                *
                * The Font is searched from the maximum composite priority to the lowest
                *  until a code point range that contains the input code point is found. The
                *  first matching Font is returned. If no range is found, the default font
                *  is returned.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @param codePoint Input code point.
                * @return Font for the given code point.
                */
                const Font& GetFontByCodePoint(const Char* locale, Utf32 codePoint) const;

                /**
                 * Retrieve metrics of the Style object.
                 *
                 * The Style metrics is the reunion of the containing Font metrics.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @return The Style metrics.
                 */
                const Metrics& GetMetrics() const;

                /**
                * Retrieve metrics of the Style object.
                *
                * The Style metrics is the reunion of the containing Font metrics.
                *
                * @param       locale is used to operate on a style entry list other than the active one.
                * @param [out] result The Style metrics.
                */
                void GetMetrics(const Char* locale, Metrics& result) const;

                /**
                 * Hint number of StyleEntry definitions.
                 *
                 * It is used for optimizing memory allocation and should be called
                 *  only before any StyleEntry is defined.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @param styleEntryCount Hint on the number of StyleEntry definitions
                 *        that will be set.
                 */
                void HintStyleEntryCount(Int styleEntryCount);

                /**
                * Hint number of StyleEntry definitions.
                *
                * It is used for optimizing memory allocation and should be called
                *  only before any StyleEntry is defined.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @param styleEntryCount Hint on the number of StyleEntry definitions
                *      that will be set.
                */
                void HintStyleEntryCount(const Char * const locale, Int styleEntryCount);
                
                /**
                 * Add a StyleEntry definition.
                 *
                 * The StyleEntry definition will be added with the given priority. The priority must not
                 *  be higher than the number of already defined entries. The font should be setup prior to
                 *  calling this function. Otherwise, if the font is invalid, it will not be returned by
                 *  GetFont(codePoint), even if the codePoint falls between the input range.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @param priority Priority of the entry to be added.
                 * @param font Font associated to this entry.
                 * @param lowerBound Lower code point bound that will use this font.
                 * @param upperBound Upper code point bound that will use this font.
                 * @return true if the priority is not higher than the number of StyleEntry definitions,
                 *  false otherwise.
                 */
                bool AddStyleEntry(Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound);

                /**
                * Add a StyleEntry definition.
                *
                * The StyleEntry definition will be added with the given priority. The priority must not
                *  be higher than the number of already defined entries. The font should be setup prior to
                *  calling this function. Otherwise, if the font is invalid, it will not be returned by
                *  GetFont(codePoint), even if the codePoint falls between the input range.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @param priority Priority of the entry to be added.
                * @param font Font associated to this entry.
                * @param lowerBound Lower code point bound that will use this font.
                * @param upperBound Upper code point bound that will use this font.
                * @return true if the priority is not higher than the number of StyleEntry definitions,
                *  false otherwise.
                */
                bool AddStyleEntry(const Char * const locale, Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound);

                /**
                 * Remove a StyleEntry definition.
                 *
                 * The StyleEntry definition will be removed. All the StyleEntry definitions with
                 *  a higher priority will decrease their priority.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @param priority Priority of the entry to be removed.
                 * @return true if StyleEntry with given priority is found and removed, false otherwise.
                 */
                bool RemoveStyleEntry(Int priority);

                /**
                * Remove a StyleEntry definition.
                *
                * The StyleEntry definition will be removed. All the StyleEntry definitions with
                *  a higher priority will decrease their priority.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @param priority Priority of the entry to be removed.
                * @return true if StyleEntry with given priority is found and removed, false otherwise.
                */
                bool RemoveStyleEntry(const Char * const locale, Int priority);
      
                /**
                 * Retrieve number of StyleEntry definitions using the default (active) locale.
                 * If the locale does not exist, this method returns zero.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @return Number of StyleEntry definitions.
                 */
                SizeType GetStyleEntryCount() const;

                /**
                * Retrieve number of StyleEntry definitions using a custom locale.
                * If the locale does not exist, this method returns zero.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @return Number of StyleEntry definitions.
                */
                SizeType GetStyleEntryCount(const Char* locale) const;

                /**
                 * Redefine a StyleEntry.
                 *
                 * Redefine a StyleEntry with the given priority. The priority must be lower than
                 *  the number of already defined style entries. The font should be setup prior to
                 *  calling this function. Otherwise, if the font is invalid, it will not be returned by
                 *  GetFont(codePoint), even if the codePoint falls between the input range.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @param priority Priority of this entry.
                 * @param font Font associated to this entry.
                 * @param lowerBound Lower code point bound that will use this font.
                 * @param upperBound Upper code point bound that will use this font.
                 * @return true if the priority is lower than the number of StyleEntry definitions,
                 *  false otherwise.
                 */
                bool SetStyleEntry(Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound);

                /**
                 * Retrieve a StyleEntry definition.
                 *
                 * If a StyleEntry definition was provided for the given priority, the method will return
                 *  the Font, lowerBound and upperBound defined in the StyleEntry.
                 *
                 * This operation applies to the currently active style entry list based on the active locale/culture.
                 *
                 * @param       priority Priority of requested entry.
                 * @param [out] font Font associated to the requested entry.
                 * @param [out] lowerBound Lower code point bound associated to the requested entry.
                 * @param [out] upperBound Upper code point bound associated to the requested entry.
                 * @return true if a StyleEntry is defined for the given priority, false otherwise.
                 */
                bool GetStyleEntry(Int priority, Font& font, Utf32& lowerBound, Utf32& upperBound) const;

                /**
                * Redefine a StyleEntry.
                *
                * Redefine a StyleEntry with the given priority. The priority must be lower than
                *  the number of already defined style entries. The font should be setup prior to
                *  calling this function. Otherwise, if the font is invalid, it will not be returned by
                *  GetFont(codePoint), even if the codePoint falls between the input range.
                *
                * @param locale is used to operate on a style entry list other than the active one.
                * @param priority Priority of this entry.
                * @param font Font associated to this entry.
                * @param lowerBound Lower code point bound that will use this font.
                * @param upperBound Upper code point bound that will use this font.
                * @return true if the priority is lower than the number of StyleEntry definitions,
                *  false otherwise.
                */
                bool SetStyleEntry(const Char * const locale, Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound);

                /**
                * Retrieve a StyleEntry definition.
                *
                * If a StyleEntry definition was provided for the given priority, the method will return
                *  the Font, lowerBound and upperBound defined in the StyleEntry.
                *
                * @param       locale is used to operate on a style entry list other than the active one.
                * @param       priority Priority of requested entry.
                * @param [out] font Font associated to the requested entry.
                * @param [out] lowerBound Lower code point bound associated to the requested entry.
                * @param [out] upperBound Upper code point bound associated to the requested entry.
                * @return true if a StyleEntry is defined for the given priority, false otherwise.
                */
                bool GetStyleEntry(const Char * const locale, Int priority, Font& font, Utf32& lowerBound, Utf32& upperBound) const;
   
                /**
                 * Set the base Style.
                 *
                 * The base Style will resolve the GetFont method calls for code points that
                 *  do not fall between any range in the StyleEntry definitions.
                 *
                 * @param style The base Style.
                 * @return true if this is not the given style or a direct or indirect base style
                 *  of the given style, false otherwise.
                 */
                bool SetBaseStyle(const MemoryManagement::SharedPointer<SharedStyle>& style);

                /**
                 * Get the base Style.
                 *
                 * @return The base Style.
                 */
                const MemoryManagement::SharedPointer<SharedStyle>& GetBaseStyle() const { return m_baseStyle; }

                /**
                 * Set the default Font.
                 *
                 * The default Font will be retrieved by the GetFont method for code points that
                 *  do not fall between any range in the StyleEntry definitions of either this or
                 *  the base style.
                 *
                 * @param font The default Font.
                 */
                void SetDefaultFont(const Font& font);

                /**
                 * Get the default Font.
                 *
                 * @return The default Font.
                 */
                const Font& GetDefaultFont() const;

                /**
                 * Set the default codepoint.
                 *
                 *  The default codepoint will be used during rendering to replace characters
                 *  that don't match any glyph in the style. 
                 *
                 * @param codepoint The default codepoint.
                 */
                void SetDefaultCodepoint(CodePoint codepoint);

                /**
                 * Get the default codepoint.
                 *
                 * @return The default codepoint.
                 */
                CodePoint GetDefaultCodepoint() const;

                /**
                 * Get the default codepoint for a font identified by @b identified
                 * @return Default codepoint for the font
                 */
                CodePoint GetDefaultCodepointByIdentifier(FontIdentifier identifier) const;

                /**
                * The active locale is used to change the style handling (font cascade) based on the culture/locale.
                * @param locale identifier 
                */
                void SetActiveLocale(const Char* locale);

                /**
                * The active locale is used to change the style handling (font cascade) based on the culture/locale.
                * @return locale identifier
                */
                const Char* GetActiveLocale() const;

                /**
                * Adds an event listener which listens to Candera::NotifyPropertyChangedEvents.
                * As soon as a property of style (or its base styles) changed the event will be sent.
                * @param eventListener which receives the event.
                * @return true if adding the event listener was successful.
                */
                virtual bool AddNotifyPropertyChangedEventListener(FeatStd::EventListener* eventListener) override;


                /**
                * Removes an event listener which listens to Candera::NotifyPropertyChangedEvents.
                * @param eventListener which received the event.
                * @return true if removing the event listener was successful.
                */
                virtual bool RemoveNotifyPropertyChangedEventListener(FeatStd::EventListener* eventListener) override;

                /**
                 * Handles events which are delegated to Style.
                 * It is currently used to send a NotifyPropertyChangedEvent to the attached listener as soon as the base style has changed.
                 * @param event the event which shall be handled by Style.
                 * @return FeatStd::EventResult::Proceed
                */
                virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& event);

            protected:
                virtual void NotifyPropertyChanged() override;
            private:
                struct StyleEntry {
                    Font m_font;
                    Utf32 m_lowerBound;
                    Utf32 m_upperBound;
                    StyleEntry(const Font& font, Utf32 lowerBound, Utf32 upperBound)
                        :m_font(font), m_lowerBound(lowerBound), m_upperBound(upperBound) {}
                };

                enum PriorityComposition {
                    AllCultures,     //< The selected culture + fallback culture
                    FallbackCulture, //< The fallback culture only
                    SelectedCulture  //< The selected culture only
                };

                struct LocaleComparator{
                    static Int32 Compare(const Char* left, const Char* right);
                };

                mutable bool m_isMetricsCacheValid;
                mutable UInt16 m_version;
                mutable UInt16 m_baseVersion;
                CodePoint m_defaultCodepoint;
                FeatStd::EventSource m_notifyPropertyChangedEventSource;
                MemoryManagement::SharedPointer<SharedStyle> m_baseStyle;
                typedef Candera::Internal::Vector<StyleEntry, FeatStd::Internal::LinearIncreasePolicy<1> > StyleEntries;
                Candera::Internal::Map<const Char*, StyleEntries, LocaleComparator> m_styleEntries;
                mutable Metrics m_metricsCache;
                Font m_defaultFont;
                const Char* m_locale;
                Candera::Globalization::CultureChangeListener * m_cultureChangeListener;
                FeatStd::EventListener * m_baseStyleEventListener;
                

                static void UpdateMetrics(const Metrics& newMetrics, Metrics& result);
                
                void InvalidateMetricsCache();

                bool IsStyleInvalidByCultureChange(const Char* newLocale) const;

                bool IsCustomLocaleFontAvailable(const Char* locale) const;

                bool IsCodepointInFontRange(StyleEntry const& entry, Utf32 const codepoint) const;

                Int GetBaseMaximumCompositePriority(const Char* locale, PriorityComposition priorityComposition, bool includeBase) const;

                Int GetMaximumCompositePriority(const Char* locale, PriorityComposition priorityComposition, bool includeBase) const;

                bool IsValidBase(Style* style) const;

                void SetCurrentCulture();

                /**
                * Represents the current version of a style.
                * The version of a style changes when any property has been changed. 
                * It also changes its version if a base style internally changes a property.
                * @return version of style which indicates changes within the style
                */
                UInt16 GetVersion() const;

                /**
                * Access to the correct localized style entry container
                * This method has access to base styles!
                * This function is provided to only receive results of the style.
                * The result cannot and should not be altered as the result can come from a base style.
                *
                * @param locale a user defined culture to select the associated font cascade (StyleEntries)
                * @return The font cascade of the current active culture (Should only be used for the receiving the result - not altering it)
                */
                const StyleEntries * GetStyleEntries(const Char* locale) const;

                /**
                * Access to the correct localized style entry container
                * This method has no access to base styles!
                * This is intended behavior as this function provided to alter the style.
                * This can happen directly within the method or indirectly by operations on the pointer.
                * Altering the style should never alter the base style.
                *
                * @param locale a user defined culture to select the associated font cascade (StyleEntries)
                * @param createNewLocale is used for direct altering (if a culture font cascade does not exist, it creates one)
                * @return The font cascade of the user defined culture (result can be altered - has only impact on this style)
                */
                StyleEntries * GetStyleEntries(const Char* locale, bool createNewLocale);

                /**
                * Returns the style entries with locale zero. Which is the locale when no culture is set.
                * @return the default style entries - returns null pointer when allocation of default entries failed.
                */
                const StyleEntries * GetDefaultStyleEntries() const;
                
                Int GetCompositePriorityWithFixedLocale(const Char* locale, Utf32 codePoint, Int maximumCompositePriority) const;

                /**
                * Retrieve the font for the given composite priority.
                * The locale is already known. It can be assumed that the locale will not change back to a fallback,
                * as this decision has been made already
                *
                * If the composite priority is -1, than the default font is returned.
                * @param locale is used to operate on a style entry list other than the active one.
                * @param compisitePriority Composite priority of the requested Font.
                * @return Font with the given composite priority.
                */
                const Font& GetFontByCompositePriorityWithCorrectLocaleSelected(const Char* locale, Int compositePriority) const;


                const Candera::TextRendering::Font& GetValidDefaultFont() const;


        };

        /**
         * Defines a shareable Style.
         *
         * @see Candera::TextRendering::Style
         */
        class SharedStyle: public Style {
            public:
                typedef MemoryManagement::SharedPointer<SharedStyle> SharedPointer;

                /**
                 * Construct a new shared Style.
                 *
                 * @return the new shared Style.
                 */
                FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            private:
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, Use SharedStyle::Create)
                SharedStyle() {}

                FEATSTD_LINT_NEXT_EXPRESSION(1511, "false positive - SharedStyle::SharedPointer is also used in base class")
                CANDERA_SHARED_POINTER_DECLARATION();
        };

        /** @} */ // end of CanderaTextEngine

    }// namespace TextRendering

}// namespace Candera


#endif// Candera_Style_h
