//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Style.h"
#include <FeatStd/Platform/Thread.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <Candera/System/Diagnostics/Log.h>

#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>
#include <Candera/TextEngine/TextRenderer.h>
#include <Candera/System/GlobalizationBase/Culture.h>
#include <Candera/System/GlobalizationBase/CultureChangeListener.h>
#include <Candera/System/GlobalizationBase/CultureManager.h>
#include <Candera/TextEngine/Internal/StyleTools.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);

    using namespace MemoryManagement;
    namespace TextRendering {

        class StyleCultureChangeListener : public Candera::Globalization::CultureChangeListener {
        public:

            StyleCultureChangeListener(Style& style) :m_style(style)
            {
                Candera::Globalization::CultureManager::GetInstance().AddCultureChangeListener(this);
            }

            virtual ~StyleCultureChangeListener()
            {
                bool result = Candera::Globalization::CultureManager::GetInstance().RemoveCultureChangeListener(this);
                FEATSTD_UNUSED(result);
                FEATSTD_DEBUG_ASSERT(result);
                // check whether culture change listener were added more than a single time
                FEATSTD_DEBUG_ASSERT(!Candera::Globalization::CultureManager::GetInstance().RemoveCultureChangeListener(this));
            }

        protected:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(StyleCultureChangeListener);
            Style& m_style;

            virtual void OnPreCultureChanged(const Candera::Globalization::Culture& culture) override
            {
                m_style.SetActiveLocale(culture.GetLocale());
            }


            virtual void OnCultureChanged(const Candera::Globalization::Culture&) override{ /* Nothing to do here - everything has to happen in pre hook */ }


#ifdef FEATSTD_THREADSAFETY_ENABLED
            virtual void Obtain()
            {
                (void)FeatStd::Internal::AtomicOp::Inc(m_obtainCounter);
            }

            virtual void Release()
            {
                (void)FeatStd::Internal::AtomicOp::Dec(m_obtainCounter);
            }

            virtual void WaitForRelease()
            {
                while (!FeatStd::Internal::AtomicOp::TestAndSet(m_obtainCounter, 0, 0)) {
                    FeatStd::Internal::Thread::Sleep(5);
                }
            }

            FeatStd::Internal::AtomicOp::Atomic m_obtainCounter;
#endif
        };

        class BaseStyleChangeListener : public FeatStd::EventListener {
        public:
            BaseStyleChangeListener(Style& style) :m_style(style)
            {

            }
            ~BaseStyleChangeListener()
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                this->WaitForRelease();
#endif
            }
            virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& event) override
            {
                return m_style.OnEvent(event);
            }

        protected:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(BaseStyleChangeListener);
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FEATSTD_EVENTLISTENER_STANDARD_THREADSAFETY;
#endif
            Style& m_style;
        };

        FEATSTD_LINT_CURRENT_SCOPE(1938, "Global variable is not being modified.")
            Style::Style(const Font& font) :
            m_isMetricsCacheValid(false),
            m_version(0),
            m_baseVersion(0),
            m_defaultCodepoint(Internal::FullBoundCodePointIterator::c_defaultCharacter),
            m_metricsCache(Metrics()),
            m_defaultFont(font),
            m_locale(0),
            m_cultureChangeListener(0),
            m_baseStyleEventListener(0)
        {
            m_cultureChangeListener = TEXTENGINE_TRANSIENT_NEW(StyleCultureChangeListener)(*this);
            static_cast<void>(GetStyleEntries(0, true));
            SetCurrentCulture();
            m_baseStyleEventListener = TEXTENGINE_TRANSIENT_NEW(BaseStyleChangeListener)(*this);
            if (m_baseStyleEventListener == 0) {
                FEATSTD_LOG_ERROR("Creation of a base style event listener failed!");
            }
            if (!m_baseStyle.PointsToNull()) {
                if (m_baseStyleEventListener != 0) {
                    if(!m_baseStyle->AddNotifyPropertyChangedEventListener(m_baseStyleEventListener)) {
                        FEATSTD_LOG_ERROR("Style was not able to register the event listener to the base style!");
                    }
                }
            }
        }

        Style::Style() :
            m_isMetricsCacheValid(false),
            m_version(0),
            m_baseVersion(0),
            m_defaultCodepoint(Internal::FullBoundCodePointIterator::c_defaultCharacter),
            m_metricsCache(Metrics()),
            m_defaultFont(),
            m_locale(0),
            m_cultureChangeListener(0),
            m_baseStyleEventListener(0)
        {
            m_cultureChangeListener = TEXTENGINE_TRANSIENT_NEW(StyleCultureChangeListener)(*this);
            static_cast<void>(GetStyleEntries(0, true));
            SetCurrentCulture();
            m_baseStyleEventListener = TEXTENGINE_TRANSIENT_NEW(BaseStyleChangeListener)(*this);
            if (m_baseStyleEventListener == 0) {
                FEATSTD_LOG_ERROR("Creation of a base style event listener failed!");
            }
            if (!m_baseStyle.PointsToNull()) {
                if (m_baseStyleEventListener != 0) {
                    if(!m_baseStyle->AddNotifyPropertyChangedEventListener(m_baseStyleEventListener)) {
                        FEATSTD_LOG_ERROR("Style was not able to register the event listener to the base style!");
                    }
                }
            }
        }


        Style::~Style()
        {
            if (!m_baseStyle.PointsToNull()) {
                if (m_baseStyleEventListener != 0) {
                    if(!m_baseStyle->RemoveNotifyPropertyChangedEventListener(m_baseStyleEventListener)) {
                        FEATSTD_LOG_ERROR("Style was not able to unregister the event listener to the base style!");
                    }
                }
            }
            if (m_baseStyleEventListener != 0) {
                TEXTENGINE_DELETE(m_baseStyleEventListener);
                m_baseStyleEventListener = 0;
            }
            if (m_cultureChangeListener != 0) {
                FEATSTD_SAFE_DELETE(m_cultureChangeListener);
            }
            m_locale = 0;
        }

        void Style::HintStyleEntryCount(Int styleEntryCount)
        {
            return HintStyleEntryCount(GetActiveLocale(), styleEntryCount);
        }

        void Style::HintStyleEntryCount(const Char * const locale, Int styleEntryCount)
        {
            if (styleEntryCount > 0) {
                StyleEntries * se = GetStyleEntries(locale, true);
                FEATSTD_DEBUG_ASSERT(se != 0); // can only be 0 when memmory allocation fails
                if (se != 0) {
                    static_cast<void>(se->Reserve(styleEntryCount));
                }
            }
        }

        bool Style::AddStyleEntry(Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound)
        {
            return AddStyleEntry(GetActiveLocale(), priority, font, lowerBound, upperBound);
        }

        bool Style::AddStyleEntry(const Char * const locale, Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound)
        {
            if (priority < 0) {
                return false;
            }
            StyleEntries * se = GetStyleEntries(locale, true);
            FEATSTD_DEBUG_ASSERT(se != 0);
            if ((se == 0) || (!se->Insert(static_cast<SizeType>(priority), StyleEntry(font, lowerBound, upperBound)))) {
                return false;
            }

            InvalidateMetricsCache();
            return true;
        }

        bool Style::RemoveStyleEntry(Int priority)
        {
            return RemoveStyleEntry(GetActiveLocale(), priority);
        }

        bool Style::RemoveStyleEntry(const Char * const locale, Int priority)
        {
            if (priority < 0) {
                return false;
            }
            StyleEntries * se = GetStyleEntries(locale, false);
            if ((se == 0) || (!se->Remove(static_cast<SizeType>(priority)))) {
                return false;
            }

            InvalidateMetricsCache();
            return true;
        }

        Candera::SizeType Style::GetStyleEntryCount(const Char* locale) const
        {
            const StyleEntries * se = GetStyleEntries(locale);
            if (se == 0) {
                return 0;
            }
            return GetStyleEntries(locale)->Size();
        }

        Candera::SizeType Style::GetStyleEntryCount() const
        {
            return GetStyleEntryCount(GetActiveLocale());
        }

        bool Style::SetStyleEntry(Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound)
        {
            return SetStyleEntry(GetActiveLocale(), priority, font, lowerBound, upperBound);
        }

        bool Style::GetStyleEntry(Int priority, Font& font, Utf32& lowerBound, Utf32& upperBound) const
        {
            return GetStyleEntry(GetActiveLocale(), priority, font, lowerBound, upperBound);
        }

        bool Style::SetStyleEntry(const Char * const locale, Int priority, const Font& font, Utf32 lowerBound, Utf32 upperBound)
        {
            if ((priority < 0) || (priority >= FeatStd::Internal::NumericConversion<Int>(GetStyleEntryCount(locale)))) {
                return false;
            }
            StyleEntries * se = GetStyleEntries(locale, false);
            FEATSTD_DEBUG_ASSERT(se != 0); // this cannot happen - GetStyleEntryCount would be 0 otherwise
            (*se)[priority] = StyleEntry(font, lowerBound, upperBound);
            InvalidateMetricsCache();
            return true;
        }

        bool Style::GetStyleEntry(const Char * const locale, Int priority, Font& font, Utf32& lowerBound, Utf32& upperBound) const
        {
            if ((priority < 0) || (priority >= FeatStd::Internal::NumericConversion<Int>(GetStyleEntryCount(locale)))) {
                return false;
            }
            const StyleEntries * se = GetStyleEntries(locale);
            FEATSTD_DEBUG_ASSERT(se != 0); // this cannot happen - GetStyleEntryCount would be 0 otherwise

            font = (*se)[priority].m_font;
            lowerBound = (*se)[priority].m_lowerBound;
            upperBound = (*se)[priority].m_upperBound;
            return true;
        }

        bool Style::SetBaseStyle(const MemoryManagement::SharedPointer<SharedStyle>& style)
        {
            if ((!style.PointsToNull()) && (!style->IsValidBase(this))) {
                return false;
            }
            if (!m_baseStyle.PointsToNull()) {
                if (m_baseStyleEventListener != 0) {
                    if(!m_baseStyle->RemoveNotifyPropertyChangedEventListener(m_baseStyleEventListener)) {
                        FEATSTD_LOG_ERROR("Style was not able to unregister the event listener to the base style!");
                    }
                }
            }
            m_baseStyle = style;
            if (!style.PointsToNull()) {
                if (m_baseStyleEventListener != 0) {
                    if(!style->AddNotifyPropertyChangedEventListener(m_baseStyleEventListener)) {
                        FEATSTD_LOG_ERROR("Style was not able to register the event listener to the base style!");
                    }
                }
            }
            InvalidateMetricsCache();
            return true;
        }

        void Style::SetDefaultFont(const Font& font)
        {
            InvalidateMetricsCache();
            m_defaultFont = font;
        }

        const Candera::TextRendering::Font& Style::GetDefaultFont() const
        {
            return m_defaultFont;
        }

        void Style::SetDefaultCodepoint(CodePoint codepoint)
        {
            ++m_version;
            m_defaultCodepoint = codepoint;
            NotifyPropertyChanged();
        }

        Candera::TextRendering::CodePoint Style::GetDefaultCodepoint() const
        {
            return m_defaultCodepoint;
        }

        CodePoint Style::GetDefaultCodepointByIdentifier(FontIdentifier identifier) const
        {
            if (identifier.m_value >= 0) {
                return GetDefaultCodepoint();
            }
            else 
            {
                const Style* baseStyle = this;
                while (baseStyle != 0) {
                    if (identifier.m_value == -1) {
                        return baseStyle->GetDefaultCodepoint();
                    }
                    baseStyle = baseStyle->GetBaseStyle().GetPointerToSharedInstance();
                    ++identifier.m_value;
                }
            }
            return 0;
        }

        void Style::SetActiveLocale(const Char* locale)
        {
            if (IsStyleInvalidByCultureChange(locale)) {
                InvalidateMetricsCache();
            }
            m_locale = locale;
        }

        const Candera::Char* Style::GetActiveLocale() const
        {
            return m_locale;
        }


        bool Style::AddNotifyPropertyChangedEventListener(FeatStd::EventListener* eventListener)
        {
            return m_notifyPropertyChangedEventSource.AddEventListener(eventListener);
        }

        bool Style::RemoveNotifyPropertyChangedEventListener(FeatStd::EventListener* eventListener)
        {
            return m_notifyPropertyChangedEventSource.RemoveEventListener(eventListener);
        }


        FeatStd::EventResult::Enum Style::OnEvent(const FeatStd::Event& event)
        {
            const NotifyPropertyChangedEvent * notifyEvent = Candera::Dynamic_Cast<const NotifyPropertyChangedEvent*>(&event);
            if (notifyEvent != 0) {
                if (notifyEvent->GetBaseObject()->GetTypeId() == Candera::TextRendering::Style::GetTypeId()) {
                    InvalidateMetricsCache();
                }
            }
            return FeatStd::EventResult::Proceed;
        }

        Int Style::GetCompositePriority(Utf32 codePoint, Int maximumCompositePriority) const
        {
            return GetCompositePriority(GetActiveLocale(), codePoint, maximumCompositePriority);
        }

        bool Style::IsCodepointInFontRange(StyleEntry const& entry, Utf32 const codepoint) const
        {
            return ((codepoint >= entry.m_lowerBound) && (codepoint <= entry.m_upperBound) && (entry.m_font.IsValid()));
        }

        Int Style::GetMaximumCompositePriority(const Char* locale, PriorityComposition priorityComposition, bool includeBase) const
        {
            Int result = 0;
            const StyleEntries * styleEntries;
            if (locale == 0) {// in this case selected culture == fallback culture (dont count it twice)
                priorityComposition = SelectedCulture;
            }
            if ((priorityComposition == AllCultures) || (priorityComposition == SelectedCulture)) {
                styleEntries = GetStyleEntries(locale);
                if (styleEntries != 0) {
                    result += static_cast<Int>(styleEntries->Size());
                }
            }
            if ((priorityComposition == AllCultures) || (priorityComposition == FallbackCulture)) {
                styleEntries = GetStyleEntries(0);
                if (styleEntries != 0) {
                    result += static_cast<Int>(styleEntries->Size());
                }
            }
            if (includeBase) {
                result += GetBaseMaximumCompositePriority(locale, priorityComposition, includeBase);
            }
            return result;
        }

        Int Style::GetCompositePriority(const Char* locale, Utf32 codePoint, Int maximumCompositePriority) const
        {
            // find out max prio level. Get border between custom culture and no culture.
            Int maxDefaultPriority = GetMaximumCompositePriority(locale, FallbackCulture, true);
            FEATSTD_DEBUG_ASSERT(GetMaximumCompositePriority(locale, AllCultures, true) >= maxDefaultPriority);
            FEATSTD_DEBUG_ASSERT(((locale == 0) && (GetMaximumCompositePriority(locale, AllCultures, true) == maxDefaultPriority)) || (locale != 0));
            Int maxInputPriority = maximumCompositePriority;
            Int resultPriority = -1;

            if (maxInputPriority > maxDefaultPriority) {
                FEATSTD_DEBUG_ASSERT(locale != 0);
                resultPriority = GetCompositePriorityWithFixedLocale(locale, codePoint, maxInputPriority - maxDefaultPriority);
                if (resultPriority >= 0) {
                    FEATSTD_DEBUG_ASSERT(resultPriority < maxInputPriority);
                    return resultPriority + maxDefaultPriority;
                }
                maxInputPriority = maxDefaultPriority;
            }

            FEATSTD_DEBUG_ASSERT(maximumCompositePriority >= maxInputPriority);
            FEATSTD_DEBUG_ASSERT(maxInputPriority <= maxDefaultPriority);
            return GetCompositePriorityWithFixedLocale(0, codePoint, maxInputPriority);
        }

        Int Style::GetMaximumCompositePriority() const
        {
            return GetMaximumCompositePriority(GetActiveLocale());
        }

        Int Style::GetMaximumCompositePriority(const Char* locale) const
        {
            return GetMaximumCompositePriority(locale, AllCultures, true);
        }

        const Font& Style::GetFontByCompositePriority(Int compositePriority) const
        {
            return GetFontByCompositePriority(GetActiveLocale(), compositePriority);
        }

        const Candera::TextRendering::Font& Style::GetFontByCompositePriority(const Char* locale, Int compositePriority) const
        {
            Int maxCompositePriority = GetMaximumCompositePriority(locale, AllCultures, true);
            // check if compositePriority is available or not
            if ((compositePriority < 0) || (compositePriority >= maxCompositePriority)) {
                return GetValidDefaultFont();
            }
            Int maxDefaultCompositePriority = maxCompositePriority;
            if (locale != 0) {
                maxDefaultCompositePriority = GetMaximumCompositePriority(locale, FallbackCulture, true);
            }
            else {
                // this else is only for consistency check
                FEATSTD_DEBUG_ASSERT(maxCompositePriority == GetMaximumCompositePriority(locale, FallbackCulture, true));
            }

            if (compositePriority < maxDefaultCompositePriority) {
                // has to be a fallback culture - so the actual culture is not available.
                // so continue with the fallback culture
                locale = 0;
            }
            else {
                // adjust the priority level down to the culture only level
                compositePriority -= maxDefaultCompositePriority;
            }
            return GetFontByCompositePriorityWithCorrectLocaleSelected(locale, compositePriority);
        }

        const Candera::TextRendering::Font& Style::GetFontByCodePoint(Utf32 codePoint) const
        {
            return GetFontByCodePoint(GetActiveLocale(), codePoint);
        }

        const Candera::TextRendering::Font& Style::GetFontByCodePoint(const Char* locale, Utf32 codePoint) const
        {
            return GetFontByCompositePriority(locale, GetCompositePriority(locale, codePoint, GetMaximumCompositePriority(locale)));
        }

        const Metrics& Style::GetMetrics() const
        {

            GetMetrics(GetActiveLocale(), m_metricsCache);
            return m_metricsCache;
        }

        void Style::GetMetrics(const Char* locale, Metrics& result) const
        {

            if ((!m_isMetricsCacheValid) || (locale != GetActiveLocale())) {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(TextRendering::Internal::TextRenderLocks::GetGlyphLock());              
#endif
                result.ascender = 0;
                result.descender = 0;
                result.lineHeight = 0;
                result.maxAdvance = 0;
                Metrics tempMetrics = Metrics();
                const StyleEntries * styleEntries = GetStyleEntries(0);
                // default/fallback culture
                if (styleEntries != 0) {
                    for (SizeType entryIndex = 0; entryIndex < styleEntries->Size(); ++entryIndex) {
                        if ((*styleEntries)[entryIndex].m_font.GetMetrics(tempMetrics)) {
                            UpdateMetrics(tempMetrics, result);
                        }
                    }
                }

                // selected culture
                // if selected culture is 0 it were already 'upated' before
                if (locale != 0) {
                    styleEntries = GetStyleEntries(locale);
                    if (styleEntries != 0) {
                        for (SizeType entryIndex = 0; entryIndex < styleEntries->Size(); ++entryIndex) {
                            if ((*styleEntries)[entryIndex].m_font.GetMetrics(tempMetrics)) {
                                UpdateMetrics(tempMetrics, result);
                            }
                        }
                    }
                }

                if (!m_baseStyle.PointsToNull()) {
                    m_baseStyle->GetMetrics(locale, tempMetrics);
                    UpdateMetrics(tempMetrics, result);
                }

                if (m_defaultFont.IsValid()) {
                    if (m_defaultFont.GetMetrics(tempMetrics)) {
                        UpdateMetrics(tempMetrics, result);
                    }
                }

                if ((!m_isMetricsCacheValid) && (locale == GetActiveLocale())) {
                    m_metricsCache = result;
                    m_isMetricsCacheValid = true;
                }
            }
            else {
                result = m_metricsCache;
            }
        }

        void Style::UpdateMetrics(const Metrics& newMetrics, Metrics& result)
        {
            if (newMetrics.ascender > result.ascender) {
                result.ascender = newMetrics.ascender;
            }
            if (newMetrics.descender < result.descender) {
                result.descender = newMetrics.descender;
            }
            if (newMetrics.lineHeight > result.lineHeight) {
                result.lineHeight = newMetrics.lineHeight;
            }
            if (newMetrics.maxAdvance > result.maxAdvance) {
                result.maxAdvance = newMetrics.maxAdvance;
            }
        }

        void Style::InvalidateMetricsCache()
        {
            m_isMetricsCacheValid = false; ++m_version;
            NotifyPropertyChanged();
        }

        bool Style::IsStyleInvalidByCultureChange(const Char* newLocale) const
        {
            if (newLocale == m_locale) {
                return false;
            }
            // if a custom locale (not 0) were set before, the style has changed for sure
            // as the new locale is not equal to the old one, and nevertheless it changes to 'default' locale
            // or the new custom locale
            if ((m_locale != 0) && (IsCustomLocaleFontAvailable(m_locale))) {
                return true;
            }
            // this code is reached when: current locale is 0 or the previous locale had no effect (used locale 0 only)
            // so when new locale also uses locale 0 only, nothing has to be changed
            if (newLocale == 0) {
                return false;
            }
            return IsCustomLocaleFontAvailable(newLocale);
        }

        bool Style::IsCustomLocaleFontAvailable(const Char* locale) const
        {
            if (GetStyleEntries(locale) != 0) {
                return true;
            }
            if (!m_baseStyle.PointsToNull()) {
                return m_baseStyle->IsStyleInvalidByCultureChange(locale);
            }
            return false;
        }

        bool Style::IsValidBase(Style* style) const
        {
            return (0 != style) && (this != style) && ((m_baseStyle.PointsToNull())?(true):(m_baseStyle->IsValidBase(style)));
        }


        void Style::SetCurrentCulture()
        {
            Candera::Globalization::Culture::SharedPointer currentCulture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();
            if (!currentCulture.PointsToNull()) {
                SetActiveLocale(currentCulture->GetLocale());
            }
            else {
                SetActiveLocale(0);
            }
        }

        Int Style::GetBaseMaximumCompositePriority(const Char* locale, PriorityComposition priorityComposition, bool includeBase) const
        {
            return (m_baseStyle.PointsToNull()) ? 0 : m_baseStyle->GetMaximumCompositePriority(locale, priorityComposition, includeBase);
        }
        
        UInt16 Style::GetVersion() const
        {
            if ((!m_baseStyle.PointsToNull()) && (m_baseVersion != m_baseStyle->GetVersion())) {
                ++m_version;
                m_baseVersion = m_baseStyle->GetVersion();
            }
            return m_version;
        }

        const Candera::TextRendering::Style::StyleEntries * Style::GetStyleEntries(const Char* locale) const
        {
            const StyleEntries * styleEntries = m_styleEntries.Find(locale);
            return styleEntries;
        }

        Candera::TextRendering::Style::StyleEntries * Style::GetStyleEntries(const Char* locale, bool createNewLocale)
        {
            StyleEntries * styleEntries = m_styleEntries.Find(locale);
            if (styleEntries == 0) {
                if (createNewLocale) {
                    StyleEntries se;
                    static_cast<void>(m_styleEntries.Insert(locale, se));
                    styleEntries = m_styleEntries.Find(locale);// returns 0 if allocation/insert failed
                }
            }
            return styleEntries;
        }

        const Candera::TextRendering::Style::StyleEntries * Style::GetDefaultStyleEntries() const
        {
            const StyleEntries * styleEntries = m_styleEntries.Find(0);
            // Default has to exist - created in constructor. Even if there are no entries in it
            FEATSTD_DEBUG_ASSERT(styleEntries != 0);
            return styleEntries;
        }

        Int Style::GetCompositePriorityWithFixedLocale(const Char* locale, Utf32 codePoint, Int maximumCompositePriority) const
        {
            Int baseMaxPriority = GetBaseMaximumCompositePriority(locale, SelectedCulture, true);
            if (maximumCompositePriority > baseMaxPriority) {// style has entries which have to be checked 
                const StyleEntries * styleEntries = GetStyleEntries(locale);
                FEATSTD_DEBUG_ASSERT(styleEntries != 0); // entries have to be available - otherwise a calculation were made completely wrong before
                FEATSTD_DEBUG_ASSERT(styleEntries->Size() >= static_cast<SizeType>(maximumCompositePriority - baseMaxPriority));
                while (maximumCompositePriority > baseMaxPriority) {
                    --maximumCompositePriority;
                    Int entryIndex = maximumCompositePriority - baseMaxPriority;
                    if (IsCodepointInFontRange((*styleEntries)[entryIndex], codePoint)) {
                            return maximumCompositePriority;
                    }
                }
            }

            if (maximumCompositePriority > 0) {
                if (!m_baseStyle.PointsToNull()) {
                    return m_baseStyle->GetCompositePriorityWithFixedLocale(locale, codePoint, maximumCompositePriority);
                }
            }

            return -1;
        }

        const Candera::TextRendering::Font& Style::GetFontByCompositePriorityWithCorrectLocaleSelected(const Char* locale, Int compositePriority) const
        { 
            // previously we reduced it to either the selected culture or the fallback culture. So the actual decision were already done
            Int baseMaxPriority = GetBaseMaximumCompositePriority(locale, SelectedCulture, true);

            // it is assumed that the caller of this method already checked the boundaries - otherwise he would not be able to determine
            // whether it is a valid priority or not - Selecting a base 'default font' were never possible to begin with.
            // This will been done in StyleTools
            FEATSTD_DEBUG_ASSERT(!((compositePriority < 0) || (compositePriority >= GetMaximumCompositePriority(locale, SelectedCulture, true))));

            if (compositePriority < baseMaxPriority) {
                return m_baseStyle->GetFontByCompositePriorityWithCorrectLocaleSelected(locale, compositePriority);
            }
            const StyleEntries * styleEntries = GetStyleEntries(locale);
            FEATSTD_DEBUG_ASSERT(compositePriority >= baseMaxPriority);
            FEATSTD_DEBUG_ASSERT((styleEntries != 0) && (styleEntries->Size() > static_cast<SizeType>(compositePriority - baseMaxPriority)));
            if (styleEntries == 0) {
                return m_defaultFont;
            }
            return (*styleEntries)[FeatStd::Internal::NumericConversion<SizeType>(compositePriority - baseMaxPriority)].m_font;
        }

        const Candera::TextRendering::Font& Style::GetValidDefaultFont() const
        {
            if (m_defaultFont.IsValid()) {
                return m_defaultFont;
            }
            // if nothing is valid - and no default font exists, the returned default font
            // is automatically an invalid font (default ctor of font)
            return (m_baseStyle != 0)?(m_baseStyle->GetValidDefaultFont()):(m_defaultFont);
        }


        void Style::NotifyPropertyChanged()
        {
            NotifyPropertyChangedEvent e(this);
            m_notifyPropertyChangedEventSource.DispatchEvent(e);
        }

        SharedStyle::SharedPointer SharedStyle::Create()
        {
            return SharedStyle::SharedPointer(TEXTENGINE_TRANSIENT_NEW(SharedStyle));
        }


        Candera::Int32 Style::LocaleComparator::Compare(const Char* left, const Char* right)
        {
            if (left == right) {
                return 0;
            }
            if (left == 0) {
                return -1;
            }
            if (right == 0) {
                return 1;
            }
            FEATSTD_DEBUG_ASSERT((left != 0) && (right != 0));
            return FeatStd::Internal::String::CompareStrings(left, right);
        }

    }// namespace TextRendering

}// namespace Candera
