//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_ShapingOptions_h)
#define Candera_ShapingOptions_h

#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/System/GlobalizationBase/Culture.h>

namespace Candera {
    namespace TextRendering {

/** @addtogroup CanderaTextEngine
 *  @{
 */
        
        namespace ShaperType {
            enum Enum {
                Default,
                NoShaper,
                ComplexScript,
                HarfBuzz,
                WorldTypeShaper,
                _ShaperType_LAST_ITEM
            };
        }

        /**
         *  @brief This class defines how the text is shaped.
         *
         *  Shaping options define the style used to draw text.
         *  They also cover additional options needed to use
         *  special feature of the text engine, like creating ligatures,
         *  changing text direction etc.
         */
        class ShapingOptions {
            public:
                /**
                 *  Constructor with no options. This object can not be used to render text
                 *  At least a style should be bound to it before.
                 */
                ShapingOptions();

                /**
                 *  Constructor with style. The style is bound to this object. See Bind*.
                 *  @param style Bound style.
                 *  @param bidiBaseLevel sets the base level of the text. This will decide of the initial text flow direction.
                 */
                ShapingOptions(const SharedStyle::SharedPointer& style, BidiBaseLevel::Enum bidiBaseLevel = BidiBaseLevel::Deprecated);

                /**
                 *  Constructor with style and culture. The style and the culture are
                 *  bound to this object. See Bind*.
                 *  @param style Bound style.
                 *  @param culture Bound culture.
                 *  @param bidiBaseLevel sets the base level of the text. This will decide of the initial text flow direction.
                 */
                ShapingOptions(const SharedStyle::SharedPointer& style, const Globalization::Culture::SharedPointer& culture, BidiBaseLevel::Enum bidiBaseLevel = BidiBaseLevel::Deprecated);

                /**
                 *  Binds a style to this object. The bound style should not be changed for the
                 *  duration of this binding. To remove the binding, bind a different style or
                 *  destroy this object.
                 *  @param style Bound style. 0 to remove binding.
                 */
                void BindStyle(const SharedStyle::SharedPointer& style);

                /**
                 *  Retrieve the style bound to this object. This method can be used to check if the
                 *  a style is bound. This is mandatory for text rendering.
                 *  @return Bound style. 0 if no style was bound.
                 */
                const SharedStyle::SharedPointer& GetStyle() const;

                /**
                 *  Binds a culture to this object. The bound culture should not be changed for the
                 *  duration of this binding. To remove the binding, bind a different culture or
                 *  destroy this object.
                 *  @param culture Bound culture. 0 to remove binding.
                 */
                void BindCulture(const Globalization::Culture::SharedPointer& culture);

                /**
                 *  Retrieve the culture bound to this object. This method can be used to check if the
                 *  a culture is bound.
                 *  @return Bound culture. 0 if no culture was bound.
                 */
                const Globalization::Culture::SharedPointer& GetCulture() const;

                ShaperType::Enum GetShaperType() const;

                /**
                *  Retrieve the selected bidi base level for parsing/shaping.
                *  @return selected bidi base level.
                */
                Candera::TextRendering::BidiBaseLevel::Enum GetBidiBaseLevel() const;

                /**
                *  Sets a bidi base level for parsing/shaping.
                *  @param: val the bidi base level which shall be set.
                */
                void SetBidiBaseLevel(Candera::TextRendering::BidiBaseLevel::Enum val);
        private:
                SharedStyle::SharedPointer m_style;
                Globalization::Culture::SharedPointer m_culture;
                ShaperType::Enum m_shaperType;
                BidiBaseLevel::Enum m_bidiBaseLevel;
        };

 /** @} */ // end of CanderaTextEngine

    }    // namespace TextRendering

}    // namespace Candera

#endif    // Candera_ShapingOptions_h
