//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_PreprocessingContext_h)
#define Candera_PreprocessingContext_h

#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/TextMeasureContext.h>

namespace Candera { namespace TextRendering { 
    class Style;

/** @addtogroup CanderaTextEngine
 *  @{
 */

/**
 *  @brief PreprocessingContext is used to store glyph information, to simplify.
 *  text rendering in subsequent render passes.
 *
 *  This is a TextRenderContext that may be used with TextRenderer::Render.
 *  The render call will collect glyph information within the
 *  PreprocessingContext. This represents all the data required for rendering
 *  glyphs when calling TextRenderer::RenderPreprocessedGlyphs.
 */
class PreprocessingContext : public TextMeasureContext
{
protected:
    class ReferenceIterator;

public:
    /**
     * @brief Class used as interface for accessing preprocessed data.
     */
    class GlyphData
    {
    public:
        /**
         * Retrieve the font identifier, by which the font for this glyph is
         * identified within the style.
         * @return The font identifier.
         */
        virtual FontIdentifier GetFontIdentifier() const = 0;

        /**
         * Retrieve the glyph index, by which this glyph is identified within
         * the font.
         * @return The glyph index.
         */
        virtual GlyphIndex GetGlyphIndex() const = 0;

        /**
         * Retrieve the position of the glyph within the original text.
         * @return The character position.
         */
        virtual TextPosition GetCharacterPosition() const = 0;

        /**
         * Retrieve the position of the current glyph.
         * This is the position at which the glyph bitmaps is located.
         * @return A structure containing the position.
         */
        virtual PixelPosition2D GetPosition() const = 0;

        /**
        * Retrieve the xAdvance of the current glyph.
        * This is the advance from the glyph bitmap position to the next one in x direction.
        * @return The xAdvance.
        */
        virtual PixelPosition GetXAdvance() const {
            return 0;
        }
    };

    /**
     * @brief Type used for iterating over the glyph information.
     */
    class Iterator {
    public:
        inline Iterator();
        inline Iterator(const Iterator& other);
        inline ~Iterator();

        /**
         * Check iterator validity.
         * @return Whether the iterator is valid and can be dereferenced or not.
         */
        inline bool IsValid() const;

        /**
         * Indirection operator. Dereferencing an invalid iterator leads to
         * undefined behavior.
         * @return Referenced object.
         */
        inline const GlyphData& operator*() const;

        /**
         * Dereference operator. Dereferencing an invalid iterator leads to
         * undefined behavior.
         * @return Dereferenced object.
         */
        inline const GlyphData* operator->() const;

        /**
         * Increment iterator. Incrementing an invalid iterator leads to
         * undefined behavior.
         * @return Iterator in the state found after being incremented.
         */
        inline Iterator& operator++();

        /**
         * Equal to operator.
         * @param other Operand to compare this iterator to.
         * @result true if operators are equal.
         * @result false otherwise.
         */
        inline bool operator==(const Iterator& other) const;

        /**
         * Not equal to operator.
         * @param other Operand to compare this iterator to.
         * @result false if operators are equal.
         * @result true otherwise.
         */
        inline bool operator!=(const Iterator& other) const;

        /**
         * Copy operator.
         * @param other Operand to compare this iterator to.
         * @result true if operators are equal.
         * @result false otherwise.
         */
        inline const Iterator& operator=(const Iterator& other);
        inline Iterator(ReferenceIterator* reference);

    private:
        // Disable postfix operator for performance reasons.
        FEATSTD_LINT_NEXT_EXPRESSION(970, "modifier or type 'int' outside of a typedef not used here but to disable the operator for 'int'.")
        inline Iterator& operator++(int);


        ReferenceIterator* m_refIterator;
    };

    /**
     * Constructor.
     */
    explicit PreprocessingContext(const TextRenderContext* reference = 0);

    /**
     * Retrieve an iterator for the glyph data.
     * @return The glyph iterator.
     */
    Iterator GetIterator() const;

    /**
     * Set the order in which the glyph are processed.
     * @param glyphOrder The glyph order.
     */
    void SetGlyphOrder(GlyphOrder glyphOrder);

    /**
     * Retrieve the glyph order set by SetGlyphOrder, or the default RenderOrder.
     * @return The glyph order.
     */
    virtual GlyphOrder GetGlyphOrder() const;

protected:

    /**
     * @brief Abstract class implemented by children of Preprocessed text,
     * to allow iteration over GlyphData.
     */
    class ReferenceIterator {
    public:
        /**
         * Crete a clone of this iterator.
         * @return A new iterator indicating to the same glyph data.
         */
        virtual ReferenceIterator* Clone() const = 0;

        /**
         * Destroy this iterator.
         */
        virtual void Destroy() = 0;

        /**
         * Check iterator validity.
         * @return Whether the iterator is valid and can be dereferenced or not.
         */
        virtual bool IsValid() const = 0;

        /**
         * Retrieve data corresponding to this iterator.
         * @return Referenced object.
         */
        virtual const GlyphData* GetData() const = 0;

        /**
         * Advance iterator to next item.
         * @return Iterator in the state found after being incremented.
         */
        virtual void Advance() = 0;
    };

private:
    // Overrides function from TextMeasureContext.
    virtual void Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph) override = 0;

    /**
     * Retrieve a iterator for the glyph data stored within this object.
     */
    virtual ReferenceIterator* GetReferenceIterator() const = 0;

    GlyphOrder m_glyphOrder;
};

 /** @} */ // end of CanderaTextEngine

 inline PreprocessingContext::Iterator::Iterator() :
    m_refIterator(0)
{}

inline PreprocessingContext::Iterator::Iterator(const Iterator& other) :
    m_refIterator((other.m_refIterator == 0) ? 0 : other.m_refIterator->Clone())
{}

inline PreprocessingContext::Iterator::~Iterator() 
{ 
    if (m_refIterator != 0) {
        m_refIterator->Destroy(); 
        m_refIterator = 0;
    }
}

inline bool PreprocessingContext::Iterator::IsValid() const
{
    return (m_refIterator != 0) && m_refIterator->IsValid();
}

inline const PreprocessingContext::GlyphData& PreprocessingContext::Iterator::operator*() const
{
    return *m_refIterator->GetData();
}

inline const PreprocessingContext::GlyphData* PreprocessingContext::Iterator::operator->() const
{
    return m_refIterator->GetData();
}

inline PreprocessingContext::Iterator& PreprocessingContext::Iterator::operator++()
{
    m_refIterator->Advance();
    return *this;
}

inline bool PreprocessingContext::Iterator::operator==(const Iterator& other) const
{
    return other.m_refIterator == m_refIterator;
}

inline bool PreprocessingContext::Iterator::operator!=(const Iterator& other) const
{
    return other.m_refIterator != m_refIterator;
}

inline const PreprocessingContext::Iterator& PreprocessingContext::Iterator::operator=(const Iterator& other)
{
    if (other.m_refIterator != m_refIterator) {
        if (m_refIterator != 0) {
            m_refIterator->Destroy();
        }
        m_refIterator = (other.m_refIterator == 0) ? 0 : other.m_refIterator->Clone();
    }
    return *this;
}

inline PreprocessingContext::Iterator::Iterator(ReferenceIterator* reference) :
    m_refIterator(reference)
{}

}// namespace TextRendering
}// namespace Candera

#endif// Candera_PreprocessingContext_h
