//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_MaximalPreprocessingContext_h)
#define Candera_MaximalPreprocessingContext_h

#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/PreprocessingContext.h>
#include <Candera/TextEngine/TextRect.h>

#include <Candera/System/Container/ForwardIterator.h>
#include <FeatStd/Container/Vector.h>

namespace Candera { namespace TextRendering { 
    class Style;

/** @addtogroup CanderaTextEngine
 *  @{
 */

/**
 *  @brief PreprocessingContext with minimum collected data.
 */
class MaximalPreprocessingContext : public PreprocessingContext
{
public:
    /**
     * @brief Class used as interface for accessing preprocessed data.
     */
    class MaximalGlyphData : public PreprocessingContext::GlyphData
    {
        friend class MaximalPreprocessingContext;
    public:

        // Override functions from PreprocessingContext::GlyphData
        virtual FontIdentifier GetFontIdentifier() const;
        virtual GlyphIndex GetGlyphIndex() const;
        virtual TextPosition GetCharacterPosition() const;
        virtual PixelPosition2D GetPosition() const;

        /**
         * Retrieve the advancement of the current glyph.
         * Advancement is relative to cursor. Not that this is not the
         * position given by GetPostion. See GetRect for details.
         * @return A structure containing the advancement.
         */
        PixelSize2D GetAdvance() const;

        /**
         * Retrieve the rectangle of the current glyph.
         * Rectangle is relative to cursor. Adding cursor and rectagle
         * position give the bitmap position, as defined by GetPosition.
         * @return A structure containing the rectangle.
         */
        TextRect GetRect() const;

        /**
         * Retrieve the current text direction of the current glyph.
         * @return The text direction.
         */
        GlyphBitmap::Direction GetDirection() const;

    private:
        GlyphBitmap::Direction m_direction;
        FontIdentifier m_fontIdentifier;
        GlyphIndex m_glyphIndex;
        TextPosition m_characterPosition;
        PixelPosition2D m_position;
        PixelSize2D m_advance;
        TextRect m_rect;
    };
private:
    typedef FeatStd::Internal::Vector<MaximalGlyphData> DataContainer;
    typedef DataContainer::ConstIterator DataContainerIterator;

public:
    /**
     * @brief Type used for iterating over Maximal the glyph information.
     */
    typedef ForwardIterator<DataContainerIterator> MaximalGlyphDataIterator;

    /**
     * Constructor.
     */
    explicit MaximalPreprocessingContext(const TextRenderContext* reference = 0);
    
    /**
     * Retrieve an interator usable with Maximal glyph data.
     * @return An iterator.
     */
    MaximalGlyphDataIterator GetMaximalGlyphDataIterator() const;

    /**
     * Reserve a certain amount of memory for this object, to avoid reallocations during rendering.
     * @param size The amount of glyphs to reserve.
     */
    void Reserve(SizeType size);

    /**
     * Reset the size of the preprocessed data to 0.
     */
    void Clear();

private:
    // overrides functions from PreprocessingContext.
    virtual void Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph) final;
    virtual ReferenceIterator* GetReferenceIterator() const final;

    DataContainer m_container;
};

 /** @} */ // end of CanderaTextEngine

}// namespace TextRendering
}// namespace Candera

#endif// Candera_MaximalPreprocessingContext_h
