//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/TextEngine/Internal/TextProcessProperties.h>

#include <FeatStd/Util/StaticObject.h>

#include <Candera/System/GlobalizationBase/Culture.h>
#include <Candera/System/GlobalizationBase/CultureManager.h>

#include <Candera/TextEngine/Internal/StyleTools.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/GlyphCacheAccess.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/Thread.h>
#endif

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);
    namespace TextRendering {
        namespace Internal {

            class TextProcessPropertiesCultureChangeListener : public Candera::Globalization::CultureChangeListener {
            public:

                TextProcessPropertiesCultureChangeListener(TextProcessProperties& properties) :m_properties(properties)
                {
                    Candera::Globalization::CultureManager::GetInstance().AddCultureChangeListener(this);
                }

                virtual ~TextProcessPropertiesCultureChangeListener()
                {
                    bool result = Candera::Globalization::CultureManager::GetInstance().RemoveCultureChangeListener(this);
                    FEATSTD_UNUSED(result);
                    FEATSTD_DEBUG_ASSERT(result);
                    // check whether culture change listener were added more than a single time
                    FEATSTD_DEBUG_ASSERT(!Candera::Globalization::CultureManager::GetInstance().RemoveCultureChangeListener(this));
                }

            protected:
                FEATSTD_MAKE_CLASS_UNCOPYABLE(TextProcessPropertiesCultureChangeListener);
                TextProcessProperties& m_properties;

                // post hook has to be used as text changes in OnCultureChanged 
                virtual void OnPostCultureChanged(const Candera::Globalization::Culture& /*culture*/) override
                {
                    m_properties.InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }

                virtual void OnCultureChanged(const Candera::Globalization::Culture&) override { /* Nothing to do here - everything has to happen in post hook */ }


#ifdef FEATSTD_THREADSAFETY_ENABLED
                virtual void Obtain()
                {
                    (void)FeatStd::Internal::AtomicOp::Inc(m_obtainCounter);
                }

                virtual void Release()
                {
                    (void)FeatStd::Internal::AtomicOp::Dec(m_obtainCounter);
                }

                virtual void WaitForRelease()
                {
                    while (!FeatStd::Internal::AtomicOp::TestAndSet(m_obtainCounter, 0, 0)) {
                        FeatStd::Internal::Thread::Sleep(5);
                    }
                }

                FeatStd::Internal::AtomicOp::Atomic m_obtainCounter;
#endif
            };


            static FeatStd::String const& GetEllipsis()
            {
                FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::String, s_ellipsis, "...");
                return s_ellipsis;
            }

            static FeatStd::String const& s_forceGetEllipsis = GetEllipsis();

            SharedTextProcessProperties::SharedPointer SharedTextProcessProperties::Create()
            {
                return SharedTextProcessProperties::SharedPointer(TEXTENGINE_TRANSIENT_NEW(SharedTextProcessProperties)());
            }

            TextProcessProperties::TextProcessProperties() :
                m_text(),
                m_style(0),
                m_culture(), 
                m_textProcessUser(0),
                // Dimensions
                m_outerSizeRestriction(-1, -1),
                m_innerSizeRestriction(-1, -1),

                // Truncation
                m_truncationText(),
                m_truncationMode(TruncationMode::None),

                // Layout
                m_textAlignment(HorizontalTextAlignment::Left),
                m_lineSpacing(0),
                m_glyphSpacing(0),
                m_styleVersion(0),
                m_pixelwiseLineSpacing(true),
                m_wordWrapEnabled(false),
                m_multiLineEnabled(true),
                m_isRightToLeftLayoutDirection(false),
                m_clipEnabled(false)
            {
                m_cultureChangeListener = TEXTENGINE_TRANSIENT_NEW(TextProcessPropertiesCultureChangeListener)(*this);
                if (m_cultureChangeListener == 0) {
                    FEATSTD_LOG_ERROR("Culture change listener could not be added to a TextNode2D");
                }
            }


            TextProcessProperties::TextProcessProperties(TextProcessProperties const& properties):
                m_text(properties.m_text),
                m_style(properties.m_style),
                m_culture(properties.m_culture),
                m_textProcessUser(properties.m_textProcessUser),
                // Dimensions
                m_outerSizeRestriction(properties.m_outerSizeRestriction),
                m_innerSizeRestriction(properties.m_innerSizeRestriction),

                // Truncation
                m_truncationText(properties.m_truncationText),
                m_truncationMode(properties.m_truncationMode),

                // Layout
                m_textAlignment(properties.m_textAlignment),
                m_lineSpacing(properties.m_lineSpacing),
                m_glyphSpacing(properties.m_glyphSpacing),
                m_styleVersion(properties.m_styleVersion),
                m_pixelwiseLineSpacing(properties.m_pixelwiseLineSpacing),
                m_wordWrapEnabled(properties.m_wordWrapEnabled),
                m_multiLineEnabled(properties.m_multiLineEnabled),
                m_isRightToLeftLayoutDirection(properties.m_isRightToLeftLayoutDirection),
                m_clipEnabled(properties.m_clipEnabled)
            {
                m_cultureChangeListener = TEXTENGINE_TRANSIENT_NEW(TextProcessPropertiesCultureChangeListener)(*this);
                if (m_cultureChangeListener == 0) {
                    FEATSTD_LOG_ERROR("Culture change listener could not be added to a TextNode2D");
                }
            }

            TextProcessProperties::~TextProcessProperties()
            {
                if (m_cultureChangeListener != 0) {
                    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::TextRendering::Internal::TextProcessProperties::m_cultureChangeListener, TextProcessProperties is freeing and zeroing in the line below)
                        FEATSTD_SAFE_DELETE(m_cultureChangeListener);
                }
                m_textProcessUser = 0;
            }

            void TextProcessProperties::CheckPassivePropertiesInvalidation()
            {
                if (!m_style.PointsToNull()) {
                    UInt16 styleVersion = Internal::StyleTools::GetStyleVersion(*m_style);
                    if (styleVersion != m_styleVersion) {
                        m_styleVersion = styleVersion;
                        InvalidateProperties(TextProcessInvalidation::InvalidAll);
                    }
                }
            }

            FeatStd::String TextProcessProperties::GetText() const
            {
                return m_text;
            }

            void TextProcessProperties::SetText(FeatStd::String const& val)
            {
                if (m_text != val) {
                    m_text = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            Candera::TextRendering::SharedStyle::SharedPointer TextProcessProperties::GetStyle() const
            {
                return m_style;
            }

            void TextProcessProperties::SetStyle(Candera::TextRendering::SharedStyle::SharedPointer val)
            {
                if ((m_style != val) || 
                    ((!val.PointsToNull()) && 
                    (m_styleVersion != StyleTools::GetStyleVersion(*m_style)))) {
                    m_style = val;
                    if (!m_style.PointsToNull()) {
                        m_styleVersion = StyleTools::GetStyleVersion(*m_style);
                    }
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            FeatStd::String TextProcessProperties::GetSelectedTruncationText() const
            {
                FeatStd::String truncationText;
                switch (m_truncationMode) {
                case Candera::TextRendering::Internal::TruncationMode::None:
                    break;
                case Candera::TextRendering::Internal::TruncationMode::Ellipsis:
                    truncationText = GetEllipsis();
                    break;
                case Candera::TextRendering::Internal::TruncationMode::CustomText:
                    truncationText = m_truncationText;
                    break;
                default:
                    // This should never be reached - as all cases have to be covered. If this is not the case,
                    // the new case has to be added.
                    FEATSTD_DEBUG_FAIL();
                    break;
                }
                return truncationText;
            }
            FeatStd::String TextProcessProperties::GetTruncationText() const
            {
                return m_truncationText;
            }

            void TextProcessProperties::SetTruncationText(FeatStd::String const& val)
            {
                if (m_truncationText != val) {
                    m_truncationText = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            TruncationMode::Enum TextProcessProperties::GetTruncationMode() const
            {
                return m_truncationMode;
            }

            void TextProcessProperties::SetTruncationMode(TruncationMode::Enum val)
            {
                if (m_truncationMode != val) {
                    m_truncationMode = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            bool TextProcessProperties::IsClipEnabled() const
            {
                return m_clipEnabled;
            }

            void TextProcessProperties::SetClipEnabled(bool val)
            {
                if (m_clipEnabled != val) {
                    m_clipEnabled = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            bool TextProcessProperties::IsPixelwiseLineSpacing() const
            {
                return m_pixelwiseLineSpacing;
            }

            void TextProcessProperties::SetPixelwiseLineSpacing(bool val)
            {
                if (m_pixelwiseLineSpacing != val) {
                    m_pixelwiseLineSpacing = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            bool TextProcessProperties::IsWordWrapEnabled() const
            {
                return m_wordWrapEnabled;
            }

            void TextProcessProperties::SetWordWrapEnabled(bool val)
            {
                if (m_wordWrapEnabled != val) {
                    m_wordWrapEnabled = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            bool TextProcessProperties::IsMultiLineEnabled() const
            {
                return m_multiLineEnabled;
            }

            void TextProcessProperties::SetMultiLineEnabled(bool val)
            {
                if (m_multiLineEnabled != val) {
                    m_multiLineEnabled = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            Candera::TextRendering::PixelSize TextProcessProperties::GetLineSpacing() const
            {
                return m_lineSpacing;
            }

            void TextProcessProperties::SetLineSpacing(Candera::TextRendering::PixelSize val)
            {
                if (m_lineSpacing != val) {
                    m_lineSpacing = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            HorizontalTextAlignment::Enum TextProcessProperties::GetHorizontalTextAlignment() const
            {
                return m_textAlignment;
            }

            void TextProcessProperties::SetHorizontalTextAlignment(HorizontalTextAlignment::Enum val)
            {
                if (m_textAlignment != val) {
                    m_textAlignment = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            Candera::TextRendering::TextSize TextProcessProperties::GetOuterSizeRestriction() const
            {
                return m_outerSizeRestriction;
            }

            void TextProcessProperties::SetOuterSizeRestriction(Candera::TextRendering::TextSize val)
            {
                if (m_outerSizeRestriction != val) {
                    m_outerSizeRestriction = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            Candera::TextRendering::TextSize TextProcessProperties::GetInnerSizeRestriction() const
            {
                return m_innerSizeRestriction;
            }

            void TextProcessProperties::SetInnerSizeRestriction(Candera::TextRendering::TextSize val)
            {
                if (m_innerSizeRestriction != val) {
                    m_innerSizeRestriction = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }

            
            Candera::Globalization::Culture::SharedPointer TextProcessProperties::GetCulture() const
            {
                if (m_culture.PointsToNull()) {
                    return Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();
                }
                return m_culture;
            }


            void TextProcessProperties::SetCulture(Candera::Globalization::Culture::SharedPointer val)
            {
                if (m_culture != val) {
                    m_culture = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }
            
            Candera::TextRendering::PixelSize TextProcessProperties::GetGlyphSpacing() const
            {
                return m_glyphSpacing;
            }

            void TextProcessProperties::InvalidateProperties(TextProcessInvalidation::Enum invalidationType)
            {
                if (m_textProcessUser != 0) {
                    m_textProcessUser->InvalidateProperties(invalidationType);
                }
            }

            void TextProcessProperties::SetGlyphSpacing(Candera::TextRendering::PixelSize val)
            {
                if (m_glyphSpacing != val) {
                    m_glyphSpacing = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }


            bool TextProcessProperties::IsRightToLeftLayoutDirection() const
            {
                return m_isRightToLeftLayoutDirection;
            }

            void TextProcessProperties::SetRightToLeftLayoutDirection(bool val)
            {
                if (m_isRightToLeftLayoutDirection != val) {
                    m_isRightToLeftLayoutDirection = val;
                    InvalidateProperties(TextProcessInvalidation::InvalidAll);
                }
            }


            Candera::TextRendering::Internal::ITextProcessUser* TextProcessProperties::GetTextProcessUser() const
            {
                return m_textProcessUser;
            }

            void TextProcessProperties::SetTextProcessUser(Candera::TextRendering::Internal::ITextProcessUser* val)
            {
                m_textProcessUser = val;
            }


            Candera::TextRendering::Internal::TextProcessProperties::TextCache * TextProcessProperties::GetTextCache() const
            {
                if (m_textProcessUser != 0) {
                    return m_textProcessUser->GetGlyphCache();
                }
                return 0;
            }

        }
    }
}
