//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/TextEngine/Internal/TextProcessHelper.h>

#include <Candera/System/Diagnostics/Log.h>

#include <Candera/TextEngine/LayoutOptions.h>
#include <Candera/TextEngine/Internal/TextProcessProperties.h>
#include <Candera/TextEngine/Internal/TextProcessResult.h>
#include <Candera/TextEngine/Async/TextRenderArgs.h>
#include <Candera/TextEngine/TextRenderContexts/TextToGlyphIteratorContext.h>
#include <Candera/TextEngine/TextRenderer.h>
#include <Candera/TextEngine/ShapingOptions.h>
#include <Candera/TextEngine/TextRenderContexts/NestedTextLayoutStrategy.h>
#include <Candera/TextEngine/TextRenderContexts/LineBreakLayoutStrategy.h>
namespace Candera {
    namespace TextRendering {
        namespace Internal {
            FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);

            inline static Int16 GetSize(Int16 innerSize, Int16 outerSize, bool wordWrapEnabled, bool truncationEnabled)
            {
                if (outerSize < 0) {
                    return innerSize;
                }
                if ((wordWrapEnabled) || (truncationEnabled)) {
                    return ((innerSize >= 0) && (innerSize < outerSize)) ? (innerSize) : (outerSize);
                }
                return innerSize;
            }

            static LayoutOptions GetLayoutOptions(TextProcessProperties const& properties)
            {
                TextSize size;
                bool isOrientationCorrectionUsed = (TruncationMode::None != properties.GetTruncationMode());
                size.SetWidth(GetSize(properties.GetInnerSizeRestriction().GetWidth(), properties.GetOuterSizeRestriction().GetWidth(), properties.IsWordWrapEnabled(), isOrientationCorrectionUsed));
                size.SetHeight(GetSize(properties.GetInnerSizeRestriction().GetHeight(), properties.GetOuterSizeRestriction().GetHeight(), properties.IsWordWrapEnabled(), isOrientationCorrectionUsed));

                HorizontalAlignment hAlignment;
                if (properties.GetHorizontalTextAlignment() == HorizontalTextAlignment::Left) {
                    hAlignment = HLeft;
                }
                else if (properties.GetHorizontalTextAlignment() == HorizontalTextAlignment::Right) {
                    hAlignment = HRight;
                }
                else if (properties.GetHorizontalTextAlignment() == HorizontalTextAlignment::Justified) {
                    hAlignment = HStretch;
                }
                else if (properties.GetHorizontalTextAlignment() == HorizontalTextAlignment::Center) {
                    hAlignment = HCenter;
                }
                else if (properties.GetHorizontalTextAlignment() == HorizontalTextAlignment::Auto) {
                    hAlignment = HLeft;
                }
                else {
                    hAlignment = HLeft;
                }

                return LayoutOptions(
                    hAlignment,
                    VTop,
                    TextCoordinate(),
                    size,
                    properties.IsMultiLineEnabled(),
                    properties.IsWordWrapEnabled(),
                    properties.GetLineSpacing(), properties.GetGlyphSpacing(), properties.IsRightToLeftLayoutDirection(), isOrientationCorrectionUsed);
            }

            static ShapingOptions GetShaperOptions(TextProcessProperties const& properties)
            {
                return ShapingOptions(properties.GetStyle(), properties.GetCulture());
            }

            static TextProperties GetTextProperties(TextProcessResult const& properties)
            {
                return TextProperties(properties.GetResultContainer()->GetIterator());
            }

            static TextProperties GetTextProperties(TextProcessProperties const& properties)
            {
                return TextProperties(properties.GetText().GetCString());
            }

            static TextRenderArgs::SharedPointer ConvertPropertiesToArgs(TextProcessProperties const& properties, LayoutOptions const& layoutOptions)
            {
                SharedStyle::SharedPointer style = properties.GetStyle();
                if (style.PointsToNull()) {
                    FEATSTD_LOG_WARN("Tried to process text without style set!");
                    return TextRenderArgs::SharedPointer();
                }

                TextRenderArgs::SharedPointer args = TextRenderArgs::Create();
                if (args.PointsToNull()) {
                    FEATSTD_LOG_ERROR("Allocation of TextRenderArgs failed!");
                    return args;
                }
                TruncationToGlyphIteratorContext::SharedPointer truncationContext;
                if (properties.GetTruncationMode() != TruncationMode::None) {
                    truncationContext = TruncationToGlyphIteratorContext::Create();
                    if (!truncationContext.PointsToNull()) {
                        TextProperties truncationText(properties.GetSelectedTruncationText().GetCString());
                        truncationContext->Reinit(style, properties.GetTextCache());
                        if (!TextRenderer().Render(*truncationContext, LayoutingOptions(), ShapingOptions(style), truncationText)) {
                            FEATSTD_LOG_ERROR("Render truncation text failed!");
                        }
                    }
                }
                TextProperties textProperties = GetTextProperties(properties);
                static_cast<void>(args->MeasureInit(style, layoutOptions.GetSize(), textProperties, properties.GetTextCache(), layoutOptions, truncationContext));
                return args;
            }

            static TextRect MoveRectangle(TextRect const& rectToMove, TextCoordinate const& moveToPosition)
            {

                PixelPosition moveInX = rectToMove.GetLeft() - moveToPosition.GetX();
                PixelPosition moveInY = rectToMove.GetTop() - moveToPosition.GetY();
                return TextRect(rectToMove.GetLeft() - moveInX,
                                rectToMove.GetTop() - moveInY,
                                rectToMove.GetRight() - moveInX,
                                rectToMove.GetBottom() - moveInY);
            }


            //----------------------------------------------------------------------------------------------
            bool TextProcessHelper::ProcessText(TextProcessProperties& properties, TextProcessResult& result)
            {
                TextRenderArgs::SharedPointer args = ConvertPropertiesToArgs(properties, GetLayoutOptions(properties));
                if (args.PointsToNull()) {
                    return false;
                }
                TextToGlyphIteratorContext context(args);
                NestedTextLayoutStrategy baseStrategy;
                LineBreakLayoutStrategy lineBreakStrategy(context);
                baseStrategy.AddStrategy(&lineBreakStrategy);
                args->GetLayoutingOptionsRef().SetLayoutStrategy(&baseStrategy);
                args->GetLayoutingOptionsRef().SetCultureHandlingRequired(false);
                bool renderResult = TextRenderer().Render(context, args->GetLayoutingOptionsRef(), GetShaperOptions(properties), GetTextProperties(properties));
                GlyphDataContainer::SharedPointer container = args->GetGlyphContainer();
                result.SetResultContainer(container);

                // Correction of offsets based on outer definition: This is the case when inner dimension is not set
                // but the outer one. The outer one can be 'oversized' and therefore the text could be positioned wrongly.
                TextCoordinate startPosition(context.GetLayoutRectangle().GetLeft(), context.GetLayoutRectangle().GetTop());
                TextCoordinate offsetPosition(context.GetTextRectangle().GetLeft() - context.GetLayoutRectangle().GetLeft(),
                                              context.GetTextRectangle().GetTop() - context.GetLayoutRectangle().GetTop());
                bool shiftContainer = false;
                if (properties.GetInnerSizeRestriction().GetWidth() <= 0) {//width correction
                    startPosition.SetX(0);
                    shiftContainer = true;
                }
                else {
                    offsetPosition.TranslateX(startPosition.GetX());
                }
                if (properties.GetInnerSizeRestriction().GetHeight() <= 0) {//height correction
                    startPosition.SetY(0);
                    shiftContainer = true;
                }
                else {
                    offsetPosition.TranslateY(startPosition.GetY());
                }
                if (shiftContainer) {
                    result.SetResultLayoutRectangle(MoveRectangle(context.GetLayoutRectangle(), startPosition));
                    result.SetResultTextRectangle(MoveRectangle(context.GetTextRectangle(), offsetPosition));
                    if (!result.GetResultContainer().PointsToNull()) {
                        result.GetResultContainer()->MoveInDirection(startPosition.GetX() - context.GetLayoutRectangle().GetLeft(), startPosition.GetY() - context.GetLayoutRectangle().GetTop());
                    }
                }
                else {
                    result.SetResultLayoutRectangle(context.GetLayoutRectangle());
                    result.SetResultTextRectangle(context.GetTextRectangle());
                }
                SharedStyle::SharedPointer style = properties.GetStyle();
                result.SetStyle(style);
                result.SetResultConsumed(false);
                return renderResult;
            }

            bool TextProcessHelper::RenderText(TextProcessResult const& properties, TextRenderContext& renderContext)
            {
                if (properties.GetStyle().PointsToNull()) {
                    FEATSTD_LOG_WARN("Try blitting text without style set!");
                    return false;
                }
                if (properties.GetResultContainer().PointsToNull() ) {
                    FEATSTD_LOG_WARN("Try blitting text whilst no glyph iterator is set!");
                    return false;
                }
                return TextRenderer().Render(renderContext, LayoutOptions(properties.GetOffset()), ShapingOptions(properties.GetStyle()), GetTextProperties(properties));
            }

        }
    }
}
