//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StyleTools.h"
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/Internal/GlyphRenderer.h>

namespace Candera {
    namespace TextRendering {
        namespace Internal {

/**
 * @param style Style that contains the font.
 * @param glyphRenderer Glyph Renderer to use when testing the font
 * @param[OUT] foundIdentifier The font identifier
 * @param useDefaultCodepoint look for the default codepoint in the style
 * @param codePoint The codepoint looked up within the style.
 * @return if a font was found or not for the looked up codepoint
 */
static bool GetFontIdentifierByCodePointImpl(
    const Style &style,
    const GlyphRenderer &glyphRenderer, 
    FontIdentifier &foundIdentifier,  
    bool useDefaultCodepoint, 
    Utf32 codePoint = static_cast<Utf32>(0))
{
    if (useDefaultCodepoint)
    {
       codePoint = style.GetDefaultCodepoint(); 
    }
    const Int compositePriority = style.GetCompositePriority(codePoint, style.GetMaximumCompositePriority());
    FontIdentifier identifier = { compositePriority };
    while (identifier.m_value >= 0) {
        const Font& font = style.GetFontByCompositePriority(identifier.m_value);
        if (font.IsValid()) {
            if (glyphRenderer.GetGlyphIndex(font, codePoint, true) != 0) {
                foundIdentifier = identifier;
                return true;
            }
        }
        identifier.m_value = style.GetCompositePriority(codePoint, identifier.m_value);
    }

    const Style* baseStyle = &style; 
    identifier.m_value = -1;
    while (baseStyle != 0) {
        const Font& font = baseStyle->GetDefaultFont();
        if (font.IsValid()) {
            if (glyphRenderer.GetGlyphIndex(
                    font, 
                    useDefaultCodepoint ? baseStyle->GetDefaultCodepoint() : codePoint,
                    true
                    ) != 0) {
                foundIdentifier = identifier;
                return true;
            }
        }
        baseStyle = baseStyle->GetBaseStyle().GetPointerToSharedInstance();
        --identifier.m_value;
    }

    return false;
}

FontIdentifier StyleTools::GetFontIdentifierByCodePoint(const Style& style, Utf32 codePoint)
{
    GlyphRenderInfo glyphRendererInfo;
    glyphRendererInfo.renderGlyph = false;
    glyphRendererInfo.useCache = true;
    glyphRendererInfo.defaultGlyph = style.GetDefaultCodepoint();
    glyphRendererInfo.access = 0;
    GlyphRenderer glyphRenderer(glyphRendererInfo);

    FontIdentifier identifier;
    if (GetFontIdentifierByCodePointImpl(style, glyphRenderer, identifier, false, codePoint))
    {
        return identifier;
    }

    // Search for default codepoint
    (void)GetFontIdentifierByCodePointImpl(style, glyphRenderer, identifier, true);
    return identifier;
}


const Font& StyleTools::GetFontByIdentifier(const Style& style, FontIdentifier identifier)
{
    
    if (identifier.m_value >= 0) {
        return style.GetFontByCompositePriority(identifier.m_value);
    }
    else {
        const Style* baseStyle = &style;
        while (baseStyle != 0) {
            if (identifier.m_value == -1) {
                return baseStyle->GetDefaultFont();
            }
            baseStyle = baseStyle->GetBaseStyle().GetPointerToSharedInstance();
            ++identifier.m_value;
        }
    }

    static const Font c_font;
    return c_font;
}

        } // namespace Internal
    }// namespace TextRendering
}// namespace Candera
