//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ParserData.h"
#include <Candera/TextEngine/Internal/GlyphRenderer.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/System/GlobalizationBase/Culture.h>

namespace Candera {
    namespace TextRendering {
        namespace Internal {

            bool ParserData::Initialize(const TChar* text, TextLength size, const Globalization::Culture* culture, bool isMultilineTextEnabled, bool layoutIsRightToLeft,
                                        BidiBaseLevel::Enum bidiBaseLevel/* = BidiBaseLevel::Deprecated*/)
            {
                m_isMultilineEnabled = isMultilineTextEnabled;
                m_text = text;
                m_size = size;

                bool overrideParagraphBreak = !isMultilineTextEnabled;
                UInt8 baseLevel = 255; // implicit
                switch (bidiBaseLevel) {
                case Candera::TextRendering::BidiBaseLevel::Implicit:
                    // Keep baseLevel 255
                    break;
                case Candera::TextRendering::BidiBaseLevel::LeftToRight:
                    baseLevel = 0;
                    break;
                case Candera::TextRendering::BidiBaseLevel::RightToLeft:
                    baseLevel = 1;
                    break;
                case Candera::TextRendering::BidiBaseLevel::Culture:
                    baseLevel = GetCultureLevel(culture);
                    break;
                case Candera::TextRendering::BidiBaseLevel::Deprecated:
                    baseLevel = GetDeprecatedLevel(culture, layoutIsRightToLeft);
                    break;
                default:
                    // defaults int base level implicit
                    break;
                }

                return m_bidirectionalParserData.Initialize(text, size, baseLevel, overrideParagraphBreak);
            }


            // Compute baseLevel on culture text direction
            // If culture is not available - implicit base level is used
            Candera::UInt8 ParserData::GetCultureLevel(const Globalization::Culture* culture)
            {
                UInt8 baseLevel = 255; // implicit

                if (culture != 0) {
                    switch (culture->GetTextDirection()) {
                    case Globalization::LeftToRight:
                        baseLevel = 0;
                        break;
                    case Globalization::RightToLeft:
                        baseLevel = 1;
                        break;
                    default:
                        break;
                    }
                }
                return baseLevel;
            }

            Candera::UInt8 ParserData::GetDeprecatedLevel(const Globalization::Culture* culture, bool layoutIsRightToLeft)
            {
                UInt8 baseLevel = 255; // implicit

                if (culture != 0) {
                    // Compute baseLevel based both on culture text direction and layout direction 
                    switch (culture->GetTextDirection()) {
                    case Globalization::LeftToRight:
                        baseLevel = 0;
                        break;
                    case Globalization::RightToLeft:
                        // For RightToLeft cultures, render in LeftToRight order if layout forces it 
                        baseLevel = layoutIsRightToLeft?1:0;
                        break;
                    default:
                        break;
                    }
                }
                return baseLevel;
            }

        }
    }
}
