//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_Parser_h)
#define Candera_Parser_h

#include <Candera/Environment.h>
#include <Candera/TextEngine/Types.h>

#include <Candera/TextEngine/Internal/UnibreakInclude.h>
#include <Candera/TextEngine/Internal/TextPointer.h>
#include <Candera/TextEngine/Internal/ParserData.h>

#ifdef CANDERA_BIDIRECTIONAL_TEXT_ENABLED
#include <Candera/TextEngine/Internal/BidirectionalParser/BidirectionalParser.h>
#else
namespace Candera { namespace TextRendering { namespace Internal {
    class BidirectionalParser {};
}}}
#endif

namespace Candera {
    namespace TextRendering {
        class Style;
        class Font;
        namespace Internal {
            class ParserData;
/** @addtogroup CanderaTextEngine
 *  @{
 */

/**
 * Text parser.
 * 
 *  The text Parser splits a text into homogeneous chunks of text that can be further processed by 
 * a Shaper. The resulted chunk has the following properties:
 * - can specify that a line break is mandatory after the chunk is shaped, but it will never contain
 *  mandatory breaks inside it.
 * - it will be rendered using a unique Font, one of the ones configured in the Style.
 */
class Parser {
public:
    enum LineBreakType{
        UndefinedLineBreak, ///< Linebreak type is undefined.
        MandatoryLineBreak, ///< Linebreak is mandatory.
        AllowLineBreak,     ///< Linebreak is allowed.
        NoLineBreak         ///< No break is possible. 
    };

    // Initialization.
    /**
     * Constructor.
     * @param style Style used for retrieving font matchings.
     * @param parserData Data used by parsers.
     */
    Parser(const Style& style, ParserData& parserData);

    /**
     * Initialize the parser.
     * 
     * Initialize the parser with the given text.
     * @param isWordBreakEnabled Whether each word needs to be broken independently.
     */
    bool Initialize(bool isWordBreakEnabled);

    // Getters.
    /**
     * Get start address of text.
     */
    const TChar* GetText() const { return m_parserData->m_text; }

    /**
     * Get position in text of the current chunk.
     */
    TextPosition GetStart() const { return m_chunkStart.GetPosition(); }

    /**
     * Get current chunk length.
     */
    TextLength GetLength() const { return m_chunkLength; }

    /**
     * Get the direction of the current chunk.
     */
    UInt8 GetDirection() const { return m_direction; }

    /**
     * Get current chunk Font.
     */
    FontIdentifier GetFontIdentifier() const;

    /**
     * Get position in text of the current line.
     */
    TextPosition GetLineStart() const { return m_lineStart.GetPosition(); }

    // Manipulation.
    /**
     * Search for a new position for the line break cursor.
     * The type of character on which the line break should be set is
     * enabled by type.
     * 
     * On each new line this function has to be called before chunks can be
     * iterated.
     */
    void AdvanceLineBreak(LineBreakType type);

    /**
     * Copy line break information from another parser.
     */
    void SetLineBreak(const Parser& src);

    /**
     * Set up the parser for parsing a new line.
     * @param produceCorrectChunkOrdering When false the chunks will not be sorted in display order.
     */
    void StartLineParsing(bool produceCorrectChunkOrdering);

    /**
     * Advance to the next line.
     */
    void AdvanceToNextLine();

    /**
     * Advance to the next chunk.
     */
    void AdvanceToNextChunk();

    // Iteration state.
    /**
     * Return whether parsing is complete.
     */
    bool IsEnd() const { return IsLineBreak() && (m_textRemainingLength <= 0); }

    /**
     * Return whether parsing is at end of line.
     */
    bool IsLineBreak() const { return (m_chunkLength <= 0) && (m_unidirectionalChunkLength <= 0); }

    /**
     * Return whether parsing is at a mandatory line break.
     */
    bool IsMandatoryLineBreak() const { return IsLineBreak() && m_isMandatoryLineBreak; }

    /**
     * Return whether parsing is at a word break.
     */
    bool IsWordBreak() const { return m_isWordBreak; }

    TextLength GetCurrentLineLength() const { return m_lineLength; }
private:
    bool m_isMandatoryLineBreak;
    bool m_isWordBreak;
    bool m_isWordBreakEnabled;
    bool m_produceCorrectChunkOrder;

    LineBreakType m_lineBreakType;

    UInt8 m_direction;

    const Style* m_style;
    ParserData* m_parserData;

    TextLength m_lineLength;
    TextPointer m_lineStart;
    TextLength m_textRemainingLength;

    TextLength m_chunkLength;
    TextPointer m_chunkStart;

    TextLength m_unidirectionalChunkLength;
    TextPointer m_unidirectionalChunkStart;

    Unibreak::lineBreakContext m_lineBreakContext;
    Unibreak::wordBreakContext m_wordBreakContext;

    BidirectionalParser m_bidiParser;
};

 /** @} */ // end of CanderaTextEngine

        }     // namespace Internal
    }    // namespace TextRendering
}    // namespace Candera

#endif    // Candera_Parser_h
