//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MeasureTextRenderContext.h"

#include <Candera/TextEngine/Style.h>

namespace Candera { namespace TextRendering { namespace Internal {

    MeasureTextRenderContext::MeasureTextRenderContext(const MeasuringOptions& options, const Style& style, const TextRenderContext* referenceTextRenderContext) :
        m_ascender(style.GetMetrics().ascender),
        m_descender(style.GetMetrics().descender),
        m_referenceContext(referenceTextRenderContext),
        m_finalAdvance(),
        m_box(TextRect::GetMin()),
        m_finalAdvanceOption(options.GetFinalAdvance()),
        m_transversalSizeOption(options.GetTransversalSize())
    {
    }

    MeasureTextRenderContext::MeasureTextRenderContext(
        MeasuringOptions::FinalAdvance finalAdvanceOption,
        MeasuringOptions::TransversalSize transversalSizeOption,
        const Style& style,
        const TextRenderContext* referenceTextRenderContext) :
        m_ascender(style.GetMetrics().ascender),
        m_descender(style.GetMetrics().descender),
        m_referenceContext(referenceTextRenderContext),
        m_finalAdvance(),
        m_box(TextRect::GetMin()),
        m_finalAdvanceOption(finalAdvanceOption),
        m_transversalSizeOption(transversalSizeOption)
    {
    }

    void MeasureTextRenderContext::Blit(Int16 x, Int16 y, const GlyphBitmap &glyph)
    {
        // compute bounds
        PixelPosition left = x;
        PixelPosition right = (x + static_cast<PixelPosition>(glyph.width)) - 1;

        PixelPosition top;
        PixelPosition bottom;
        
        if (m_transversalSizeOption == MeasuringOptions::ConstantTransversalSize) {
            PixelPosition glyphBaseLine = y + glyph.top;
            top = (glyphBaseLine - m_ascender) + 1;
            bottom = glyphBaseLine - m_descender;
        }
        else {
            top = y;
            bottom = (y + static_cast<PixelPosition>(glyph.height)) - 1;
        }

        if (left < m_box.GetLeft()){
            m_box.SetLeft(left);
        }
        if (right > m_box.GetRight()){
            m_box.SetRight(right);
        }

        if (top < m_box.GetTop()){
            m_box.SetTop(top);
        }
        if (bottom > m_box.GetBottom()){
            m_box.SetBottom(bottom);
        }

        //compute final advance
        m_finalAdvance.x = (((-glyph.left) + glyph.xadvance) - static_cast<PixelPosition>(glyph.width)) + 1;
        m_finalAdvance.y = 0;
    }

    TextRect MeasureTextRenderContext::GetTextRectangle() const
    {
        bool boxEqualsToMinRect = (m_box.GetTop() == TextRect::GetMin().GetTop())
                                  && (m_box.GetLeft() == TextRect::GetMin().GetLeft())
                                  && (m_box.GetBottom() == TextRect::GetMin().GetBottom())
                                  && (m_box.GetRight() == TextRect::GetMin().GetRight());

        if (boxEqualsToMinRect) {
            return TextRect();
        }

        TextRect box (m_box);

        // Only horizontal text is momentarily supported.

        if (m_finalAdvanceOption == MeasuringOptions::IncludeFinalAdvance) {
            box.SetRight(box.GetRight() + ((m_finalAdvance.x > 0) ? m_finalAdvance.x : 0));
            box.SetBottom(box.GetBottom() + ((m_finalAdvance.y > 0) ? m_finalAdvance.y : 0));
        }

        return box;
    }

        }    //namespace Internal
    }    // namespace TextRendering
}    // namespace Candera
