//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_TextEngine_LookupTools_h)
#define Candera_TextEngine_LookupTools_h

#include <Candera/Environment.h>

namespace Candera {
    namespace TextRendering {
        namespace Internal {
/** @addtogroup CanderaTextEngine
 *  @{
 */

            /**
             * Functions useful for looking up data in tables.
             */
            class LookupTools {
            public:
                
                template <typename Iterator, typename Value>
                static Iterator FindRange(Iterator start, Iterator end, Value point);
                template <typename Iterator, typename Value>
                static Iterator Find1(Iterator start, Iterator end, Value point);
                template <typename Iterator, typename Value>
                static Iterator Find2(Iterator start, Iterator end, Value point);

                template <typename Iterator, typename Item, typename Compare>
                static Iterator Find(Iterator start, Iterator end, Item point);

            };

 /** @} */ // end of CanderaTextEngine

 template <typename Iterator, typename Value>
struct CompareRange {
    static bool Less(Value point, Iterator it) { return point < it->start; }
    static bool More(Value point, Iterator it) { return point > it->end; }
};

template <typename Iterator, typename Value>
struct Compare1 {
    static bool Less(Value point, Iterator it) { return point < it->value; }
    static bool More(Value point, Iterator it) { return point > it->value; }
};

template <typename Iterator, typename Value>
struct Compare2 {
    static bool Less(Value point, Iterator it) { return point->first < it->first && point->second < it->second; }
    static bool More(Value point, Iterator it) { return point->first > it->first || point->second > it->second; }
};

template <typename Iterator, typename Value>
Iterator LookupTools::FindRange(Iterator start, Iterator end, Value point)
{
    return Find<Iterator, Value, CompareRange<Iterator, Value> >(start, end, point);
}

template <typename Iterator, typename Value>
Iterator LookupTools::Find1(Iterator start, Iterator end, Value point)
{
    return Find<Iterator, Value, Compare1<Iterator, Value> >(start, end, point);
}

template <typename Iterator, typename Value>
Iterator LookupTools::Find2(Iterator start, Iterator end, Value point)
{
    return Find<Iterator, Value, Compare2<Iterator, Value> >(start, end, point);
}

template <typename Iterator, typename Item, typename Compare>
Iterator LookupTools::Find(Iterator start, Iterator end, Item point)
{
    Iterator invalid = end;
    while (start < end) {
        Iterator middle = start + (end - start) / 2;
        
        if (Compare::Less(point, middle)) {
            end = middle;
            continue;
        }

        if (Compare::More(point, middle)) {
            start = middle + 1;
            continue;
        }

        return middle;
    }

    return invalid;
}

        }     // namespace Internal
    }    // namespace TextRendering
}    // namespace Candera

#endif    // Candera_TextEngine_LookupTools_h
