//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Layouter.h"

#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/LayoutingOptions.h>
#include <Candera/TextEngine/TextRect.h>
#include <Candera/System/GlobalizationBase/Culture.h>

namespace Candera {
    namespace TextRendering {
        namespace Internal {

            // ----------------------------------------------------------------------------
            static PixelPosition GetHorizontalAlignmentFactor(HorizontalAlignment horizontalAlignment)
            {
                PixelPosition ret = 0;
                switch (horizontalAlignment) {
                case HLeft: ret = 0;
                    break;
                case HCenter: ret = 1;
                    break;
                case HRight: ret = 2;
                    break;
                default: ret = 0;
                    break;
                }
                return ret;
            }

            // ----------------------------------------------------------------------------
            static PixelPosition GetVerticalAlignmentFactor(VerticalAlignment verticalAlignment)
            {
                PixelPosition ret = 0;
                switch (verticalAlignment) {
                case VTop: ret = 0;
                    break;
                case VCenter: ret = 1;
                    break;
                case VBottom: ret = 2;
                    break;
                default: ret = 0;
                    break;
                }
                return ret;
            }

            static bool IsRightToLeftDirection(const Globalization::Culture *culture, bool const isRightToLeftDirection)
            {
                if ((culture != 0) && (culture->GetTextDirection() == Globalization::RightToLeft)) {
                    return true;
                }
                else if (isRightToLeftDirection) {
                    return true;
                }
                else {
                    return false;
                }
            }

            // ----------------------------------------------------------------------------
            static PixelPosition GetLocalizedHorizontalAlignmentFactor(HorizontalAlignment horizontalAlignment,  const Globalization::Culture *culture, bool isCultureHandlingRequired)
            {
                PixelPosition factor = GetHorizontalAlignmentFactor(horizontalAlignment);
                if ((isCultureHandlingRequired) && (IsRightToLeftDirection(culture, false))) {
                   factor = 2 - factor;
                }

                return factor;
            }

            Layouter::Layouter(const Style& style) :
                m_isWordByWord(false),
                m_isWordWrapEnabled(false),
                m_isRightToLeftDirection(false),
                m_isOrientationRealignmentEnabled(false),
                m_lineCount(0),
                m_wordCount(0),
                m_totalWordCount(0),
                m_ascender(style.GetMetrics().ascender - 1),
                m_lineSpacing(style.GetMetrics().lineHeight),
                m_glyphSpacing(0),
                m_wordSpacing(0),
                m_horizontalAlignment(0),
                m_verticalAlignment(0)
            {}

            void Layouter::Initialize(const LayoutingOptions& layoutingOptions, const Globalization::Culture* culture)
            {
                m_isOrientationRealignmentEnabled = layoutingOptions.IsOrientationRealignmentEnabled();
                m_isWordWrapEnabled = layoutingOptions.IsWordWrapEnabled();
                m_size = layoutingOptions.GetSize();
                m_offset = layoutingOptions.GetOffset(); 

                    HorizontalAlignment horizontalAligntment = layoutingOptions.GetHorizontalAlignment();
                m_isWordByWord = (horizontalAligntment == HStretch);
                if (m_isWordByWord == true) {
                    horizontalAligntment = HLeft;
                }
                m_isRightToLeftDirection = IsRightToLeftDirection(culture, layoutingOptions.IsRightToLeftDirection());
                m_horizontalAlignment = GetLocalizedHorizontalAlignmentFactor(horizontalAligntment, culture, layoutingOptions.IsCultureHandlingRequired());
                m_verticalAlignment = GetVerticalAlignmentFactor(
                    layoutingOptions.GetVerticalAlignment());

                m_cursorPosition = m_offset;
                m_cursorPosition.TranslateY(m_ascender);

                if (layoutingOptions.GetLineSpacing() > 0) {
                    m_lineSpacing = static_cast<PixelPosition>(layoutingOptions.GetLineSpacing());
                }

                m_glyphSpacing = static_cast<PixelPosition>(layoutingOptions.GetGlyphSpacing());
            }

            void Layouter::InitializeForMeasuring(const Layouter& layouter)
            {
                m_glyphSpacing = layouter.GetGlyphSpacing();
            }

            void Layouter::SetTextMeasurement(const TextRect& rect)
            {
                // Set new size.
                if (m_size.GetWidth() <= 0) {
                    // Use measured size when the layout rectangle is not specified.
                    m_size.SetWidth(rect.GetWidth());
                }

                // Correct vertical position.
                if ((m_verticalAlignment != 0) && (m_size.GetHeight() > 0)) {
                    m_offset.TranslateY(
                        (m_verticalAlignment *
                        (m_size.GetHeight() -
                        rect.GetHeight())) / 2);
                }

                m_cursorPosition = m_offset;
                m_cursorPosition.TranslateY(m_ascender);
            }

            void Layouter::SetLineMeasurement(const TextRect& rect, Int16 wordBreakCount, bool forcedEnd)
            {
                PixelPosition horizontalAdjustment;

                if ((m_isWordByWord == true) && (!forcedEnd) && (wordBreakCount > 0)) {
                    m_wordSpacing = m_size.GetWidth() - rect.GetWidth();
                    m_totalWordCount = wordBreakCount;
                    horizontalAdjustment = 0;
                }
                else {
                    m_wordSpacing = 0;
                    m_totalWordCount = 0;
                    horizontalAdjustment = m_horizontalAlignment;
                }
                m_cursorPosition.SetX(m_offset.GetX() - rect.GetLeft());
                PixelPosition startingX = m_size.GetWidth() - rect.GetWidth();
                PixelPosition adjustmentX = (horizontalAdjustment * (startingX)) / 2;
                if ((m_isWordWrapEnabled == false) && (m_isOrientationRealignmentEnabled == true)) {
                    if (startingX <= 0) {// text is greater than available space
                        if (m_isRightToLeftDirection == false) {// LTR-Text: align on left side
                            adjustmentX = 0;
                        }
                        else {// right aligned
                            adjustmentX = startingX;
                        }

                    }
                }
                m_cursorPosition.TranslateX(adjustmentX);
            }

            void Layouter::AdvanceToNextLine()
            {
                m_lineCount++;
                m_cursorPosition = m_offset;
                m_cursorPosition.TranslateY(m_ascender + (m_lineSpacing * m_lineCount));

                m_wordCount = 0;
            }

            PixelPosition Layouter::ComputeNextWordSpacing()
            {
                m_wordCount++;
                // -1 is important as the spacing has to be calculated on the spaces and not the words 
                PixelPosition wc = m_totalWordCount - 1;
                if (m_totalWordCount > 1) {
                    PixelPosition currentWordSpacing = m_wordSpacing * m_wordCount;
                    PixelPosition previousWordSpacing = currentWordSpacing - m_wordSpacing;
                    PixelPosition result = (currentWordSpacing / (wc)) -
                        (previousWordSpacing / (wc));
                    return result;
                }
                return 0;
            }

            bool Layouter::IsLineMatching(const TextRect& line) const
            {
                return line.GetWidth() <= m_size.GetWidth();
            }

            bool Layouter::IsTextMeasurementNeeded() const
            {
                // Measuring happens when horizontal layout requires a width,
                // and width is not provided, or when vertical alignment is
                // enabled and a height is available.
                return ((m_horizontalAlignment != 0) && (m_size.GetWidth() <= 0)) ||
                       ((m_verticalAlignment != 0) && (m_size.GetHeight() > 0));
            }

            bool Layouter::IsLineMeasurementNeeded() const
            {
                return m_horizontalAlignment != 0;
            }

        }
    }
}
