//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "HbFreetypeFont.h"
#include <Candera/TextEngine/Freetype/FtGlyphRenderer.h>
#include <Candera/TextEngine/Freetype/FtFont.h>
#include <Candera/TextEngine/TextCoordinate.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

namespace Candera { namespace TextRendering {
    using namespace Internal;
namespace Internal { namespace HarfBuzzShaper {

static hb_bool_t HbFreetypeFontGetGlyph (
    hb_font_t* /*font*/, void *font_data,
    hb_codepoint_t unicode, hb_codepoint_t /*variation_selector*/,
    hb_codepoint_t *glyph,
    void* /*user_data*/)
{
    HbFreetypeFont *data = FeatStd::Internal::PointerToPointer<HbFreetypeFont*>(font_data);
    *glyph = data->GetGlyph(unicode);
    return *glyph != 0;
}

static hb_position_t HbFreetypeFontGetGlyphHAdvance (
    hb_font_t* /*font*/, void *font_data,
    hb_codepoint_t glyph,
    void* /*user_data*/)
{
    HbFreetypeFont *data = FeatStd::Internal::PointerToPointer<HbFreetypeFont*>(font_data);
    return data->GetGlyphHAdvance(glyph);
}

static hb_position_t HbFreetypeFontGetGlyphVAdvance (
    hb_font_t* /*font*/, void* /*font_data*/,
    hb_codepoint_t /*glyph*/,
    void* /*user_data*/)
{
    return 0;
}

static hb_bool_t HbFreetypeFontGetGlyphHOrigin (
    hb_font_t* /*font*/, void * /*font_data*/,
    hb_codepoint_t /*glyph*/,
    hb_position_t *x, hb_position_t *y,
    void* /*user_data*/)
{
    *x = 0;
    *y = 0;
    return false;
}

static hb_bool_t HbFreetypeFontGetGlyphVOrigin (
    hb_font_t* /*font*/, void* /*font_data*/,
    hb_codepoint_t /*glyph*/,
    hb_position_t *x, hb_position_t *y,
    void* /*user_data*/)
{
    *x = 0;
    *y = 0;
    return false;
}

static hb_position_t HbFreetypeFontGetGlyphHKerning (
    hb_font_t* /*font*/, void* font_data,
    hb_codepoint_t first_glyph, hb_codepoint_t second_glyph,
    void* /*user_data*/)
{
    HbFreetypeFont *data = FeatStd::Internal::PointerToPointer<HbFreetypeFont*>(font_data);
    return data->GetGlyphHKerning(first_glyph, second_glyph);
}

static hb_position_t HbFreetypeFontGetGlyphVKerning (
    hb_font_t* /*font*/, void* /*font_data*/,
    hb_codepoint_t /*first_glyph*/, hb_codepoint_t /*second_glyph*/,
    void* /*user_data*/)
{
    return 0;
}

static hb_bool_t HbFreetypeFontGetGlyphExtents (
    hb_font_t* /*font*/, void* /*font_data*/,
    hb_codepoint_t /*glyph*/,
    hb_glyph_extents_t *extents,
    void* /*user_data*/)
{
    MemoryPlatform::Set(extents, 0, sizeof (*extents));
    return false;
}

static hb_bool_t HbFreetypeFontGetGlyphContourPoint (
    hb_font_t* /*font*/, void* font_data,
    hb_codepoint_t glyph, HbInterfaceUnsignedInt point_index,
    hb_position_t *x, hb_position_t *y,
    void* /*user_data*/)
{
    HbFreetypeFont *data = FeatStd::Internal::PointerToPointer<HbFreetypeFont*>(font_data);
    return data->GetGlyphContourPoint(glyph, point_index, *x, *y);
}

static hb_bool_t HbFreetypeFontGetGlyphName (
    hb_font_t* /*font*/, void* /*font_data*/,
    hb_codepoint_t /*glyph*/,
    HbInterfaceChar *name, HbInterfaceUnsignedInt size,
    void* /*user_data*/)
{
    if (size > 0) {
        name[0] = '\0';
    }
    return false;
}

static hb_bool_t HbFreetypeFontGetGlyphFromName (
    hb_font_t* /*font*/, void* /*font_data*/,
    const HbInterfaceChar* /*name*/, HbInterfaceInt /*len*/,
    hb_codepoint_t *glyph,
    void* /*user_data*/)
{
    *glyph = 0;
    return false;
}

HbFreetypeFont::HbFreetypeFont() :
    m_hbFont(0),
    m_font(0),
    m_glyphRenderer(0),
    m_lastGlyphIndex(0),
    m_lastGlyphInfo()
{
}
HbFreetypeFont::~HbFreetypeFont()
{
    Finalize();
}

void HbFreetypeFont::Initialize(const FtGlyphRenderer& glyphRenderer, const FtFont& font)
{
    m_glyphRenderer = &glyphRenderer;

    if ((0 == m_font) || (!(*m_font == font)))
    {
        m_lastGlyphIndex = 0;
        if (0 != m_hbFont)
        {
            hb_font_destroy(m_hbFont);
        }

        m_font = &font;
        FT_Face face = m_glyphRenderer->GetFace(font);

        if (face == 0) {
            return;
        }
        hb_face_t *hbFace = hb_ft_face_create_cached(face);
        if (hbFace == 0) {
            return;
        }
        m_hbFont = hb_font_create(hbFace);
        hb_face_destroy(hbFace);
        if (m_hbFont == 0) {
            return;
        }

        hb_font_funcs_t *funcs = CreateFuncs();

        // Set custom functions to retrieve data from the font.
        hb_font_set_funcs(m_hbFont, funcs, this, 0);
        // The object is  reference counted, so reduce reference count.
        hb_font_funcs_destroy(funcs);

        // HarfBuzz uses the following formula to transform
        // items from lookup tables to positions returned after shaping:
        // output = lookUpValue * scale / unitsPerEm
        // where
        // lookUpValue is given in "units".
        // unitsPerEm is given in "units"/"Em". (i.e. the size of Em in units)
        // output should be in "pixels". (the way TextRenderer interprets the values.)
        // It follows that scale should be:
        // "pixels"/"Em".
        // This value is stored in the size metrics.
        // The direction of the Y axes is top down for the text engine
        // and bottom up in the glyph vectorial space.
        hb_font_set_scale(m_hbFont,
            static_cast<HbInterfaceInt>(face->size->metrics.x_ppem),
            -static_cast<HbInterfaceInt>(face->size->metrics.y_ppem));

        // The font engine abstraction always requests hinted glyph rendering.
        // This will inform HarfBuzz about hinting.
        // TO DO: figure out when this is used in HarfBuzz, and whether the
        // magnitude is correct.
        hb_font_set_ppem(m_hbFont,
            face->size->metrics.x_ppem,
            face->size->metrics.y_ppem);
    }
}

void HbFreetypeFont::Finalize()
{
    if (m_hbFont != 0) {
        hb_font_destroy (m_hbFont);
        m_hbFont = 0;
    }

    m_font = 0;
    m_glyphRenderer = 0;
    m_lastGlyphIndex = 0;
}

hb_font_funcs_t *HbFreetypeFont::CreateFuncs ()
{
    hb_font_funcs_t *funcs = hb_font_funcs_create ();
    if (funcs != 0) {
        hb_font_funcs_set_glyph_func(funcs, HbFreetypeFontGetGlyph, 0, 0);

        hb_font_funcs_set_glyph_h_advance_func(funcs, HbFreetypeFontGetGlyphHAdvance, 0, 0);
        hb_font_funcs_set_glyph_v_advance_func(funcs, HbFreetypeFontGetGlyphVAdvance, 0, 0);

        hb_font_funcs_set_glyph_h_origin_func(funcs, HbFreetypeFontGetGlyphHOrigin, 0, 0);
        hb_font_funcs_set_glyph_v_origin_func(funcs, HbFreetypeFontGetGlyphVOrigin, 0, 0);

        hb_font_funcs_set_glyph_h_kerning_func(funcs, HbFreetypeFontGetGlyphHKerning, 0, 0);
        hb_font_funcs_set_glyph_v_kerning_func(funcs, HbFreetypeFontGetGlyphVKerning, 0, 0);

        hb_font_funcs_set_glyph_extents_func(funcs, HbFreetypeFontGetGlyphExtents, 0, 0);
        hb_font_funcs_set_glyph_contour_point_func(funcs, HbFreetypeFontGetGlyphContourPoint, 0, 0);

        hb_font_funcs_set_glyph_name_func(funcs, HbFreetypeFontGetGlyphName, 0, 0);
        hb_font_funcs_set_glyph_from_name_func(funcs, HbFreetypeFontGetGlyphFromName, 0, 0);
    }

    return funcs;
}

inline hb_codepoint_t HbFreetypeFont::GetGlyph(hb_codepoint_t unicode) const
{
    Utf32 codePoint = Utf32(unicode);
    GlyphIndex glyphIndex = m_glyphRenderer->GetGlyphIndex(*m_font, codePoint);
    return static_cast<hb_codepoint_t>(glyphIndex);
}

inline hb_position_t HbFreetypeFont::GetGlyphHAdvance(hb_codepoint_t glyph)
{
    UpdateLastGlyphInfo(glyph);
    return static_cast<hb_position_t>(m_lastGlyphInfo.xadvance);
}

inline hb_bool_t HbFreetypeFont::GetGlyphHOrigin(hb_codepoint_t glyph, hb_position_t &x, hb_position_t &y)
{
    UpdateLastGlyphInfo(glyph);

    x = -static_cast<hb_position_t>(m_lastGlyphInfo.left);
    y = -static_cast<hb_position_t>(m_lastGlyphInfo.top);

    return true;
}

inline hb_position_t HbFreetypeFont::GetGlyphHKerning(hb_codepoint_t glyphA, hb_codepoint_t glyphB) const
{
    return static_cast<hb_position_t>(m_glyphRenderer->GetGlyphKerning(glyphA, glyphB, *m_font).x);
}

inline hb_bool_t HbFreetypeFont::GetGlyphContourPoint(hb_codepoint_t glyph, HbInterfaceUnsignedInt pointIndex, hb_position_t &x, hb_position_t &y) const
{
    TextCoordinate contour;
    if (!static_cast<const Internal::FtGlyphRenderer*>(m_glyphRenderer)->LoadContourPoint(contour, *m_font, glyph, pointIndex)){
        return false;
    }

    x = static_cast<hb_position_t>(contour.GetX());
    y = static_cast<hb_position_t>(contour.GetY());

    return true;
}


void HbFreetypeFont::UpdateLastGlyphInfo(hb_codepoint_t glyph)
{
    GlyphIndex glyphIndex = static_cast<GlyphIndex>(glyph);
    if (glyphIndex == m_lastGlyphIndex) {
        return;
    }

    m_lastGlyphIndex = glyphIndex;
    static_cast<void>(m_glyphRenderer->GetGlyphBitmap(m_lastGlyphInfo, *m_font, glyphIndex));
}


}}}} // current namespace

