//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BidiUnicode.h"
#include <Candera/TextEngine/Internal/LookupTools.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>

namespace Candera { namespace TextRendering { namespace Internal {
    // Bidi.cpp - version 26

    // Reference implementation for Unicode Bidirectional Algorithm


    /*------------------------------------------------------------------------
    File: Bidi.Cpp

    Description
    -----------

    Sample Implementation of the Unicode Bidirectional Algorithm as it
    was revised by Revision 5 of the Uniode Technical Report # 9
    (1999-8-17)

    Verified for changes to the algorithm up through Unicode 5.2.0 (2009).

    This implementation is organized into several passes, each implemen-
    ting one or more of the rules of the Unicode Bidi Algorithm. The
    resolution of Weak Types and of Neutrals each use a state table
    approach.

    Both a printf based interface and a Windows DlgProc are provided for
    interactive testing.

    A stress harness comparing this implementation (v24) to a Java based
    implementation was used by Doug Felt to verify that the two
    implementations produce identical results for all strings up to six
    bidi classes and stochastic strings up to length 20.

    Version 26 was verified by the author against the Unicode 5.2.0
    file BidiTest.txt, which provides an exhaustive text of strings of
    length 4 or less, but covers some important cases where the language
    in UAX#9 had been clarified.

    To see this code running in an actual Windows program,
    download the free Unibook uitlity from http://unicode.org/unibook
    The bidi demo is executed from the tools menu. It is build from
    this source file.

    Build Notes
    -----------

    To compile the sample implementation please set the #define
    directives above so the correct headers get included. Not all the
    files are needed for all purposes. For the commandline version
    only bidi.h and bidi.cpp are needed.

    The Win32 version is provided as a dialog procedure. To use as a
    standalone program compile with the the lbmain.cpp file. If all you
    need is the ability to run the code "as is", you can instead download
    the unibook utility from http://uincode.org/unibook/ which contains
    the bidi demo compiled from this source file.

    This code uses an extension to C++ that gives variables declared in
    a for() statement function the same scope as the for() statement.
    If your compiler does not support this extension, you may need to
    move the declaration, e.g. int ich = 0; in front of the for statement.

    Implementation Note
    -------------------

    NOTE: The Unicode Bidirectional Algorithm removes all explicit
    formatting codes in rule X9, but states that this can be
    simulated by conformant implementations. This implementation
    attempts to demonstrate such a simulation

    To demonstrate this, the current implementation does the
    following:

    in resolveExplicit()
    - change LRE, LRO, RLE, RLO, PDF to BN
    - assign nested levels to BN

    in resolveWeak and resolveNeutrals
    - assign L and R to BN's where they exist in place of
    sor and eor by changing the last BN in front of a
    level change to a strong type
    - skip over BN's for the purpose of determining actions
    - include BN in the count of deferred runs
    which will resolve some of them to EN, AN and N

    in resolveWhiteSpace
    - set the level of any surviving BN to the base level,
    or the level of the preceding character
    - include LRE,LRO, RLE, RLO, PDF and BN in the count
    whitespace to be reset

    This will result in the same order for non-BN characters as
    if the BN characters had been removed.

    The clean() function can be used to remove boundary marks for
    verification purposes.

    Notation
    --------
    Pointer variables generally start with the letter p
    Counter variables generally start with the letter c
    Index variables generally start with the letter i
    Boolean variables generally start with the letter f

    The enumerated bidirectional types have the same name as in the
    description for the Unicode Bidirectional Algorithm


    Using this code outside a demo context
    --------------------------------------

    The way the functions are broken down in this demo code is based
    on the needs of the demo to show the evolution in internal state
    as the algorithm proceeds. This obscures how the algorithm would
    be used in practice. These are the steps:

    1. Allocate a pair of arrays large enough to hold bidi class
    and calculated levels (one for each input character)

    2. Provide your own function to assign directional types (bidi
    classes) corresponding to each character in the input,
    conflating ON, WS, S to N. Unlike the classify function in this
    demo, the input would be actual Unicode characters.

    3. Process the first paragraph by calling BidiParagraph. That
    function changes B into BN and returns a length including the
    paragraph separator. (The iteration over multiple paragraphs
    is left as excercise for the reader).

    4. Assign directional types again, but now assign specific types
    to whitespace characters.

    5. Instead of reordering the input in place it is often desirable
    to calculate an array of offsets indicating the reordering.
    To that end, allocate such an array here and use it instead
    of the input array in the next step.

    6. Resolve and reorder the lines by calling BidiLines. That
    function 'breaks' lines on LS characters. Provide an optional
    array of flags indicating the location of other line breaks as
    needed.


    Update History
    --------------
    Version 24 is the initial published and verified version of this
    reference implementation. Version 25 and its updates fix various
    minor issues with the scaffolding used for demonstrating the
    algorithm using pseudo-alphabets from the command line or dialog
    box. No changes to the implementation of the actual bidi algrithm
    are made in any of the minor updates to version 25. Version 26
    also makes no change to the actual algorithm but was verified
    against the official BidiTest.txt file for Unicode 5.2.0.

    - updated pseudo-alphabet

    - Last Revised 12-10-99 (25)

    - enable demo mode for release builds - no other changes

    - Last Revised 12-10-00 (25a)

    - fix regression in pseudo alphabet use for Windows UI

    - Last Revised 02-01-01 (25b)

    - fixed a few comments, renamed a variable

    - Last Revised 03-04-01 (25c)

    - make base level settable, enable mirror by default,
    fix dialog size

    - Last Revised 06-02-01 (25e)

    - fixed some comments

    - Last Revised 09-29-01 (25f)

    - fixed classification for LS,RLM,LRM in pseudo alphabet,
    focus issues in UI, regression fix to commandline from 25(e)
    fix DEMO switch

    - Last Revised 11-07-01 (25g)

    - fixed classification for plus/minus in pseudo alphabet
    to track changes made in Unicode 4.0.1

    - Last Revised 12-03-04 (25h)

    - now compiles as dialog-only program for WINDOWS_UI==1
    using new bidimain.cpp

    - Last Revised 12-02-07 (25i)

    - cleaned up whitespace and indenting in the source,
    fixed two comments (table headers)

    - Last Revised 15-03-07 (25j)

    - named enumerations

    - Last Revised 30-05-07 (25k)

    - added usage notes, minor edits to comments, indentation, etc
    throughout. Added the bidiParagraph function. Checked against
    changes in the Unicode Bidi Algorithm for Unicode 5.2.0. No
    changes needed to this implementation to match the values in
    the BidiTest.txt file in the Unicode Character Database.
    Minor fixes to dialog/windows proc, updated preprocessor directives.

    - Last Revised 03-08-09 (26)

    Credits:
    -------
    Written by: Asmus Freytag
    Command line interface by: Rick McGowan
    Verification (v24): Doug Felt

    Disclaimer and legal rights:
    ---------------------------
    Copyright (C) 1999-2009, ASMUS, Inc. All Rights Reserved.
    Distributed under the Terms of Use in http://www.unicode.org/copyright.html.

    THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
    OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
    IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS NOTICE
    BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES,
    OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
    WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
    ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THE SOFTWARE.

    The file bid.rc is included in the software covered by the above.
    ------------------------------------------------------------------------*/


    /** the maximum embedding level as defined by UAX #9 */
    static const UInt8 MAX_LEVEL = 61;

    struct ResolveExplicitStackFrame;

    /** Set a run of cval values at locations all prior to, but not including
        iStart, to the new value nval.
        @param pval pointer to levels array
        @param cval number of levels to set
        @param iStart start index of the run
        @param nval level value to set */
    static inline void SetDeferredRun(UInt8 *pval, TextLength cval, TextPosition iStart, UInt8 nval);

    /** Determines the base level
        Implements rule P2 of the Unicode Bidi Algorithm.
        Note: Ignores explicit embeddings
        @param types bidi type array
        @param nChars number of characters in input string
        @return the computed base level */
    static UInt8 DetermineBaseLevel(const UInt8 *types,  TextLength nChars);

    /** returns of the given level is even or odd number
        @param level embedding level
        @return true if level is odd, false if even */
    static inline bool odd(UInt8 level);

    /** returns the next number greater and even to given number
        @param i the input number
        @return the next greater than input and even number */
    static inline UInt8 GreaterEven(UInt8 i);

    /** returns the next number greater and odd to given number
        @param i the input number
        @return the next greater than input and odd number */
    static inline UInt8 GreaterOdd(UInt8 i);

    /** returns the embedding direction of the given level
        even levels resolve to direction left, odd levels to direction right
        @param level the input level
        @return */
    static inline UInt8 EmbeddingDirection(UInt8 level);

    /** resolves explicit embedding levels and overrides.
        Implements rules X1-X9, of the Unicode Bidirectional Algorithm.
        @param frame processing stack frame buffer. large enough to hold MAX_LEVEL frames
        @param baseLevel the base level
        @param types the bidi type array
        @param levels levels the embedding level array
        @param nChars the number of entries in the types and levels
        @return BidiUnicodeL if level is even, BidiUnicodeR if odd level */
    static void resolveExplicit(ResolveExplicitStackFrame *frame, UInt8 baseLevel,
                                UInt8 *types, UInt8 *levels, TextLength nChars);

    /** returns the deferred type of the given action
        @param action the action value
        @return the deferred type */
    static inline UInt8 GetDeferredType(UInt16 action);

    static inline UInt8 GetResolvedType(UInt16 action);
    /** Resolves the directionality of numeric and other weak character types
        Implements rules X10 and W1-W6 of the Unicode Bidirectional Algorithm.
        @param baselevel the base level
        @param pcls array of bidi types. On input only these directional classes are expected
                    AL, HL, R, L,  ON, BN, NSM, AN, EN, ES, ET, CS,
        @param plevel array of embedding levels
        @param cch number of entries in pcls and plevel */
    static void resolveWeak(UInt8 baselevel, UInt8* pcls,
                            UInt8 *plevel, TextLength cch);

    /** return deferred neutrals embedding direction
        @param action the action value
        @param level the specified embedding level
        @return BidiUnicodeL or BidiUnicodeR depending on action */
    static inline UInt8 GetDeferredNeutrals(UInt16 action, UInt8 level);

    /** get the resolved neutrals
        @param action the action value
        @return the resolved value */
    static inline UInt8 GetResolvedNeutrals(UInt16 action);

    /** Resolves the directionality of neutral character types.
        Implements rules W7, N1 and N2 of the Unicode Bidi Algorithm.
        W8 resolves a number of ENs to L
        @param baselevel the level to start from
        @param pcls the array of bidi type information. On input only these
                    directional classes are expected R, L, N, AN, EN and BN
        @param plevel the array of embedding levels
        @param cch the number of entries in pcls and plevel */
    static void resolveNeutrals(UInt8 baselevel, UInt8 *pcls,
                                const UInt8 *plevel, TextLength cch);

    /** resolves implicit embedding levels.
        Implements rules I1 and I2 of the Unicode Bidirectional Algorithm.
        Accepted subset of direction classes R, L, AN, EN
        @param types the bidi type information array.
        @param levels the embedding levels array. levels may exceed 15 on output
        @param nChars the number of entries in types and levels */
    static void resolveImplicit(const UInt8 *types, UInt8 *levels, TextLength nChars);

    /** Resolves levels for WS and S
        Implements rule L1 of the Unicode bidi Algorithm.
        @param string the input string for bidi algorithm
        @param baselevel the embedding level to start from
        @param types the bidi type information array
        @param levels the embedding levels array
        @param nChars the number of entries in string, types, and levels
        @return the maximum embedding level in levels */
    static void resolveWhitespace(UInt8 baselevel, const UInt8 *types, UInt8 *levels,
                                   TextLength nChars);

    /** WS, LS and S are not explicitly needed except for L1. Therefore this
        table conflates ON, S, WS, and LS to N, all others unchanged */
    static const UInt8 NTypes[] = {
        UInt8(BidiUnicode::ON),        // ON,
        UInt8(BidiUnicode::L),        // L,
        UInt8(BidiUnicode::R),        // R,
        UInt8(BidiUnicode::AN),     // AN,
        UInt8(BidiUnicode::EN),     // EN,
        UInt8(BidiUnicode::AL),     // AL
        UInt8(BidiUnicode::NSM),    // NSM
        UInt8(BidiUnicode::CS),     // CS
        UInt8(BidiUnicode::ES),     // ES
        UInt8(BidiUnicode::ET),     // ET
        UInt8(BidiUnicode::BN),     // BN
        UInt8(BidiUnicode::ON),        // S
        UInt8(BidiUnicode::ON),        // WS
        UInt8(BidiUnicode::BN),        // B Paragraphing is done outside.
        UInt8(BidiUnicode::RLO),    // RLO
        UInt8(BidiUnicode::RLE),    // RLE
        UInt8(BidiUnicode::LRO),    // LRO
        UInt8(BidiUnicode::LRE),    // LRE
        UInt8(BidiUnicode::PDF),    // PDF
        UInt8(BidiUnicode::ON),        // LS
    };

    //  -------------------------------------------------------------------------
    inline void SetDeferredRun(UInt8 *pval, TextLength cval, TextPosition iStart, UInt8 nval)
    {
        MemoryPlatform::Set((pval + iStart) - cval, nval, cval);
    }

    //  -------------------------------------------------------------------------
    Candera::UInt8 DetermineBaseLevel(const UInt8 *types, TextLength nChars)
    {
        UInt8 baselevel = 0;

        bool found = false;
        for (TextPosition i = 0; (!found) && (i < nChars); ++i) {
            if (types[i] == UInt8(BidiUnicode::L)) {
                found = true;
            }
            else if ((types[i] == UInt8(BidiUnicode::R)) || (types[i] == UInt8(BidiUnicode::AL))) {
                baselevel = 1;
                found = true;
            }
            else {
                //do nothing
            }
        }
        return baselevel;
    }

    //  -------------------------------------------------------------------------
    static inline bool odd(UInt8 level)
    {
        return (level & 1U) != 0U;
    }

    //  -------------------------------------------------------------------------
    static inline UInt8 GreaterEven(UInt8 i)
    {
        return odd(i) ? (i + 1) : (i + 2);
    }

    //  -------------------------------------------------------------------------
    static inline UInt8 GreaterOdd(UInt8 i)
    {
        return odd(i) ? (i + 2) : (i + 1);
    }

    //  -------------------------------------------------------------------------
    static inline UInt8 EmbeddingDirection(UInt8 level)
    {
        return UInt8(odd(level) ? BidiUnicode::R : BidiUnicode::L);
    }

    /** stores a stack frame in the processing of resolveExplicit */
    struct ResolveExplicitStackFrame {
        UInt8 currLevel;            ///< current embedding level
        UInt8 dir;                    ///< LTR/RTL
        UInt8 nLastValid;            ///< last valid embedding level
        UInt8 nNest;                ///< current nesting count
    };

    //  -------------------------------------------------------------------------
    static void resolveExplicit(ResolveExplicitStackFrame *frame, UInt8 baseLevel, UInt8 *types, UInt8 *levels, TextLength nChars)
    {
        enum {
            StackNOP = 0,
            StackPush = 1,
            StackPop = 2
        };

        ResolveExplicitStackFrame *currFrame = frame;

        // always called with a valid nesting currLevel
        // nesting levels are != embedding levels
        TextPosition i = 0;

        currFrame->currLevel = baseLevel;
        currFrame->dir = UInt8(BidiUnicode::ON);
        currFrame->nLastValid = 0;
        currFrame->nNest = 0;

        // process the text
        while (i < nChars) {
            UInt8 stackOp = StackNOP;
            UInt8 type = types[i];
            if ((type == UInt8(BidiUnicode::LRO)) || (type == UInt8(BidiUnicode::LRE))) {
                currFrame->nNest++;
                if (GreaterEven(currFrame->currLevel) <= MAX_LEVEL) {
                    levels[i] = GreaterEven(currFrame->currLevel);
                    types[i] = UInt8(BidiUnicode::BN);

                    type = (type == UInt8(BidiUnicode::LRE)) ? UInt8(BidiUnicode::ON) : UInt8(BidiUnicode::L);
                    stackOp = StackPush;
                }
                else {
                    types[i] = UInt8(BidiUnicode::BN);
                    type = types[i];
                }
            }
            else if ((type == UInt8(BidiUnicode::RLO)) || (type == UInt8(BidiUnicode::RLE))) {
                currFrame->nNest++;
                if (GreaterOdd(currFrame->currLevel) <= MAX_LEVEL) {
                    levels[i] = GreaterOdd(currFrame->currLevel);
                    types[i] = UInt8(BidiUnicode::BN);

                    type = (type == UInt8(BidiUnicode::RLE)) ? UInt8(BidiUnicode::ON) : UInt8(BidiUnicode::R);
                    stackOp = StackPush;
                }
                else {
                    types[i] = UInt8(BidiUnicode::BN);
                    type = types[i];
                }
            }
            else if (type == UInt8(BidiUnicode::PDF)) {
                types[i] = UInt8(BidiUnicode::BN);
                type = types[i];
                if (currFrame->nNest != 0) {
                    if (currFrame->nLastValid < currFrame->nNest) {
                        currFrame->nNest--;
                    }
                    else {
                        stackOp = StackPop;
                    }
                }
            }
            else {
                //do nothing
            }

            switch(stackOp) {
                case StackNOP:
                    // Apply the override
                    if (currFrame->dir != UInt8(BidiUnicode::ON)) {
                        type = currFrame->dir;
                    }
                    levels[i] = currFrame->currLevel;
                    if (types[i] != UInt8(BidiUnicode::BN)) {
                        types[i] = type;
                    }
                    break;

                case StackPop:
                    FEATSTD_DEBUG_ASSERT(currFrame != frame);
                    currFrame--;
                    break;

                case StackPush: {
                    ResolveExplicitStackFrame *prevFrame = currFrame++;
                    FEATSTD_DEBUG_ASSERT(currFrame != frame + MAX_LEVEL - 1);
                    currFrame->currLevel = levels[i];
                    currFrame->dir = type;
                    currFrame->nLastValid = prevFrame->nNest;
                    currFrame->nNest = prevFrame->nNest;
                    break;
                }

                default:
                    break;
            }

            ++i;
        }
    }

    // === RESOLVE WEAK TYPES ================================================

    /** states definition for bidi */
    enum BidiState {
        xa,        ///< arabic letter
        xr,        ///< right leter
        xl,        ///< left letter

        ao,        ///< arabic lett. foll by ON
        ro,        ///< right lett. foll by ON
        lo,        ///< left lett. foll by ON

        rt,        ///< ET following R
        lt,        ///< ET following L

        cn,        ///< EN, AN following AL
        ra,        ///< arabic number foll R
        re,        ///< european number foll R
        la,        ///< arabic number foll L
        le,        ///< european number foll L

        ac,        ///< CS following cn
        rc,        ///< CS following ra
        rs,        ///< CS,ES following re
        lc,        ///< CS following la
        ls,        ///< CS,ES following le

        ret,    ///< ET following re
        let     ///< ET following le
    };


    /** weak state transition table */
    static const UInt8 stateWeak[][10] =
    {
        //    N,  L,  R,  AN, EN, AL,NSM, CS, ES, ET,
        /*xa*/    { UInt8(ao), UInt8(xl), UInt8(xr), UInt8(cn), UInt8(cn), UInt8(xa), UInt8(xa), UInt8(ao), UInt8(ao), UInt8(ao) }, /* arabic letter          */
        /*xr*/    { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa), UInt8(xr), UInt8(ro), UInt8(ro), UInt8(rt) }, /* right letter           */
        /*xl*/    { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa), UInt8(xl), UInt8(lo), UInt8(lo), UInt8(lt) }, /* left letter              */

        /*ao*/    { UInt8(ao), UInt8(xl), UInt8(xr), UInt8(cn), UInt8(cn), UInt8(xa), UInt8(ao), UInt8(ao), UInt8(ao), UInt8(ao) }, /* arabic lett. foll by ON*/
        /*ro*/    { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa), UInt8(ro), UInt8(ro), UInt8(ro), UInt8(rt) }, /* right lett. foll by ON */
        /*lo*/    { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa), UInt8(lo), UInt8(lo), UInt8(lo), UInt8(lt) }, /* left lett. foll by ON  */

        /*rt*/    { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa), UInt8(rt), UInt8(ro), UInt8(ro), UInt8(rt) }, /* ET following R          */
        /*lt*/    { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa), UInt8(lt), UInt8(lo), UInt8(lo), UInt8(lt) }, /* ET following L          */

        /*cn*/    { UInt8(ao), UInt8(xl), UInt8(xr), UInt8(cn), UInt8(cn), UInt8(xa), UInt8(cn), UInt8(ac), UInt8(ao), UInt8(ao) }, /* EN, AN following AL      */
        /*ra*/    { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa), UInt8(ra), UInt8(rc), UInt8(ro), UInt8(rt) }, /* arabic number foll R   */
        /*re*/    { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa), UInt8(re), UInt8(rs), UInt8(rs), UInt8(ret) }, /* european number foll R */
        /*la*/    { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa), UInt8(la), UInt8(lc), UInt8(lo), UInt8(lt) }, /* arabic number foll L   */
        /*le*/    { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa), UInt8(le), UInt8(ls), UInt8(ls), UInt8(let) }, /* european number foll L */

        /*ac*/    { UInt8(ao), UInt8(xl), UInt8(xr), UInt8(cn), UInt8(cn), UInt8(xa), UInt8(ao), UInt8(ao), UInt8(ao), UInt8(ao) }, /* CS following cn          */
        /*rc*/    { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa), UInt8(ro), UInt8(ro), UInt8(ro), UInt8(rt) }, /* CS following ra          */
        /*rs*/    { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa), UInt8(ro), UInt8(ro), UInt8(ro), UInt8(rt) }, /* CS,ES following re      */
        /*lc*/    { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa), UInt8(lo), UInt8(lo), UInt8(lo), UInt8(lt) }, /* CS following la          */
        /*ls*/    { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa), UInt8(lo), UInt8(lo), UInt8(lo), UInt8(lt) }, /* CS,ES following le      */

        /*ret*/ { UInt8(ro), UInt8(xl), UInt8(xr), UInt8(ra), UInt8(re), UInt8(xa),UInt8(ret), UInt8(ro), UInt8(ro), UInt8(ret) }, /* ET following re          */
        /*let*/ { UInt8(lo), UInt8(xl), UInt8(xr), UInt8(la), UInt8(le), UInt8(xa),UInt8(let), UInt8(lo), UInt8(lo), UInt8(let) }, /* ET following le          */
    };

    enum BidiAction {
        // primitives
        IX = 0x100,                                    ///< increment
        XX = 0xF,                                    ///< no-op

        // actions
        xxx = (XX * 16) + XX,                        ///< no-op
        xIx = IX + xxx,                                ///< increment run
        xxN = (XX * 16) + BidiUnicode::ON,                ///< set current to N
        xxE = (XX * 16) + BidiUnicode::EN,                ///< set current to EN
        xxA = (XX * 16) + BidiUnicode::AN,                ///< set current to AN
        xxR = (XX * 16) + BidiUnicode::R,                ///< set current to R
        xxL = (XX * 16) + BidiUnicode::L,                ///< set current to L
        Nxx = (BidiUnicode::ON * 16) + 0xF,            ///< set run to neutral
        Axx = (BidiUnicode::AN * 16) + 0xF,            ///< set run to AN
        ExE = (BidiUnicode::EN * 16) + BidiUnicode::EN,    ///< set run to EN, set current to EN
        NIx = (BidiUnicode::ON * 16) + (0xF + IX),        ///< set run to N, increment
        NxN = (BidiUnicode::ON * 16) + BidiUnicode::ON,    ///< set run to N, set current to N
        NxR = (BidiUnicode::ON * 16) + BidiUnicode::R,    ///< set run to N, set current to R
        NxE = (BidiUnicode::ON * 16) + BidiUnicode::EN,    ///< set run to N, set current to EN

        AxA = (BidiUnicode::AN * 16) + BidiUnicode::AN,    ///< set run to AN, set current to AN
        NxL = (BidiUnicode::ON * 16) + BidiUnicode::L,    ///< set run to N, set current to L
        LxL = (BidiUnicode::L * 16) + BidiUnicode::L        ///< set run to L, set current to L
    };

    /** action definition table
        States can be of two kinds:
        - Immediate Resolution State, where each input token
        is resolved as soon as it is seen. These states have
        only single action codes (xxN) or the no-op (xxx)
        for static input tokens.
        - Deferred Resolution State, where input tokens either
        either extend the run (xIx) or resolve its Type (e.g. Nxx).

        Input classes are of three kinds
        - Static Input Token, where the class of the token remains
        unchanged on output (AN, L, N, R)
        - Replaced Input Token, where the class of the token is
        always replaced on output (AL, BN, NSM, CS, ES, ET)
        - Conditional Input Token, where the class of the token is
        changed on output in some but not all cases (EN)

        Where tokens are subject to change, a double action
        (e.g. NxA, or NxN) is _required_ after deferred states,
        resolving both the deferred state and changing the current token.

        These properties of the table are verified by assertions below.
        This code is needed only during debugging and maintenance */
    static const UInt16 actionWeak[][10] =
    {
        //   N,.. L,   R,  AN,  EN,  AL, NSM,  CS,..ES,  ET,
        /*xa*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxA), UInt16(xxR), UInt16(xxR), UInt16(xxN), UInt16(xxN), UInt16(xxN) }, /* arabic letter            */
        /*xr*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxE), UInt16(xxR), UInt16(xxR), UInt16(xxN), UInt16(xxN), UInt16(xIx) }, /* right leter             */
        /*xl*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxL), UInt16(xxR), UInt16(xxL), UInt16(xxN), UInt16(xxN), UInt16(xIx) }, /* left letter             */

        /*ao*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxA), UInt16(xxR), UInt16(xxN), UInt16(xxN), UInt16(xxN), UInt16(xxN) }, /* arabic lett. foll by ON    */
        /*ro*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxE), UInt16(xxR), UInt16(xxN), UInt16(xxN), UInt16(xxN), UInt16(xIx) }, /* right lett. foll by ON    */
        /*lo*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxL), UInt16(xxR), UInt16(xxN), UInt16(xxN), UInt16(xxN), UInt16(xIx) }, /* left lett. foll by ON    */

        /*rt*/ { UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(ExE), UInt16(NxR), UInt16(xIx), UInt16(NxN), UInt16(NxN), UInt16(xIx) }, /* ET following R            */
        /*lt*/ { UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(LxL), UInt16(NxR), UInt16(xIx), UInt16(NxN), UInt16(NxN), UInt16(xIx) }, /* ET following L            */

        /*cn*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxA), UInt16(xxR), UInt16(xxA), UInt16(xIx), UInt16(xxN), UInt16(xxN) }, /* EN, AN following  AL    */
        /*ra*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxE), UInt16(xxR), UInt16(xxA), UInt16(xIx), UInt16(xxN), UInt16(xIx) }, /* arabic number foll R    */
        /*re*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxE), UInt16(xxR), UInt16(xxE), UInt16(xIx), UInt16(xIx), UInt16(xxE) }, /* european number foll R    */
        /*la*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxL), UInt16(xxR), UInt16(xxA), UInt16(xIx), UInt16(xxN), UInt16(xIx) }, /* arabic number foll L    */
        /*le*/ { UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxL), UInt16(xxR), UInt16(xxL), UInt16(xIx), UInt16(xIx), UInt16(xxL) }, /* european number foll L    */

        /*ac*/ { UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(Axx), UInt16(AxA), UInt16(NxR), UInt16(NxN), UInt16(NxN), UInt16(NxN), UInt16(NxN) }, /* CS following cn         */
        /*rc*/ { UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(Axx), UInt16(NxE), UInt16(NxR), UInt16(NxN), UInt16(NxN), UInt16(NxN), UInt16(NIx) }, /* CS following ra         */
        /*rs*/ { UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(ExE), UInt16(NxR), UInt16(NxN), UInt16(NxN), UInt16(NxN), UInt16(NIx) }, /* CS,ES following re        */
        /*lc*/ { UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(Axx), UInt16(NxL), UInt16(NxR), UInt16(NxN), UInt16(NxN), UInt16(NxN), UInt16(NIx) }, /* CS following la         */
        /*ls*/ { UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(Nxx), UInt16(LxL), UInt16(NxR), UInt16(NxN), UInt16(NxN), UInt16(NxN), UInt16(NIx) }, /* CS,ES following le        */

        /*ret*/{ UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxE), UInt16(xxR), UInt16(xxE), UInt16(xxN), UInt16(xxN), UInt16(xxE) }, /* ET following re            */
        /*let*/{ UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxx), UInt16(xxL), UInt16(xxR), UInt16(xxL), UInt16(xxN), UInt16(xxN), UInt16(xxL) }, /* ET following le            */
    };

    //  -------------------------------------------------------------------------
    static inline UInt8 GetDeferredType(UInt16 action)
    {
        return UInt8((action >> 4) & 0xFU);
    }

    //  -------------------------------------------------------------------------
    static inline UInt8 GetResolvedType(UInt16 action)
    {
        return UInt8(action & 0xFU);
    }

    //  -------------------------------------------------------------------------
    static void resolveWeak(UInt8 baselevel, UInt8* pcls, UInt8 *plevel, TextLength cch)
    {
        UInt8 state = UInt8(odd(baselevel) ? xr : xl);
        UInt8 cls;

        TextLength cchRun = 0;
        UInt8 level = baselevel;

        TextPosition ich;
        for (ich = 0; ich < cch; ich++) {
            // ignore boundary neutrals
            if (pcls[ich] == UInt8(BidiUnicode::BN)) {
                // must flatten levels unless at a level change;
                plevel[ich] = level;

                // lookahead for level changes
                if (((ich + 1) == cch) && (level != baselevel)) {
                    // have to fixup last BN before end of the loop, since
                    // its fix-upped value will be needed below the assert
                    pcls[ich] = EmbeddingDirection(level);
                }
                else if (((ich + 1) < cch) && (level != plevel[ich+1]) && (pcls[ich+1] != UInt8(BidiUnicode::BN))) {
                    // fix up LAST BN in front / after a level run to make
                    // it act like the SOR/EOR in rule X10
                    UInt8 newlevel = plevel[ich+1];
                    if (level > newlevel) {
                        newlevel = level;
                    }
                    plevel[ich] = newlevel;

                    // must match assigned level
                    pcls[ich] = EmbeddingDirection(newlevel);
                    level = plevel[ich+1];
                }
                else {
                    // don't interrupt runs
                    if (cchRun != 0) {
                        cchRun++;
                    }
                    continue;
                }
            }

            cls = pcls[ich];

            UInt16 action = actionWeak[state][cls];

            // resolve the directionality for deferred runs
            UInt8 clsRun = GetDeferredType(action);
            if (clsRun != UInt8(XX)) {
                SetDeferredRun(pcls, cchRun, ich, clsRun);
                cchRun = 0;
            }

            // resolve the directionality class at the current location
            UInt8 clsNew = GetResolvedType(action);
            if (clsNew != UInt8(XX)) {
                pcls[ich] = clsNew;
            }

            // increment a deferred run
            if ((UInt16(IX) & action) != 0) {
                cchRun++;
            }

            state = stateWeak[state][cls];
        }

        // resolve any deferred runs
        // use the direction of the current level to emulate PDF
        cls = EmbeddingDirection(level);

        // resolve the directionality for deferred runs
        UInt8 clsRun = GetDeferredType(actionWeak[state][cls]);
        if (clsRun != UInt8(XX)) {
            SetDeferredRun(pcls, cchRun, ich, clsRun);
        }
    }

    /** action values for neutral types */
    enum NeutralAction {
        // action to resolve previous input
        nL = BidiUnicode::L,                 ///< resolve EN to L
        En = 3 * 16,                    ///< resolve neutrals run to embedding level direction
        Rn = BidiUnicode::R * 16,            ///< resolve neutrals run to strong right
        Ln = BidiUnicode::L * 16,            ///< resolved neutrals run to strong left
        In = (1 * 256),                    ///< increment count of deferred neutrals
        LnL = (1 * 16) + BidiUnicode::L    ///< set run and EN to L
    };

    //  -------------------------------------------------------------------------
    static inline UInt8 GetDeferredNeutrals(UInt16 action, UInt8 level)
    {
        action = (action >> 4) & 0xFU;
        return (action == (UInt16(En) >> 4)) ? EmbeddingDirection(level) : UInt8(action);
    }

    //  -------------------------------------------------------------------------
    static inline UInt8 GetResolvedNeutrals(UInt16 action)
    {
        action = action & 0xFU;
        return (action == UInt16(In)) ? 0 : UInt8(action);
    }

    /** neutral types state values
        By rule W7, whenever a EN is 'dominated' by an L (including start of
        run with embedding direction = L) it is resolved to, and further treated
        as L.
        This leads to the need for 'a' and 'na' states. */
    enum NeutralState {
        // new temporary class
        r,            ///< R and characters resolved to R
        l,            ///< L and characters resolved to L
        rn,            ///< N preceded by right
        ln,            ///< N preceded by left
        a,            ///< AN preceded by left (the abbrev 'la' is used up above)
        na            ///< N preceeded by a
    };

    /** neutrals action table */
    static const UInt16 actionNeutrals[][5] = {
        //    N,    L,    R, AN, EN, = cls
        // state =
        { UInt16(In),          0,          0,          0,           0 },            // r    right
        { UInt16(In),          0,          0,          0,  UInt16(BidiUnicode::L) },     // l    left

        { UInt16(In), UInt16(En), UInt16(Rn), UInt16(Rn), UInt16(Rn) },             // rn    N preceded by right
        { UInt16(In), UInt16(Ln), UInt16(En), UInt16(En), UInt16(LnL) },            // ln    N preceded by left

        { UInt16(In),          0,          0,          0, UInt16(BidiUnicode::L) },     // a   AN preceded by left
        { UInt16(In), UInt16(En), UInt16(Rn), UInt16(Rn), UInt16(En) },             // na    N  preceded by a
    } ;

    /** neutrals state transition table */
    static const UInt8 stateNeutrals[][5] =
    {
        //     N, L,    R,    AN, EN = cls
        // state =
        { UInt8(rn), UInt8(l), UInt8(r), UInt8(r), UInt8(r) },        // r   right
        { UInt8(ln), UInt8(l), UInt8(r), UInt8(a), UInt8(l) },        // l   left

        { UInt8(rn), UInt8(l), UInt8(r), UInt8(r), UInt8(r) },        // rn  N preceded by right
        { UInt8(ln), UInt8(l), UInt8(r), UInt8(a), UInt8(l) },        // ln  N preceded by left

        { UInt8(na), UInt8(l), UInt8(r), UInt8(a), UInt8(l) },        // a  AN preceded by left
        { UInt8(na), UInt8(l), UInt8(r), UInt8(a), UInt8(l) },        // na  N preceded by la
    } ;

    //  -------------------------------------------------------------------------
    static void resolveNeutrals(UInt8 baselevel, UInt8 *pcls, const UInt8 *plevel, TextLength cch)
    {
        // the state at the start of text depends on the base level
        UInt8 state = UInt8(odd(baselevel) ? r : l);
        UInt8 cls;

        TextLength cchRun = 0;
        UInt8 level = baselevel;

        TextPosition ich;
        for (ich = 0; ich < cch; ich++) {
            // ignore boundary neutrals
            if (pcls[ich] == UInt8(BidiUnicode::BN)) {
                // include in the count for a deferred run
                if (cchRun != 0) {
                    cchRun++;
                }

                // skip any further processing
                continue;
            }

            cls = pcls[ich];

            UInt16 action = actionNeutrals[state][cls];

            // resolve the directionality for deferred runs
            UInt8 clsRun = GetDeferredNeutrals(action, level);
            if (clsRun != UInt8(BidiUnicode::ON)) {
                SetDeferredRun(pcls, cchRun, ich, clsRun);
                cchRun = 0;
            }

            // resolve the directionality class at the current location
            UInt8 clsNew = GetResolvedNeutrals(action);
            if (clsNew != UInt8(BidiUnicode::ON)){
                pcls[ich] = clsNew;
            }

            if ((UInt16(In) & action) != 0){
                cchRun++;
            }

            state = stateNeutrals[state][cls];
            level = plevel[ich];
        }

        // resolve any deferred runs
        cls = EmbeddingDirection(level);    // eor has type of current level

        // resolve the directionality for deferred runs
        UInt8 clsRun = GetDeferredNeutrals(actionNeutrals[state][cls], level);
        if (clsRun != UInt8(BidiUnicode::ON)){
            SetDeferredRun(pcls, cchRun, ich, clsRun);
        }
    }

    //  -------------------------------------------------------------------------
    static void resolveImplicit(const UInt8 *types, UInt8 *levels, TextLength nChars)
    {
        static const UInt8 addLevel[][4] = {
            // L,  R,    AN, EN = cls
            // level =
            { 0,    1,    2,    2 },    // EVEN
            { 1,    0,    1,    1 },    // ODD
        };

        for (TextPosition i = 0; i < nChars; i++) {
            // cannot resolve bn here, since some bn were resolved to strong
            // types in resolveWeak. To remove these we need the original
            // types, which are available again in resolveWhiteSpace
            if (types[i] != UInt8(BidiUnicode::BN)) {
                levels[i] += addLevel[odd(levels[i]) ? 1 : 0][types[i] - 1];
            }
        }
    }

    //  -------------------------------------------------------------------------
    static void resolveWhitespace(UInt8 baselevel, const UInt8 *types,
                                   UInt8 *levels, TextLength nChars)
    {
        TextLength nCharsInRun = 0;
        UInt8 oldlevel = baselevel;

        TextPosition i;
        for (i = 0; i < nChars; i++) {
            switch(types[i]) {
                case BidiUnicode::WS:
                    nCharsInRun++;
                    break;

                case BidiUnicode::RLE:
                case BidiUnicode::LRE:
                case BidiUnicode::LRO:
                case BidiUnicode::RLO:
                case BidiUnicode::PDF:
                case BidiUnicode::BN:
                    levels[i] = oldlevel;
                    nCharsInRun++;
                    break;

                case BidiUnicode::S:
                case BidiUnicode::B:
                    // reset levels for WS before eot
                    SetDeferredRun(levels, nCharsInRun, i, baselevel);
                    nCharsInRun = 0;
                    levels[i] = baselevel;
                    break;

                default:
                    nCharsInRun = 0; // any other character breaks the run
                    break;
                }
                oldlevel = levels[i];
        }
        // reset level before eot
        SetDeferredRun(levels, nCharsInRun, i, baselevel);
    }

    
    // ----------------------------------------------------------------------------
    // ----------------------------------------------------------------------------
    // ----------------------------------------------------------------------------
    
    BidiUnicode::BidiClassEnum BidiUnicode::GetBidiClass(CodePoint codepoint)
    {
        const BidiClassData* start = g_bidiClassTable;
        const BidiClassData* end = g_bidiClassTable + g_bidiClassTableSize;
        const BidiClassData* result = LookupTools::FindRange(start, end, codepoint);
        if (result == end) {
            return L;
        }
        else {
            return result->type;
        }
    }

    TextLength BidiUnicode::ResolveClasses(UInt8 *classes, const TChar* text, TextPointer start, TextPointer end)
    {
        TextLength current = 0;
        for (FullBoundCodePointIterator it(start.GetTChar(text), end - start); (*it) != 0; (void)it.Advance()) {
            classes[current++] = UInt8(GetBidiClass(*it));
        }
        return current;
    }

    TextLength BidiUnicode::ResolveNClasses(UInt8 *classes, const TChar* text, TextPointer start, TextPointer end) 
    {
        TextLength current = 0;
        for (FullBoundCodePointIterator it(start.GetTChar(text), end - start); (*it) != 0; (void)it.Advance()) {

            classes[current++] = NTypes[GetBidiClass(*it)];
        }
        return current;
    }

    TextLength BidiUnicode::GenerateParagraphLevels(UInt8 *levels, UInt8 *classes, const TChar* text, TextPointer start, TextPointer end, UInt8 level)
    {
        TextLength charCount = ResolveNClasses(classes, text, start, end);
        if (level > 1) {
            level = DetermineBaseLevel(classes, charCount);
        }

        // resolveExplicit requires a buffer to store stack frames. _convertedText
        // is used for this temporary buffer.
        ResolveExplicitStackFrame stack[MAX_LEVEL];
        resolveExplicit(stack, level, classes, levels, charCount);
        resolveWeak(level, classes, levels, charCount);
        resolveNeutrals(level, classes, levels, charCount);
        resolveImplicit(classes, levels, charCount);

        static_cast<void>(ResolveClasses(classes, text, start, end));

        return charCount;
    }

    TextLength BidiUnicode::GenerateLineLevels(UInt8 *levels, const UInt8 *classes, const TChar* text, TextPointer start, TextPointer end, UInt8 level)
    {
        TextLength codePointCount = static_cast<TextLength>(
            TextPointer::GetCodePointPosition(text, start, end));
        resolveWhitespace(level, classes, levels, codePointCount);
        return codePointCount;
    }

}}} //namespaces.
