//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################



#if !defined(Candera_GlyphCacheAccess_h)
#define Candera_GlyphCacheAccess_h

#include <Candera/TextEngine/FontEngineDefinition.h>
#include <Candera/TextEngine/Internal/CacheAccessType.h>

namespace Candera {

    namespace TextRendering {

        /** @addtogroup CanderaTextEngine
         *  @{
         */

        class TextRenderer;

        /**
         *  @brief GlyphCacheAccess is responsible with providing an interface to the caching system, by
         *  which to store cached bitmaps. The caching system works by creating a new instance
         *  of each bitmap for each GlyphCacheAccess object, so each class extending this interface
         *  should behave like a singleton.
         */
        class GlyphCacheAccess : public FontEngineDefinition::GlyphCacheAccess {
            typedef FontEngineDefinition::GlyphCacheAccess Base;

        public:
            /**
             *  Constructor
             */
            GlyphCacheAccess() : Base() {}

            /**
             * Creates a handle used to access data from the cache at a later time.
             * The input bitmap contains a buffer stored in host memory. This buffer
             * is not persistent, it may get destroyed after returning form calling this
             * function. GlyphBitmap::pixels contains the address of this buffer.
             *     @param bitmap   Glyph bitmap data.
             *     @param cacheItem Existing handle for the current bitmap, if any.
             *     @return         A handle that the application may use to access
             *                     the glyph data.
             */
            virtual UInt8* Create(const GlyphBitmap& bitmap, UInt8* cacheItem);

            /**
             *  Destroys a handle used to access data from the cache.
             *  @param buffer   A handle returned by Create.
             */
            virtual void DestroyCacheItem(PixelBuffer buffer) override;

            /**
             *  Release cached items related to this object.
             */
            void ReleaseCache();

            virtual TextRendering::Internal::CacheAccessType::Enum GetCacheAccessType() const;
            
        private:
            GlyphCacheAccess(const GlyphCacheAccess& src);
            const GlyphCacheAccess& operator = (const GlyphCacheAccess& src);
        };

        /** @} */ // end of CanderaTextEngine

        inline void GlyphCacheAccess::DestroyCacheItem(PixelBuffer /*buffer*/) {}
        inline void GlyphCacheAccess::ReleaseCache() { Base::ReleaseCache(); }

    }// namespace TextRendering

}// namespace Candera


#endif// Candera_GlyphCacheAccess_h
