//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined CANDERA_TEXTRENDERING_GLYPH_INFO_H
#define CANDERA_TEXTRENDERING_GLYPH_INFO_H

#include <Candera/TextEngine/Types.h>

namespace Candera {
    namespace TextRendering {


        /** @addtogroup CanderaTextEngine
         *  @{
         */

        /**
         * @brief Structure used for describing how hinting will be done
         * during glyph rendering.
         *
         * Hinting corrects how alignment of glyph outlines to bitmap pixels is done.
         */
        struct GlyphHinter {
            /**
             * @brief Types of hinting available to the font engine.
             */
            enum Mode
            {
                Normal,     ///< Normal hinting.
                Light,      ///< Light hinting applies less corrections then normal hinting. Might have better performance.
                Monochrome, ///< Hinting appropriate for monochrome rendering.
                Lcd,        ///< Hinting appropriate for use with horizontally decimated LCD displays.
                VerticalLcd ///< Hinting appropriate for use with vertically decimated LCD displays.
            };
        };

        /**
         * @brief Structure used for describing a Glyph bitmap within
         * the Text Engine.
         * 
         * The Glyph bitmap is generated by the Font Engine and
         * is passed to the TextRenderContext for processing.
         * The Glyph bitmap also contains information about the 
         * cursor, and offsets of the glyph bitmap from the cursor.
         * 
         * The origin of the bitmap is in the upper left corner.
         */
        struct GlyphBitmap {
            // Forward types for backward compatibility.
            typedef PixelBuffer PixelBufferType; ///< Type of the pixel buffer.
            typedef PixelPitch2D PitchType;      ///< Type used for describing buffer pitch.
            typedef PixelSize SizeType;          ///< Type used for describing sizes.
            typedef PixelPosition PositionType;  ///< Type used for describing positions.
            typedef PixelPosition AdvanceType;   ///< Type used for describing advancements.

            /**
             * @brief Format of the pixel buffer.
             */
            enum Format {
                Unknown = 0,        ///< Format is unspecified.
                Monochrome = 1,     ///< Format is monochrome, one bit per pixel with big endian packing.
                Lcd = 2,            ///< Format for use with horizontally decimated LCDs.
                VerticalLcd = 3,    ///< Format for use with horizontally decimated LCDs.
                Grayscale = 8,      ///< Format is grayscale, one byte per pixel.

                // Deprecated.
                BPP8 = Grayscale    ///< Deprecated format. Points to grayscale.
            };

            /**
             * @brief Direction of the text while processing this glyph.
             */
            enum Direction {
                LeftToRight = 0,    ///< Direction forced to left to right.
                RightToLeft = 1,    ///< Direction forced to right to left.
                Implicit = 255      ///< Direction undefined.
            };

            Direction direction; ///< Direction of the text chunk containing this glyph.
            Format format; ///< Type of the pixels buffer.

            SizeType width; ///< Width of the bitmap in pixels.
            SizeType height; ///< Height of the bitmap in pixels.

            PositionType left; ///< Offset from the cursor to the left of the bitmap. (LeftOfBitmap - CursorX)
            PositionType top; ///< Offset from the top of the bitmap to the cursor. (CursorY - TopOfBitmap)

            AdvanceType xadvance; ///< Horizontal advancement of the cursor, after drawing this bitmap.
            AdvanceType yadvance; ///< Vertical advancement of the cursor, after drawing this bitmap.

            PitchType pitch; ///< Distance in bytes between consecutive bitmap lines.

            PixelBufferType pixels; ///< Pointer to a buffer containing the bitmap. 0 if not bitmap is available.

            GlyphIndex glyphIndex; ///< Index of the glyph within the font.
            TextPosition characterPosition; ///< Start position of the glyph within the text.
            FontIdentifier fontIdentifier; ///< Unique identifier of the font associated with the current glyph.
            FaceIdentifier faceIdentifier; ///< Unique identifier of the face associated with the current glyph.
            Int fontHeight; ///< Height of the font associated with the current glyph.
            Int fontWidth; ///< Width of the font associated with the current glyph.
        };

        /** @} */ // end of CanderaTextEngine

    }    // namespace TextRendering
}    // namespace Candera

#endif // CANDERA_TEXTRENDERING_GLYPH_INFO_H
