//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_FtEngine_h)
#define Candera_FtEngine_h

#include <Candera/Environment.h>
#include <Candera/TextEngine/Freetype/FtCustomGlyphCache.h>
#include <FeatStd/Util/PointerUtil.h>

namespace Candera {
    namespace TextRendering {
        class FontStore;
        class Font;

        namespace Internal {
/** @addtogroup FreetypeTextEngine
*  @{
*/

/** @brief FTC configuration

    Description of cache inner workings.

    The caches within freetype are realized as bidirectional linked lists.
    There are three lists:
        - one list for faces, holding at most faceCountMax faces.
        - one list for sizes, holding at most faceSizeCountMax size objects.
        - one list for everything else, allocating at most glyphCacheSize bytes.

    When an item is requested, if not found in the list, it is created. 
    If the creation of the new item exceeds the threshold set for the cache, 
    items are moved until the quota is met. Insertions are always done to 
    the front of the list and removal is always done from the back, 
    i.e. always the oldest node in the list is removed.

    The following requirements are set between the lists:
        - a size in the size cache always requires a face in the face cache. 
        This implies that when faces are destroyed all associated sizes are 
        destroyed and when sizes are created, a face is created as well.
        - when a Glyph object is created within the GlyphCache, a size is required.
        - when a Glyph is found in the GlyphCache, no other object is required; 
        also removing a face or a size does not remove objects from the GlyphCache.


    Scene should fit.

    In what follows content is called "active" if it calls for TextEngine 
    routines in any way. This includes: rendering, measuring, retrieving 
    information about fonts. Typically "active" content is passed more then once.
    The following describes what happens if some content becomes "inactive", 
    while other content becomes "active", taking its place.
    If all the content "active" at one time fits within the predefined cache, 
    cache access should stabilize after a short while, as newer content should 
    never get pushed out. The only content that is currently "active" and gets 
    pushed out from the cache is old content that is reused from old states. 
    Reused content that is pushed out in one pass is reinstated as new content 
    in the next pass. Stability should be reached when enough "inactive" content 
    has been pushed out.
    If the "active" content does not fit the cache, new content will be created 
    all the time.


    Rule to compute the correct cache sizes.

    faceCountMax should be the maximum number of fonts used in "active" content 
        at one time.
    faceSizeCountMax should be the maximum number of font sizes 
        used in "active" content at one time. (Maximum number of unique 
        TextRenderer::Font objects. If Font objects are duplicated 
        (have the same font name and size), they only count once.)
    glyphCacheSize should be aproximated as follows:
        maxGlyphNumber * (((maxSize ^ 2) * 2) + 20)
    where
            maxGlyphNumber is the maximum number of "active" glyphs.
            maxSize is the maximum size used for the font.
            ^ is the power operator.


    Workarounds and solutions.

    To avoid the waterfall effect described above, one could call 
    FtFontEngine::FlushCache when switching between large blocks of 
    "active" content. This would release all the old content, so useful 
    "content is never pushed out at a later time, resulting in a more 
    "predictable usage of the CPU.
               
    Static configuration of parameters.

    The default values used by this can be changed by configuring
    CANDERA_FREETYPE_CACHE_CONFIG. See FtFontEngineCacheConfig.h.template.

@param faceCountMax maximum number of faces supported by cache
@param faceSizeCountMax maximum number of sizes supported by cache
@param glyphCacheSize the size of the buffer to cache glyph bitmaps */
template<Int32 faceCountMax = 3, Int32 faceSizeCountMax = 6, Int32 glyphCacheSize = 24 * 24 * 100> struct FtFontEngineCacheConfig {
    enum {
        FaceCountMax = faceCountMax,
        FaceSizeCountMax = faceSizeCountMax,
        GlyphCacheSize = glyphCacheSize
    };
};

/** @brief FT2 specific implementation of FontEngine class */
class FtFontEngine {
public:
    static void FlushCache() { return GetData().FlushCache(); }

    /**
     * Convert a face id to a number.
     * @param faceId  The face id to convert.
     * @return  The number of the face id.
     */
    static UInt8 Convert(FTC_FaceID faceId) {
        return static_cast<UInt8>(FeatStd::Internal::PointerToScalar<SizeType>(faceId));
    }

protected:
    FtFontEngine();

    /**
        * Initialize the font engine with the given font store object.
        * Font store is required to provide font resource data to the font engine.
        * The lifetime of the supplied object must extend at least until Shutdown is called.
        * @param store a font store object (must not be null)
        * @return true if the font engine could be setup with the given store, false otherwise
        * @see FontEngine::Init
        */
    bool Init(FontStore *store) const { return GetData().Init(store); }
    /**
        * Shutdown the font engine.
        * No more font action beyond this point
        * @see FontEngine::Shutdown
        */
    void Shutdown() const { return GetData().Shutdown(); }
    /**
        * Retrieve the font store.
        * @see FontEngine::GetFontStore
        */
    FontStore* GetFontStore() const { return TheFontStore(); }

private:
    struct Data 
    {
    // Public
        Data();
        ~Data();

        bool Init(FontStore *store);
        void Shutdown();

        void FlushCache();

    // Private. But used internally for implementing the static
    // access to FtFontEngine, used across the font engine definition.
        FontStore *m_fontStore;

        FT_Library m_ftLib;
        FTC_Manager m_cacheMgr;
        FTC_SBitCache m_sbitCache;
        FTC_ImageCache m_imageCache;
        FTC_CMapCache m_cmapCache;
        FtCustomGlyphCache m_cgcCache;
        FtCgcAccessRec m_cgcAccess;

        void FreeCache();
        bool InitCache();
    };

    static Data& GetData();

    static FT_Library LibHdl() { return GetData().m_ftLib; }
    static FTC_Manager CacheMgr() { return GetData().m_cacheMgr; }
    static FTC_SBitCache SBitCache() { return GetData().m_sbitCache; }
    static FTC_ImageCache ImageCache() { return GetData().m_imageCache; }
    static FTC_CMapCache CMapCache() { return GetData().m_cmapCache; }
    static FtCustomGlyphCache CgcCache() { return GetData().m_cgcCache; }
    static FtCgcAccess CgcDefaultAccessor() { return &GetData().m_cgcAccess; }

    static FontStore* TheFontStore() { return GetData().m_fontStore; }

    static FT_Error FaceRequester(FTC_FaceID faceId, FT_Library lib, FT_Pointer request_data, FT_Face *face);
    static void FaceFinalizer(void*  object);

    static void FreeCache();
    static bool InitCache();

    static FTC_FaceID Convert(UInt8 faceId) { 
        return FeatStd::Internal::ScalarToPointer<FTC_FaceID>(static_cast<SizeType>(faceId));
    }

    friend class FtFont;
    friend class FtGlyphRenderer;
};

/** @} */ // end of FreetypeTextEngine
        }// namespace Internal
    }// namespace TextRendering
}// namespace Candera

#endif// Candera_FtEngine_h

