//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Util/NumericUtil.h>
#include <Candera/System/Diagnostics/Log.h>

#include <Candera/TextEngine/FontEngine.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/Internal/Shaper.h>

#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <CanderaPlatform/OS/FilePlatform.h>
#include <FeatStd/Util/PointerUtil.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <Candera/TextEngine/TextRenderer.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);
    namespace TextRendering {
        namespace Internal {

extern "C" {
    FT_Byte* ft_access_process_bitmap( FtCgcAccess      access,
                                       FT_Bitmap* bitmap,
                                       FtCgcItem        item, FT_UInt glyphIndex, FTC_FaceID faceIdentifier, FT_Int fontHeight, FT_Int fontWidth);
    void ft_access_free_bitmap( FtCgcAccess      access,
                                FtCgcItem        item );
    void ft_ansi_stream_close(FT_Stream stream);
    void ft_custom_stream_close(FT_Stream stream);
    FtInterfaceUnsignedLong ft_ansi_stream_io(FT_Stream stream, FtInterfaceUnsignedLong offset, Candera::UInt8 *buffer, FtInterfaceUnsignedLong count);
    FtInterfaceUnsignedLong ft_custom_stream_io(FT_Stream stream, FtInterfaceUnsignedLong offset, Candera::UInt8 *buffer, FtInterfaceUnsignedLong count);
    /**
     * Structure to encapsulate the function pointer to a StreamReadFunction.
     * An instance of this struct can be referenced by a void*, a function pointer not.
     */
    struct StreamReadWrapper
    {
        TextRendering::FontResourceDescriptor::StreamReadFunction strmFunc;
    };
}


// ----------------------------------------------------------------------------
FtFontEngine::FtFontEngine()
{
}

// ----------------------------------------------------------------------------
FtFontEngine::Data::Data() :
    m_fontStore(0),
    m_ftLib(0),
    m_cacheMgr(0),
    m_sbitCache(0),
    m_imageCache(0),
    m_cmapCache(0),
    m_cgcCache(0)
{
    FtCgcAccessRec access = {
        ft_access_process_bitmap,
        ft_access_free_bitmap,
        0,
        0,
        1
    };
    m_cgcAccess = access;
}

FtFontEngine::Data::~Data()
{
    bool initilized =
        (m_fontStore != 0) ||
        (m_ftLib != 0) ||
        (m_cacheMgr != 0) ||
        (m_sbitCache != 0) ||
        (m_imageCache != 0) ||
        (m_cmapCache != 0) ||
        (m_cgcCache != 0);
    if (initilized) {
        FEATSTD_LOG_ERROR("The font engine was not uninitialized properly.");
        Shutdown();
    }
}

// ----------------------------------------------------------------------------
void FtFontEngine::Data::FlushCache()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(Candera::TextRendering::Internal::TextRenderLocks::GetGlyphLock());
#endif
    FTC_Manager_Reset(m_cacheMgr);
    Shaper::GetDefaultShaperInstance().CleanUp();
}

void FtFontEngine::Data::FreeCache()
{
    if (m_cacheMgr != 0) {
        Shaper::GetDefaultShaperInstance().CleanUp();
        FTC_Manager_Done(m_cacheMgr);
        m_cacheMgr = 0;

        m_sbitCache = 0;
        m_imageCache = 0;
        m_cmapCache = 0;
        m_cgcCache = 0;
    }
}
bool FtFontEngine::Data::InitCache()
{
    bool ok = ((m_cacheMgr != 0) || (FTC_Manager_New(m_ftLib,
                                                    FontEngineDefinition::FtFontEngineConfig::FaceCountMax,
                                                    FontEngineDefinition::FtFontEngineConfig::FaceSizeCountMax,
                                                    FontEngineDefinition::FtFontEngineConfig::GlyphCacheSize,
                                                    FaceRequester,
                                                    0,
                                                    &m_cacheMgr) == 0));

    ok = ok && ((m_sbitCache != 0) || (FTC_SBitCache_New(m_cacheMgr, &m_sbitCache) == 0));
    ok = ok && ((m_imageCache != 0) || (FTC_ImageCache_New(m_cacheMgr, &m_imageCache) == 0));
    ok = ok && ((m_cmapCache != 0) || (FTC_CMapCache_New(m_cacheMgr, &m_cmapCache) == 0));

    ok = ok && ((m_cgcCache != 0) || (FtCustomGlyphCache_New(m_cacheMgr, &m_cgcCache) == 0));

    return ok;
}

// ----------------------------------------------------------------------------
bool FtFontEngine::Data::Init(FontStore *store)
{
    m_fontStore = store;
    bool ok = (m_ftLib != 0) || (FT_Init_FreeType(&m_ftLib) == 0);

    ok = ok && InitCache();

    if (!ok) {
        Shutdown();
    }
    return ok;
}

// ----------------------------------------------------------------------------
void FtFontEngine::Data::Shutdown()
{
    FreeCache();

    if (m_ftLib != 0) {
        static_cast<void>(FT_Done_FreeType(m_ftLib));
        m_ftLib = 0;
    }

    m_fontStore = 0;
}

// ----------------------------------------------------------------------------
FtFontEngine::Data& FtFontEngine::GetData()
{
    static Data s_data;
    return s_data;
}

// ----------------------------------------------------------------------------
FT_Error FtFontEngine::FaceRequester(FTC_FaceID faceId, FT_Library lib, FT_Pointer /*request_data*/, FT_Face *face)
{
    if (TheFontStore() == 0) {
        FEATSTD_LOG_ERROR("The font store is no longer attached to the font engine.");
        return FT_Err_Invalid_Handle;
    }

    // load resource descriptor from FontStore
    UInt8 descId = Convert(faceId);
    if (descId == FontStore::InvalidDescriptorId) {
        //TODO: add trace
        Int16 height;
        TheFontStore()->DefaultFont(descId, height);
    }

    FontResourceDescriptor descriptor;
    if (!TheFontStore()->Load(descId, descriptor)) {
        return FT_Err_Cannot_Open_Resource;
    }

    // let FreeType create the face according to the resource descriptor
    FT_Error rc = FT_Err_Invalid_Argument;
    switch (descriptor.storageType) {
        case FontResourceDescriptor::FileResource:
            rc = FT_New_Face(lib,
                descriptor.storageSpec.fileName,
                descriptor.faceIndex,
                face);
            break;

        case FontResourceDescriptor::MemoryResource:
            rc = FT_New_Memory_Face(lib,
                FeatStd::Internal::PointerToPointer<const FT_Byte*>(descriptor.storageSpec.memoryLocation.baseAddress),
                FT_Long(descriptor.storageSpec.memoryLocation.nBytes),
                descriptor.faceIndex,
                face);
            break;

        case FontResourceDescriptor::StreamResource:
            FT_Open_Args  args;
            args.flags = FT_OPEN_STREAM;
            args.stream = TEXTENGINE_TRANSIENT_NEW(FT_StreamRec_);
            if (args.stream != 0) {
                MemoryPlatform::Set(args.stream, 0, sizeof(FT_StreamRec_));
                args.stream->size = descriptor.storageSpec.streamSpecification.size;
                args.stream->read = ft_custom_stream_io;
                args.stream->close = ft_custom_stream_close;

                StreamReadWrapper* streamRead = TEXTENGINE_TRANSIENT_NEW(StreamReadWrapper);
                streamRead->strmFunc = descriptor.storageSpec.streamSpecification.readFunction;
                args.stream->descriptor.pointer = streamRead;

                args.stream->pathname.pointer = descriptor.storageSpec.streamSpecification.data;

                rc = FT_Open_Face(lib, &args, descriptor.faceIndex, face);
            }
            break;
        default:
            rc = FT_Err_Invalid_Argument;
            *face = 0;
            break;
    }

    if (rc == 0) {
        // tag the face object with descId and set the finalizer to unload
        // the font from FontStore on face destruction
        (*face)->generic.finalizer = FtFontEngine::FaceFinalizer;
        (*face)->generic.data = FeatStd::Internal::ScalarToPointer<void*>(static_cast<SizeType>(descId));
    }
    else if (*face != 0) {
        static_cast<void>(FT_Done_Face(*face));
        *face = 0;
    }
    else {
        //do nothing
    }
    return rc;
}

// ----------------------------------------------------------------------------
void FtFontEngine::FaceFinalizer(void *object)
{
    if (TheFontStore() == 0) {
        FEATSTD_LOG_ERROR("The font store is no longer attached to the font engine.");
        return;
    }
    FT_Face face = FeatStd::Internal::PointerToPointer<FT_Face>(object);
    // to convert void* to descId, first cast to FTC_FaceID and then apply Convert
    UInt8 descId = Convert(FeatStd::Internal::PointerToPointer<FTC_FaceID>(face->generic.data));
    static_cast<void>(TheFontStore()->Unload(descId));
}

extern "C" {

    // ----------------------------------------------------------------------------
    FT_CALLBACK_DEF( void* ) ft_alloc(FT_Memory /*memory*/, FtInterfaceLong size)
    {
        void *p = TEXTENGINE_ALLOC(UInt32(size));
        if (p != 0) {
            MemoryPlatform::Set(p, 0, size);
        }
        return p;
    }

    // ----------------------------------------------------------------------------
    FT_CALLBACK_DEF( void* ) ft_realloc(FT_Memory /*memory*/, FtInterfaceLong cur_size, FtInterfaceLong new_size, void *block )
    {
        void *p = TEXTENGINE_REALLOC(block, UInt32(new_size));
        if ((p != 0) && (new_size > cur_size)) {
            MemoryPlatform::Set(FeatStd::Internal::PointerAdd<UInt8*>(p, static_cast<OffsetType>(cur_size)), 0, SizeType(new_size - cur_size));
        }
        return p;
    }

    // ----------------------------------------------------------------------------
    FT_CALLBACK_DEF( void ) ft_free(FT_Memory /*memory*/, void *block)
    {
        TEXTENGINE_FREE(block);
    }

    // ----------------------------------------------------------------------------
    FT_CALLBACK_DEF( void ) ft_ansi_stream_close(FT_Stream stream)
    {
        FileHandle hdl = FeatStd::Internal::PointerToPointer<FileHandle>(stream->descriptor.pointer);
        static_cast<void>(FilePlatform::Close(hdl));
        stream->descriptor.pointer = NULL;
        stream->size               = 0;
        stream->base               = 0;
    }

    // ----------------------------------------------------------------------------
    FT_CALLBACK_DEF( FtInterfaceUnsignedLong ) ft_ansi_stream_io(FT_Stream stream, FtInterfaceUnsignedLong offset,
                                                UInt8 *buffer, FtInterfaceUnsignedLong count)
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(818, "interface defined by Freetype -> stream parameter cannot be made const")

        if ((count != 0) && (offset > stream->size)) {
            return 1;
        }

        FileHandle hdl = FeatStd::Internal::PointerToPointer<FileHandle>(stream->descriptor.pointer);
        UInt32 seekError = FilePlatform::Seek(hdl, Int32(offset), FilePlatform::Begin) ? 0 : 1;
        SizeType lRead = FilePlatform::Read(hdl, buffer, 1, count);

        return (count > 0) ? static_cast<FtInterfaceUnsignedLong>(lRead): seekError;
    }

    // ----------------------------------------------------------------------------
    FT_CALLBACK_DEF(void) ft_custom_stream_close(FT_Stream stream)
    {
        TEXTENGINE_FREE(stream->descriptor.pointer);
        TEXTENGINE_DELETE(stream);
    }

    // ----------------------------------------------------------------------------
    FT_CALLBACK_DEF(FtInterfaceUnsignedLong) ft_custom_stream_io(FT_Stream stream, FtInterfaceUnsignedLong offset, UInt8 *buffer, FtInterfaceUnsignedLong count)
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(818, "interface defined by Freetype -> stream parameter cannot be made const")
        if (((count != 0) && (offset > stream->size)) || (stream->descriptor.pointer == 0)) {
            return 1;
        }
        StreamReadWrapper *fn = FeatStd::Internal::PointerToPointer<StreamReadWrapper *>(stream->descriptor.pointer);
        return fn->strmFunc(buffer, static_cast<UInt32>(offset), static_cast<UInt32>(count), stream->pathname.pointer);
    }

    // ----------------------------------------------------------------------------
    FT_BASE_DEF( FT_Error ) FT_Stream_Open(FT_Stream stream, const FtInterfaceChar *filepathname)
    {
        FEATSTD_COMPILETIME_ASSERT(sizeof(FileHandle) <= sizeof(stream->descriptor.pointer));

        if (stream == 0) {
            return FT_Err_Invalid_Stream_Handle;
        }

        FileHandle file = FilePlatform::Open(filepathname);
        if (!FilePlatform::IsFileOpened(file)) {
            return FT_Err_Cannot_Open_Resource;
        }

        if (!FilePlatform::Seek(file, 0, FilePlatform::End)) {
            return FT_Err_Invalid_Stream_Seek;
        }

        FEATSTD_LINT_NEXT_EXPRESSION(970, "3rd party library requires unsigned long.")
        stream->size = FeatStd::Internal::NumericConversion<unsigned long>(FilePlatform::Tell(file));
        if (!FilePlatform::Seek(file, 0, FilePlatform::Begin)) {
            return FT_Err_Invalid_Stream_Seek;
        }

        stream->descriptor.pointer = FeatStd::Internal::PointerToPointer<void*>(file);
        stream->pathname.pointer   = const_cast<Char*>(filepathname);
        stream->pos                = 0;

        stream->read  = &ft_ansi_stream_io;
        stream->close = &ft_ansi_stream_close;

        return FT_Err_Ok;
    }

    // ----------------------------------------------------------------------------
    FT_BASE_DEF( FT_Memory ) FT_New_Memory(void)
    {
        static FT_MemoryRec_ memory = {
            0,          // user data
            ft_alloc,
            ft_free,
            ft_realloc
        };
        return &memory;
    }

    // ----------------------------------------------------------------------------
    FT_BASE_DEF( void ) FT_Done_Memory(FT_Memory  /*memory*/)
    {
    }

    // ----------------------------------------------------------------------------
    // Following functions are implemented according to FreeTypes platform specific
    // ftdebug.c

#if defined(FEATSTD_DEBUG)

#if defined(FT_DEBUG_LEVEL_ERROR)
    // ----------------------------------------------------------------------------
    FT_BASE_DEF( void ) FT_Message( const FtInterfaceChar* fmt, ... )
    {
        va_list ap;
        va_start(ap, fmt);
        FEATSTD_LOG_INFO(fmt, ap);
        va_end(ap);
        FEATSTD_UNUSED(ap); // false positive from MULTI compiler about unused 'ap'
    }

    // ----------------------------------------------------------------------------
    FT_BASE_DEF( void ) FT_Panic( const FtInterfaceChar* fmt, ... )
    {
        va_list ap;
        va_start(ap, fmt);
        FEATSTD_LOG_FATAL(fmt, ap);
        va_end(ap);
        FEATSTD_UNUSED(ap); // false positive from MULTI compiler about unused 'ap'
    }

    // ----------------------------------------------------------------------------
    FT_BASE_DEF( FtInterfaceInt ) FT_Throw(FT_Error error, FtInterfaceInt line, const FtInterfaceChar* file)
    {
        FEATSTD_UNUSED(file);
        FEATSTD_UNUSED(line);
        FEATSTD_LOG_DEBUG("freetype file %s, line %d, error code %d (0x%x)", file, line, error, error);
        return error;
    }

#endif //defined(FT_DEBUG_LEVEL_ERROR)

#if defined(FT_DEBUG_LEVEL_TRACE)

    // ----------------------------------------------------------------------------
    int  ft_trace_levels[trace_count];

#endif //defined(FT_DEBUG_LEVEL_TRACE)

#else  // ! defined(FEATSTD_DEBUG)
    #if defined(FT_DEBUG_LEVEL_ERROR)
        FT_BASE_DEF( void ) FT_Message( const FtInterfaceChar*  fmt, ... ) {}
        FT_BASE_DEF( void ) FT_Panic( const FtInterfaceChar*  fmt, ... ) {}
        FT_BASE_DEF( FtInterfaceInt ) FT_Throw(FT_Error, FtInterfaceInt, const FtInterfaceChar*) { return 0; }
    #endif //defined(FT_DEBUG_LEVEL_ERROR)

    #if defined(FT_DEBUG_LEVEL_TRACE)
        int  ft_trace_levels[trace_count];
    #endif //defined(FT_DEBUG_LEVEL_TRACE)
#endif // defined(FEATSTD_DEBUG)

// ----------------------------------------------------------------------------
FT_BASE_DEF( void ) ft_debug_init( void )
{
}

// ----------------------------------------------------------------------------
// functions used to allocate bitmap cache items from main memory.
FT_Byte* ft_access_process_bitmap( FtCgcAccess      /*access*/,
                                    FT_Bitmap* bitmap,
                                    FtCgcItem        item, FT_UInt /*glyphIndex*/, FTC_FaceID /*fontIdentifier*/, FT_Int /*fontHeight*/, FT_Int /*fontWidth*/)
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(818, "other override changes bitmap -> bitmap parameter cannot be made const")
    FEATSTD_UNUSED(item);
    FT_Int pitch = bitmap->pitch;
    if ( pitch < 0 )
    {
        pitch = -pitch;
    }
    FT_UInt size = static_cast<FT_UInt>(pitch * bitmap->rows);
    FT_Byte* data = FeatStd::Internal::PointerToPointer<FT_Byte*>(TEXTENGINE_ALLOC( size ));

    if ( data != 0 )
    {
        Candera::MemoryPlatform::Copy( data, bitmap->buffer, size );
    }

    return data;
}
void ft_access_free_bitmap( FtCgcAccess      /*access*/,
                            FtCgcItem        item )
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, item, CANDERA_LINT_REASON_NONCONST)
    TEXTENGINE_FREE(item->buffer);
}

} // extern "C"

        }   // namespace Internal
    }   // namespace TextRendering
}   // namespace Candera

