//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_FtFont_h)
#define Candera_FtFont_h

#include <Candera/TextEngine/Freetype/FtFontEngine.h>
#include <Candera/TextEngine/GlyphBitmap.h>
#include <Candera/TextEngine/FontSize.h>
#include <Candera/TextEngine/FontStore.h>
#include <Candera/TextEngine/Metrics.h>

namespace Candera {

    namespace TextRendering {
        namespace Internal {

            /** @addtogroup FreetypeTextEngine
            *  @{
            */

            /** @brief Freetype2 specific implementation of Font class */
            class FtFont {
            protected:

                FtFont();

                /**
                 * prepare the font object for text rendering with the given face and size
                 * @param faceName name of the font face (e.g. Arial). This string can be released as soon
                 *                 as the function returns.
                 * @param height font height in pixel
                 * @param width font width in pixel
                 * @param index of the face
                 * @return true if setup succeeded, false otherwise
                 * @see Font::Setup
                 */
                bool Setup(const Char *faceName, FontSize const& fontSize, UInt8 faceIndex);

                /**
                *  Set font height in pixel
                *  For compatibility reasons: Setting the height also sets the width/scale of the font.
                *  If a specific width/scale is requested, use SetFontSize instead.
                *  @param height font height
                */
                void SetHeight(Int16 height);

                /**
                 * Returns the actual font height in pixel.
                 * Note: scale factors are already included if set.
                 * To get plain sizes, request GetFontSize().
                 * @return font height in pixel.
                 * @see Font::GetHeight
                 */
                Int16 GetHeight() const;

                /**
                * Returns the actual font width in pixel.
                * Note: scale factors are already included if set.
                * To get plain sizes, request GetFontSize().
                * @return font width in pixel.
                * @see Font::GetWidth
                */
                Int16 GetWidth() const;

                /**
                *  Returns the font height in pixel.
                *  @return font height in pixel.
                */
                FontSize GetFontSize() const;

                /**
                *  Set font size in pixel
                *  It sets the whole fontSize properties.
                *  Standard font behavior is width == height.
                *  @param fontSize font size properties
                */
                void SetFontSize(FontSize const& fontSize);

                /**
                 * Retrieve metrics of the font object.
                 * @param metrics The metrics data structure to receive the metrics data.
                 * @return True if metrics could be retrieved, false otherwise.
                 * @see Font::GetMetrics
                 */
                bool GetMetrics(TextRendering::Metrics &metrics) const ;

                /**
                 * Returns the name of the font face.
                 * @return the name of the font face.
                 * @see Font::GetFaceName
                 */
                const Char* GetFaceName() const;

                /**
                * Returns the index of the font face.
                * @return the index of the font face.
                * @see Font::GetFaceIndex
                */
                UInt8 GetFaceIndex() const;

                /**
                 * Returns the unique identifier of the font face.
                 * @return the unique identifier of the font face
                 */
                UInt8 GetFaceId() const;

                /**
                 * Return offset of the font baseline..
                 * @return offset of the font baseline.
                 */
                Int16 GetBaseLineOffset() const;


                /**
                 * Configure the use of FT_RENDER_MODE_XXX.
                 */
                void SetRequestedGlyphFormat(GlyphBitmap::Format format);
                GlyphBitmap::Format GetRequestedGlyphFormat() const;

                /**
                 * Configure the use of FT_LOAD_TARGET_XXX.
                 */
                void SetHintingMode(GlyphHinter::Mode mode);
                GlyphHinter::Mode GetHintingMode() const;

                /**
                 * Configure FT_LOAD_NO_HINTING.
                 */
                void SetHintingEnabled(bool enabled);
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Internal::FtFont::IsHintingEnabled, CANDERA_LINT_REASON_NONCONSTMETHOD)
                bool IsHintingEnabled();

                /**
                 * Configure FT_LOAD_VERTICAL_LAYOUT.
                 */
                void SetVerticalLayoutEnabled(bool enabled);
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Internal::FtFont::IsVerticalLayoutEnabled, CANDERA_LINT_REASON_NONCONSTMETHOD)
                bool IsVerticalLayoutEnabled();

                /**
                 * Configure FT_LOAD_NO_AUTOHINT.
                 */
                void SetAutohintEnabled(bool enabled);
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Internal::FtFont::IsAutohintEnabled, CANDERA_LINT_REASON_NONCONSTMETHOD)
                bool IsAutohintEnabled();

                /**
                 * Configure FT_LOAD_FORCE_AUTOHINT.
                 */
                void SetForceAutohintEnabled(bool enabled);
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Internal::FtFont::IsForceAutohintEnabled, CANDERA_LINT_REASON_NONCONSTMETHOD)
                bool IsForceAutohintEnabled();

                /**
                 * Configure FT_LOAD_CROP_BITMAP.
                 */
                void SetCropEnabled(bool enabled);
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Internal::FtFont::IsCropEnabled, CANDERA_LINT_REASON_NONCONSTMETHOD)
                bool IsCropEnabled();

                /**
                 * Configure FT_LOAD_PEDANTIC.
                 */
                void SetPedanticVerificationEnabled(bool enabled);
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Internal::FtFont::IsPedanticVerificationEnabled, CANDERA_LINT_REASON_NONCONSTMETHOD)
                bool IsPedanticVerificationEnabled();

                const Char* GetFamilyName() const;

                const Char* GetStyleName() const;

                static const Char* GetStyleName(const Char *faceName, UInt8 styleIndex);

            public:
                bool operator == (const FtFont& rhs) const;
            private:
                struct Properties {
                    UInt8 m_hasKerning : 1;
                    UInt8 m_unicodeCharMap;
                    Int16 m_baseLineOffset;
                };
                Properties m_properties;
                FtCgcFontTypeRec m_type;
                mutable FeatStd::String m_familyName;   ///< Required for guaranteeing availability.

                FontSize m_fontSize;

                static FTC_FaceID GetFT_Face(const Char *faceName, UInt8 styleIndex, FT_Face& face);

                friend class FtFontEngine;
                friend class FtGlyphRenderer;
            };

            /** @} */ // end of FreetypeTextEngine

        }// namespace Internal
    }// namespace TextRendering
}// namespace Candera


#endif// Candera_FtFont_h
