//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/Environment.h>

#include <Candera/TextEngine/FontSize.h>
#include <Candera/TextEngine/Freetype/FtInclude.h>
#include <Candera/TextEngine/Freetype/FtFont.h>
#include <Candera/TextEngine/Freetype/FtFontEngine.h>

#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Diagnostics/Log.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <Candera/TextEngine/TextRenderer.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);
    namespace TextRendering {
        namespace Internal {

            // ----------------------------------------------------------------------------
            FtFont::FtFont()
                : m_properties(FtFont::Properties())
            {
                CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(835, CANDERA_LINT_REASON_MAINTAINABILITY)
                m_type.face_id = reinterpret_cast<FTC_FaceID>(FontStore::InvalidDescriptorId);
                m_type.height = 0;
                m_type.width = 0;
                m_type.flags.loadFlags = c_ftLoadDefault;
                m_type.flags.renderFlags = c_ftRenderModeNormal;

            }

            FTC_FaceID FtFont::GetFT_Face(const Char *faceName, UInt8 styleIndex, FT_Face& face)
            {
                UInt8 descriptorId = FontStore::InvalidDescriptorId;
                FontStore* fontStore = FtFontEngine::TheFontStore();
                if (0 != fontStore) {
                    descriptorId = fontStore->GetDescriptorId(faceName, styleIndex);
                }
                else {
                    FEATSTD_LOG_ERROR("The font store is not attached to the font engine.");
                }
                if (descriptorId == FontStore::InvalidDescriptorId) {
                    return 0;
                }

                FTC_FaceID faceId = FeatStd::Internal::ScalarToPointer<FTC_FaceID>(static_cast<SizeType>(descriptorId));
                if (FTC_Manager_LookupFace(FtFontEngine::CacheMgr(), faceId, &face) != 0) {
                    return 0;
                }

                return faceId;
            }

            // ----------------------------------------------------------------------------
            bool FtFont::Setup(const Char *faceName, FontSize const& fontSize, UInt8 faceIndex)
            {
                FT_Int fontHeight = static_cast<FT_Int>(fontSize.GetScaledHeight());
                FT_Int fontWidth = static_cast<FT_Int>(fontSize.GetScaledWidth());
                if (fontWidth < 0) {
                    fontWidth = 0;
                }
                if ((fontHeight) < 1) {
                    return false;
                }
                m_fontSize = fontSize;
                m_type.height = fontHeight;
                m_type.width = fontWidth;

                FT_Face face;
                m_type.face_id = GetFT_Face(faceName, faceIndex, face);
                bool ok = m_type.face_id != 0;
                if (ok) {
                    m_properties.m_hasKerning = FT_HAS_KERNING(face) != 0; //lint !e1960   Bitwise operator applied to signed underlying type  


                    // choose charmap by different criteria??
                    // E.g. a set char map
                    // Freetype itself sets the map to unicode (unicode32 if available) - only missing thing is the index
                    // backward iteration is most of the time faster - see freetype documentation for more info
                    // m_unicodeCharMap is 8 bit, so don't allow charmaps larger then 255.
                    // It shouldn't be the case that fonts have more charmaps.                
                    Int16 charmapCount = static_cast<Int16>(Math::Minimum(face->num_charmaps, FT_Int(256)));
                  
                    for (Int16 n = charmapCount - 1; n >= 0; n--)
                    {
                        if (face->charmap == face->charmaps[n]) {
                            m_properties.m_unicodeCharMap = static_cast<UInt8>(n);
                            break;
                        }
                    }
                    
                }

                FTC_ScalerRec scaler;

                scaler.face_id = m_type.face_id;
                scaler.height = FT_UInt(m_type.height);
                scaler.width = FT_UInt(m_type.width);
                scaler.pixel = 1;

                FT_Size size;

                ok = FTC_Manager_LookupSize(FtFontEngine::CacheMgr(), &scaler, &size) == 0;
                if (ok) {
                    FT_ULong ubmax = static_cast<FT_ULong>(FT_MulFix(size->face->bbox.yMax, size->metrics.y_scale) + 63);
                    m_properties.m_baseLineOffset = Int16(ubmax >> 6U) - 1;
                }

                return ok;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetHeight(Int16 height)
            {
                FontSize fontSize(height);
                static_cast<void>(Setup(GetFaceName(), fontSize, GetFaceIndex()));
            }

            // ----------------------------------------------------------------------------
            bool FtFont::operator==(const FtFont& rhs) const
            {
                return
                    (m_type.face_id == rhs.m_type.face_id) &&
                    (m_type.height == rhs.m_type.height) &&
                    (m_type.width == rhs.m_type.width) &&
                    (m_type.flags.loadFlags == rhs.m_type.flags.loadFlags) &&
                    (m_type.flags.renderFlags == rhs.m_type.flags.renderFlags) &&
                    (m_properties.m_hasKerning == rhs.m_properties.m_hasKerning) &&
                    (m_properties.m_unicodeCharMap == rhs.m_properties.m_unicodeCharMap) &&
                    (m_properties.m_baseLineOffset == rhs.m_properties.m_baseLineOffset);
            }

            // ----------------------------------------------------------------------------
            bool FtFont::IsHintingEnabled()
            {
                return (m_type.flags.loadFlags & c_ftLoadNoHinting) == 0;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetVerticalLayoutEnabled(bool enabled)
            {
                if (enabled) {
                    m_type.flags.loadFlags |= c_ftLoadVerticalLayout;
                }
                else {
                    m_type.flags.loadFlags &= ~c_ftLoadVerticalLayout;
                }
            }

            // ----------------------------------------------------------------------------
            bool FtFont::IsVerticalLayoutEnabled()
            {
                return (m_type.flags.loadFlags & c_ftLoadVerticalLayout) != 0;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetAutohintEnabled(bool enabled)
            {
                if (!enabled) {
                    m_type.flags.loadFlags |= c_ftLoadNoAutohint;
                }
                else {
                    m_type.flags.loadFlags &= ~c_ftLoadNoAutohint;
                }
            }

            // ----------------------------------------------------------------------------
            bool FtFont::IsAutohintEnabled()
            {
                return (m_type.flags.loadFlags & c_ftLoadNoAutohint) == 0;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetForceAutohintEnabled(bool enabled)
            {
                if (enabled) {
                    m_type.flags.loadFlags |= c_ftLoadForceAutohint;
                }
                else {
                    m_type.flags.loadFlags &= ~c_ftLoadForceAutohint;
                }
            }

            // ----------------------------------------------------------------------------
            bool FtFont::IsForceAutohintEnabled()
            {
                return (m_type.flags.loadFlags & c_ftLoadForceAutohint) != 0;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetCropEnabled(bool enabled)
            {
                if (enabled) {
                    m_type.flags.loadFlags |= c_ftLoadCropBitmap;
                }
                else {
                    m_type.flags.loadFlags &= ~c_ftLoadCropBitmap;
                }
            }

            // ----------------------------------------------------------------------------
            bool FtFont::IsCropEnabled()
            {
                return (m_type.flags.loadFlags & c_ftLoadCropBitmap) != 0;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetPedanticVerificationEnabled(bool enabled)
            {
                if (enabled) {
                    m_type.flags.loadFlags |= c_ftLoadPedantic;
                }
                else {
                    m_type.flags.loadFlags &= ~c_ftLoadPedantic;
                }
            }

            // ----------------------------------------------------------------------------
            bool FtFont::IsPedanticVerificationEnabled()
            {
                return (m_type.flags.loadFlags & c_ftLoadPedantic) != 0;
            }

            // ----------------------------------------------------------------------------
            const Char* FtFont::GetFamilyName() const
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(Candera::TextRendering::Internal::TextRenderLocks::GetGlyphLock());
#endif
                FT_Face face;
                if (GetFT_Face(GetFaceName(), 0, face) == 0) {
                    return 0;
                }
                m_familyName = FeatStd::String(face->family_name);
                return m_familyName.GetCString();
            }

            // ----------------------------------------------------------------------------
            const Char* FtFont::GetStyleName() const
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(Candera::TextRendering::Internal::TextRenderLocks::GetGlyphLock());
#endif
                FT_Face face;
                if (GetFT_Face(GetFaceName(), GetFaceIndex(), face) == 0) {
                    return 0;
                }

                return face->style_name;
            }
            
            // ----------------------------------------------------------------------------
            const Char* FtFont::GetStyleName(const Char *faceName, UInt8 styleIndex)
            {
                FT_Face face;
                if (GetFT_Face(faceName, styleIndex, face) == 0) {
                    return 0;
                }

                return face->style_name;
            }

            // ----------------------------------------------------------------------------
            Candera::Int16 FtFont::GetHeight() const
            {
                return static_cast<Int16>(m_type.height);
            }

            // ----------------------------------------------------------------------------
            Candera::Int16 FtFont::GetWidth() const
            {
                return static_cast<Int16>(m_type.width);
            }

            // ----------------------------------------------------------------------------
            FontSize FtFont::GetFontSize() const
            {
                return m_fontSize;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetFontSize(FontSize const& fontSize)
            {
                static_cast<void>(Setup(GetFaceName(), fontSize, GetFaceIndex()));
            }

            // ----------------------------------------------------------------------------
            bool FtFont::GetMetrics(TextRendering::Metrics &metrics) const
            {
                if (GetFaceId() == FontStore::InvalidDescriptorId) {
                    return false;
                }

                FT_Face face;
                bool ok = FTC_Manager_LookupFace(FtFontEngine::CacheMgr(), m_type.face_id, &face) == 0;
                if (!ok) {
                    return false;
                }

                FTC_ScalerRec scaler;

                scaler.face_id = m_type.face_id;
                scaler.height = FT_UInt(m_type.height);
                scaler.width = FT_UInt(m_type.width);;
                scaler.pixel = 1;

                FT_Size size;

                ok = FTC_Manager_LookupSize(FtFontEngine::CacheMgr(), &scaler, &size) == 0;
                if (ok) {
                    // Check if the font is scalable (Truetype, Opentype)
                    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(909, "Implicit conversion, violates MISRA C++ Rule 5-0-13: comes from freetype macro")
                    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(1960, "Bitwise operator applied to signed underlying type, violates MISRA C++ Rule 5-0-21: comes from freetype macro")
                    if ((size->face != 0) && (FT_IS_SFNT(size->face) != 0))
                    {
                        // Get the bounding box: The global bounding box is defined as the smallest rectangle
                        //                       that can enclose all the glyphs in a font face.
                        // Scale the font metrics to points * 64
                        FT_Long asc = FT_MulFix(size->face->bbox.yMax, size->metrics.y_scale);
                        FT_Long dsc = FT_MulFix(size->face->bbox.yMin, size->metrics.y_scale);

                        asc = asc + 63; // round to nearest pixel
                        metrics.ascender = Int16(static_cast<FT_ULong>(asc) >> 6U); // ascender in pixel (shift to convert from 1/64 pixels)
#if defined(CANDERA_TEXTENGINE_NATIVE_LINEHEIGHT_ENABLED)
                        metrics.descender = Int16(dsc >> 6U); 
#else
                        metrics.descender = -Int16(static_cast<FT_ULong>(-dsc) >> 6U); // descender in pixel (avoid right shift of negative value)
#endif
                    }
                    else
                    {
                        metrics.ascender = Int16(static_cast<FT_ULong>(size->metrics.ascender) >> 6U);
                        metrics.descender = Int16(static_cast<FT_ULong>(size->metrics.descender) >> 6U);
                    }
#if defined(CANDERA_TEXTENGINE_NATIVE_LINEHEIGHT_ENABLED)
                    metrics.lineHeight = Int16((static_cast<FT_ULong>(size->metrics.height)) >> 6U);
#else
                    // Define the line height = default line spacing (i.e., the baseline-to-baseline distance)
                    // The line height is calculated as 120% of EM size, this is how Adobe calculates it.
                    metrics.lineHeight = Int16(static_cast<Float>(m_type.height) * 1.2F + 0.5F);
#endif
                    metrics.maxAdvance = Int16((static_cast<FT_ULong>(size->metrics.max_advance)) >> 6U);
                }
                return ok;
            }

            // ----------------------------------------------------------------------------
            Candera::Int16 FtFont::GetBaseLineOffset() const
            {
                return m_properties.m_baseLineOffset;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetRequestedGlyphFormat(GlyphBitmap::Format format)
            {
                switch(format){
                    case GlyphBitmap::Monochrome: m_type.flags.renderFlags =    c_ftRenderModeMono;     break;
                    case GlyphBitmap::Lcd: m_type.flags.renderFlags =           c_ftRenderModeLcd;      break;
                    case GlyphBitmap::VerticalLcd: m_type.flags.renderFlags =   c_ftRenderModeLcdV;     break;
                    case GlyphBitmap::Grayscale: m_type.flags.renderFlags =     c_ftRenderModeNormal;   break;
                    default: m_type.flags.renderFlags = c_ftRenderModeNormal; break;
                }
                if (format == GlyphBitmap::Monochrome) {
                    m_type.flags.loadFlags |= c_ftLoadMonochrome;
                }
                else {
                    m_type.flags.loadFlags &= ~c_ftLoadMonochrome;
                }
            }

            // ----------------------------------------------------------------------------
            GlyphBitmap::Format FtFont::GetRequestedGlyphFormat() const
            {
                GlyphBitmap::Format format;
                switch(m_type.flags.renderFlags){
                    case c_ftRenderModeMono: format = GlyphBitmap::Monochrome; break;
                    case c_ftRenderModeLcd: format = GlyphBitmap::Lcd; break;
                    case c_ftRenderModeLcdV: format = GlyphBitmap::VerticalLcd; break;
                    case c_ftRenderModeNormal: format = GlyphBitmap::Grayscale; break;
                    default: format = GlyphBitmap::Unknown; break;
                }
                return format;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetHintingMode(GlyphHinter::Mode mode)
            {
                //m_type.flags.loadFlags &= ~FT_LOAD_TARGET_(~0U);    //causes lint warnings
                m_type.flags.loadFlags &= 0xFFF0FFFF;
                switch(mode){
                    case GlyphHinter::Normal:       m_type.flags.loadFlags |=   c_ftLoadTargetNormal; break;
                    case GlyphHinter::Light:        m_type.flags.loadFlags |=   c_ftLoadTargetLight;  break;
                    case GlyphHinter::Monochrome:   m_type.flags.loadFlags |=   c_ftLoadTargetMono;   break;
                    case GlyphHinter::Lcd:          m_type.flags.loadFlags |=   c_ftLoadTargetLcd;    break;
                    case GlyphHinter::VerticalLcd:  m_type.flags.loadFlags |=   c_ftLoadTargetLcdV;   break;
                    default:                        m_type.flags.loadFlags |=   c_ftLoadTargetNormal; break;
                }
            }

            // ----------------------------------------------------------------------------
            GlyphHinter::Mode FtFont::GetHintingMode() const
            {
                GlyphHinter::Mode format;
                //switch(FT_LOAD_TARGET_MODE(m_type.flags.renderFlags)){   //causes lint warnings
                UInt32 renderFlags=m_type.flags.renderFlags;
                switch((renderFlags>>16U)&15U){
                    case c_ftRenderModeNormal:  format = GlyphHinter::Normal;       break;
                    case c_ftRenderModeLight:   format = GlyphHinter::Light;        break;
                    case c_ftRenderModeMono:    format = GlyphHinter::Monochrome;   break;
                    case c_ftRenderModeLcd:     format = GlyphHinter::Lcd;          break;
                    case c_ftRenderModeLcdV:    format = GlyphHinter::VerticalLcd;  break;
                    default:                    format = GlyphHinter::Normal;       break;
                }
                return format;
            }

            // ----------------------------------------------------------------------------
            void FtFont::SetHintingEnabled(bool enabled)
            {
                if (!enabled) {
                    m_type.flags.loadFlags |= c_ftLoadNoHinting;
                }
                else {
                    m_type.flags.loadFlags &= ~c_ftLoadNoHinting;
                }
            }

            // ----------------------------------------------------------------------------
            const Char* FtFont::GetFaceName() const
            {
                FontStore* fontStore = FtFontEngine::TheFontStore();
                if (0 != fontStore) {
                    return fontStore->GetFaceName(GetFaceId());
                }
                else {
                    FEATSTD_LOG_ERROR("The font store is not attached to the font engine.");
                    return 0;
                }
            }

            // ----------------------------------------------------------------------------
            UInt8 FtFont::GetFaceIndex() const
            {
                FontStore* fontStore = FtFontEngine::TheFontStore();
                if (0 != fontStore) {
                    return fontStore->GetFaceIndex(GetFaceId());
                }
                else {
                    FEATSTD_LOG_ERROR("The font store is not attached to the font engine.");
                    return 0;
                }
            }


            Candera::UInt8 FtFont::GetFaceId() const
            {
                return FtFontEngine::Convert(m_type.face_id);
            }

        }    // namespace Internal
    }    // namespace TextRendering
}    // namespace Candera
