//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "FtCustomGlyphCacheAccess.h"
#include <Candera/TextEngine/GlyphBitmap.h>
#include <Candera/TextEngine/Freetype/FtFontEngine.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/Freetype/FtGlyphRenderer.h>

namespace Candera {
namespace TextRendering {
namespace Internal {

GlyphBitmap::Format FtPixelModeToFormat(FT_Pixel_Mode pixelMode)
{
    GlyphBitmap::Format format;
    switch (pixelMode) {
        case FT_PIXEL_MODE_MONO:    format = GlyphBitmap::Monochrome;   break;
        case FT_PIXEL_MODE_GRAY:    format = GlyphBitmap::Grayscale;    break;
        case FT_PIXEL_MODE_LCD:     format = GlyphBitmap::Lcd;          break;
        case FT_PIXEL_MODE_LCD_V:   format = GlyphBitmap::VerticalLcd;  break;
        default:                    format = GlyphBitmap::Unknown;      break;
    }

    return format;
}

extern "C" FT_Byte* FtCustomGlyphCacheAccessProcess( FtCgcAccess      access,
                                          FT_Bitmap* bitmap,
                                          FtCgcItem cachedItem, FT_UInt glyphIndex, FTC_FaceID faceIdentifier, FT_Int fontHeight, FT_Int fontWidth)
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, SECURE_POINTER_CAST );
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, SECURE_POINTER_CAST );
    FtCustomGlyphCacheAccess::Base *base = reinterpret_cast<FtCustomGlyphCacheAccess::Base*>(access);
    if (base->client == 0) {
        return 0;
    }

    if (bitmap->pixel_mode == FT_PIXEL_MODE_MONO)
    {
        if (!FtGlyphRenderer::ConvertMonochromToGray(bitmap))
        {
            return 0;
        }
        if (cachedItem != 0)
        {
            cachedItem->pitch = static_cast<FT_Short>(bitmap->pitch);
            cachedItem->pixelmode = bitmap->pixel_mode;
        }
    }

    GlyphBitmap gb;

    gb.left = 0;
    gb.top = 0;
    gb.width = static_cast<UInt16>(bitmap->width);
    gb.height = static_cast<UInt16>(bitmap->rows);

    gb.xadvance = 0;
    gb.pitch = static_cast<Int16>(bitmap->pitch);

    gb.format = FtPixelModeToFormat(static_cast<FT_Pixel_Mode>(bitmap->pixel_mode));

    gb.pixels = bitmap->buffer;

    gb.glyphIndex = glyphIndex;
    gb.faceIdentifier.m_value = FtFontEngine::Convert(faceIdentifier);
    gb.fontHeight = fontHeight;
    gb.fontWidth = fontWidth;

    return reinterpret_cast<FT_Byte*> (base->client->Create(gb, (cachedItem != 0) ? cachedItem->buffer : 0));
}

extern "C" void FtCustomGlyphCacheAccessFree( FtCgcAccess      access,
                                   FtCgcItem        item )
{
    FT_UNUSED( item );
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, SECURE_POINTER_CAST );
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, SECURE_POINTER_CAST );

    FtCustomGlyphCacheAccess::Base *base = reinterpret_cast<FtCustomGlyphCacheAccess::Base*>(access);
    if (base->client == 0) {
        return;
    }

    base->client->DestroyCacheItem(item->buffer);
}

extern "C" FT_UInt FtCustomGlyphCacheAccessCheck(FtCgcAccess      access,
    FtCgcItem        item)
{
    FT_UNUSED(item);
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, SECURE_POINTER_CAST);
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, SECURE_POINTER_CAST);

    FtCustomGlyphCacheAccess::Base *base = reinterpret_cast<FtCustomGlyphCacheAccess::Base*>(access);
    if (base->client == 0) {
        return 0;
    }

    return base->client->CheckCacheItem(item->buffer) ? 1 : 0;
}

extern "C" void FtCustomGlyphCacheAccessDone( FtCgcAccess   access )
{
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, SECURE_POINTER_CAST );
    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, SECURE_POINTER_CAST );

    TEXTENGINE_DELETE(reinterpret_cast<FtCustomGlyphCacheAccess::Base*>(access));
}

FtCustomGlyphCacheAccess::FtCustomGlyphCacheAccess() :
m_base(0)
{
    m_base = TEXTENGINE_TRANSIENT_NEW(Base);

    if (m_base != 0){
        m_base->client = this;

        m_base->base.doneAccess = FtCustomGlyphCacheAccessDone;

        m_base->base.processBitmap = FtCustomGlyphCacheAccessProcess;
        m_base->base.freeBitmap = FtCustomGlyphCacheAccessFree;
        m_base->base.checkItem = FtCustomGlyphCacheAccessCheck;

        m_base->base.refCount = 1;
    }
}

FtCustomGlyphCacheAccess::~FtCustomGlyphCacheAccess()
{
    if (m_base != 0){
        m_base->base.refCount --;
        FEATSTD_DEBUG_ASSERT(m_base->base.refCount == 0);

        if (m_base->base.refCount == 0) {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(740, SECURE_POINTER_CAST );
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(826, SECURE_POINTER_CAST );

            m_base->base.doneAccess(reinterpret_cast<FtCgcAccess>(m_base));
        }
        else {
            m_base->client = 0;
        }
        m_base = 0;
    }
}

void FtCustomGlyphCacheAccess::ReleaseCache() const
{
    FtFontEngine::FlushCache();
}

#ifndef CANDERA_DEPRECATED_3_2_1
#error Create with 2 parameters shall become abstract.
#else

inline UInt8* FtCustomGlyphCacheAccess::Create(const GlyphBitmap& bitmap, UInt8* cacheItem)
{
    FEATSTD_UNUSED(cacheItem);
    static bool isRecursive = false;
    FEATSTD_DEBUG_ASSERT(!isRecursive);
    if (isRecursive) {
        return 0; //The derived class should implement at least one virtual Create overload.
    }
    isRecursive = true;
    UInt8* result = Create(bitmap, 0);
    isRecursive = false;
    return result;
}

#endif

}    // namespace Internal
}    // namespace TextRendering
}    // namespace Candera
