//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "FtCustomGlyphCache.h"

#define FT2_BUILD_LIBRARY

//access to the cache manager
//#include FT_CACHE_MANAGER_H
#include <../src/cache/ftcmanag.h>
#include <../src/cache/ftcerror.h>

//access to memory management
#include <freetype/internal/ftmemory.h>
#include <freetype/internal/ftobjs.h>


//  -------------------------------------------------------------------------

/** Lint compliant pointer to pointer cast macro. 
    Usage of this macro assumes carefull review of the cast.
    @param ToType the resulting type
    @param pointer the pointer to be casted */
#define P2P_CAST(ToType, pointer) \
    /*lint --e(925)*/ \
    ((ToType) (pointer))

/** Lint compliant down cast macro.
Usage of this macro assumes carefull review of the cast.
@param ToType the resulting type
@param pointer the pointer to be casted */
#define DOWN_CAST(ToType, pointer) \
    /*lint --e(1939)*/ \
    ((ToType) (pointer))

/** Lint compliant down cast macro. 
    Usage of this macro assumes carefull review of the cast.
    @param ScalarType the resulting type
    @param pointer the pointer to be casted */
#define POINTER_TO_UINT32(pointer) \
    /*lint --e(923, 9091)*/ \
    ((FT_UInt32) (pointer))


//  -------------------------------------------------------------------------

#define FT_CGC_ITEMS_PER_NODE  16
#define FT_CGC_FAMILY_HASH_SIZE  37

#define CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(message_number, reason) /*lint --e{message_number}*/

typedef struct FtCgcFamilyRec_ {
    struct FtCgcFamilyRec_      *prev, *next;
    FT_UInt                     refCount;
    
    FtCgcAccess     access;

    FTC_ScalerRec   scaler;

    FtCgcFlags flags;
} FtCgcFamilyRec;
typedef FtCgcFamilyRec* FtCgcFamily;

typedef struct FtCgcNodeItemRec_ {
    FtCgcItemRec item;
    FT_Bool      done;
}FtCgcNodeItemRec;
typedef FtCgcNodeItemRec* FtCgcNodeItem;

typedef struct FtCgcNodeRec_ {
    FTC_NodeRec base;
    
    FtCgcFamily family;
    FT_Offset index;
    
    FtCgcNodeItemRec items[FT_CGC_ITEMS_PER_NODE];
} FtCgcNodeRec;
typedef FtCgcNodeRec* FtCgcNode;

typedef struct FtCgcQueryRec_ {
    FtCgcFamily family;
    FT_Offset index;
} FtCgcQueryRec;

typedef FtCgcQueryRec* FtCgcQuery;

typedef struct FtCustomGlyphCacheRec_ {
    FTC_CacheRec base;
    
    FtCgcFamily familyHash[FT_CGC_FAMILY_HASH_SIZE];
} FtCustomGlyphCacheRec;

FT_LOCAL_DEF ( FT_Error )
FtCgcNodeNew( FTC_Node    *pnode,
              FT_Pointer   query,
              FTC_Cache    cache );
FT_LOCAL_DEF ( FT_Offset )
FtCgcNodeWeight( FTC_Node   node,
                 FTC_Cache  cache );
FT_LOCAL_DEF ( FT_Bool )
FtCgcNodeCompare( FTC_Node    node,
                  FT_Pointer  key,
                  FTC_Cache   cache,
                  FT_Bool*    list_changed );
FT_LOCAL_DEF ( FT_Bool )
FtCgcNodeRemoveFaceId( FTC_Node    node,
                       FT_Pointer  key,
                       FTC_Cache   cache,
                       FT_Bool*    list_changed );
FT_LOCAL_DEF ( void )
FtCgcNodeFree( FTC_Node   node,
               FTC_Cache  cache );
FT_LOCAL_DEF ( FT_Error )
FtCustomGlyphCacheInit( FTC_Cache  cache );
FT_LOCAL_DEF ( void ) 
FtCustomGlyphCacheDone( FTC_Cache  cache );

const FTC_CacheClassRec FtCustomGlyphCacheClass =
{
    FtCgcNodeNew,                   //FTC_Node_NewFunc      node_new;
    FtCgcNodeWeight,                //FTC_Node_WeightFunc   node_weight;
    FtCgcNodeCompare,               //FTC_Node_CompareFunc  node_compare;
    FtCgcNodeRemoveFaceId,          //FTC_Node_CompareFunc  node_remove_faceid;
    FtCgcNodeFree,                  //FTC_Node_FreeFunc     node_free;

    sizeof(FtCustomGlyphCacheRec),  //FT_Offset             cache_size;
    FtCustomGlyphCacheInit,         //FTC_Cache_InitFunc    cache_init;
    FtCustomGlyphCacheDone,         //FTC_Cache_DoneFunc    cache_done;
} ;

FT_LOCAL_DEF ( FT_Error )
FtCgcNodeLoad( FtCgcNode   node, 
               const FTC_Cache   cache, 
               const FtCgcQuery  query,
               FT_Bool     addWeight );
FT_LOCAL_DEF ( void )
FtCgcFamilyAddRef( FtCgcFamily         family,
                   FtCustomGlyphCache  cache );
FT_LOCAL_DEF ( void )
FtCgcFamilyRemRef( FtCgcFamily         family,
                   FtCustomGlyphCache  cache );

FT_LOCAL_DEF ( FT_Error )
FtCgcNodeNew( FTC_Node    *pnode,
              FT_Pointer   query,
              FTC_Cache    cache )
{
    FT_Memory   memory = cache->memory;
    FtCgcNode   ftCgcNode;
    FtCgcQuery  ftCgcQuery = P2P_CAST(FtCgcQuery, query);
    FT_Error    error = 0;
    
    //create and initialize node
    if ( !FT_NEW( ftCgcNode ) )
    {
        FT_MEM_SET( ftCgcNode->items, 0, sizeof(ftCgcNode->items) );
        
        ftCgcNode->index = ftCgcQuery->index - ftCgcQuery->index % FT_CGC_ITEMS_PER_NODE;
        ftCgcNode->family = ftCgcQuery->family;
        
        FtCgcFamilyAddRef(ftCgcNode->family, DOWN_CAST(FtCustomGlyphCache, cache));
        
        error = FtCgcNodeLoad( ftCgcNode, cache, ftCgcQuery, FT_BOOL(0) );
        if ( error )
        {
            FtCgcNodeFree( (FTC_Node)ftCgcNode, cache );
            ftCgcNode = NULL;
        }
    }
    
    *pnode = (FTC_Node)ftCgcNode;
    
    return error;
}
FT_LOCAL_DEF ( FT_Offset )
FtCgcNodeWeight( FTC_Node   node,
                 FTC_Cache  cache )
{
    FtCgcNode       ftCgcNode = DOWN_CAST(FtCgcNode, node);
    FtCgcNodeItem   item = ftCgcNode->items;
    FT_Int          pitch;
    FT_UInt         count;
    FT_Offset       size;

    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(438, "formal parameter list is expected, variable cache must be handed over")
    FT_UNUSED( cache );

    /* the node itself */
    size = sizeof ( *ftCgcNode );

    for (count = FT_CGC_ITEMS_PER_NODE ; count > 0; count--, item++ )
    {
        if (item->item.buffer != 0)
        {
            pitch = item->item.pitch;
            if ( pitch < 0 )
                pitch = -pitch;

            /* add the size of a given glyph image */
            size += ((FT_UInt)(pitch)) * item->item.height;
        }
    }

    return size;
}

FT_LOCAL_DEF ( FT_Bool )
FtCgcNodeCompare( FTC_Node    node,
                  FT_Pointer  key,
                  FTC_Cache   cache,
                  FT_Bool*    list_changed)
{
    FtCgcNode   ftCgcNode  = DOWN_CAST(FtCgcNode, node);
    FtCgcQuery  query = P2P_CAST(FtCgcQuery, key);
    FT_UInt     idx = (FT_UInt)( query->index - ftCgcNode->index );
    FT_Bool     result;
    FtCgcAccess access = ftCgcNode->family->access;
    
    result = FT_BOOL(ftCgcNode->family == query->family &&
                     idx < FT_CGC_ITEMS_PER_NODE);
    
    if (result){
        FT_Int not_initialized;

        if ((access != 0) && (access->checkItem != 0) && (access->checkItem(access, &ftCgcNode->items[idx].item) == 0))
        {
            ftCgcNode->items[idx].done = 0;
        }

        //check item initialized
        //an item is initialized if the data has been read, and
        //whenever a user access allocator exists, it has generated a buffer.
        not_initialized = (ftCgcNode->items[idx].done == 0);

        if ( not_initialized )
        {
            FT_Error  error;

            node->ref_count++;  /* lock node to prevent flushing */
                                /* in retry loop                 */

            FTC_CACHE_TRYLOOP( cache )
            {
                error = FtCgcNodeLoad( ftCgcNode, cache, query, FT_BOOL(1) );
            }
            FTC_CACHE_TRYLOOP_END( list_changed );

            node->ref_count--;  /* unlock the node */

            if ( error )
                result = 0;
        }
    }
    
    return result;
}
FT_LOCAL_DEF ( FT_Bool )
FtCgcNodeRemoveFaceId( FTC_Node    node,
                       FT_Pointer  key,
                       FTC_Cache   cache,
                       FT_Bool*    list_changed)
{
    FtCgcNode   ftCgcNode = DOWN_CAST(FtCgcNode, node);
    FTC_FaceID  faceId = P2P_CAST(FTC_FaceID, key);
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(438, "formal parameter list is expected, variable cache must be handed over")
    FT_UNUSED( cache );
    FT_UNUSED( list_changed );

    return FT_BOOL( ftCgcNode->family->scaler.face_id == faceId );
}
FT_LOCAL_DEF ( void )
FtCgcNodeFree( FTC_Node   node,
               FTC_Cache  cache )
{
    FT_Memory       memory = cache->memory;
    FtCgcNode       ftCgcNode = DOWN_CAST(FtCgcNode, node);
    FtCgcNodeItem   item = ftCgcNode->items;
    FtCgcAccess     access = ftCgcNode->family->access;
    FT_UInt         count;
    
    //free bitmaps and other data
    if((access != 0) && (access->freeBitmap != 0)) {
        for (count = FT_CGC_ITEMS_PER_NODE ; count > 0; count--, item++ )
        {
            if ((item->done != 0) && (item->item.buffer != 0)) {
                access->freeBitmap( access, &item->item );
            }
            item->done = 0;
        }
    }
    
    FtCgcFamilyRemRef( ftCgcNode->family, DOWN_CAST(FtCustomGlyphCache, cache));

    memory->free(memory, node);
}
FT_LOCAL_DEF ( FT_Error )
FtCustomGlyphCacheInit( FTC_Cache  cache )
{
    FT_Error error = FTC_Cache_Init( cache );
    
    if ( !error )
    {
        FtCustomGlyphCache cgc = DOWN_CAST(FtCustomGlyphCache, cache);
        FT_MEM_SET( cgc->familyHash, 0, sizeof(cgc->familyHash) );
    }
    
    return error;
}
FT_LOCAL_DEF ( void )
FtCustomGlyphCacheDone( FTC_Cache  cache )
{
    FtCustomGlyphCache  ftCgcCache = DOWN_CAST(FtCustomGlyphCache, cache);
    FT_Int idx;

    FTC_Cache_Done( cache );
    
    for (idx = 0; idx < FT_CGC_FAMILY_HASH_SIZE; idx ++){
        FtCgcFamily family = ftCgcCache->familyHash[idx];
        
        while( family != 0 )
        {
            FtCgcFamily next = family->next;
            FtCgcFamilyRemRef(family, ftCgcCache);

            FT_ASSERT(next == 0 || next->prev == 0);
            family = next;
        }
    }
}

FT_LOCAL_DEF ( FT_Error )
FtCgcNodeLoad( FtCgcNode   node, 
               const FTC_Cache cache, 
               const FtCgcQuery query,
               FT_Bool     addWeight )
{
    /*lint --esym(818, cache) cannot change the freetype typedef to point to const (parameters cache and query) */
    /*lint --esym(818, query) cannot change the freetype typedef to point to const (parameters cache and query) */

    FT_UInt         gindex = (FT_UInt)query->index;
    FtCgcAccess     access = query->family->access;
    FtCgcNodeItem   item = node->items + (FT_UInt)( gindex - node->index );
    FT_Size         size;
   
    //create bitmaps
    FT_Error error = FTC_Manager_LookupSize( cache->manager, &node->family->scaler, &size );

    if ( !error )
    {
        FT_Face face = size->face;
        FT_Int32 loadFlags = node->family->flags.loadFlags;
        FT_Int32 renderFlags = node->family->flags.renderFlags;
        
        error = FT_Load_Glyph( face, gindex, loadFlags );
        
        //always generate bitmap, to acquire bitmap properties.
        //if (access){
            if ( !error )
            {
                FT_GlyphSlot  slot   = face->glyph;
                error = FT_Render_Glyph( slot, (FT_Render_Mode)renderFlags);
            }
        //}
        if ( !error ) 
        {
            FT_GlyphSlot  slot   = face->glyph;
            FT_Bitmap*    bitmap = &slot->bitmap;
            FT_Pos        xadvance, yadvance;

            /* approximate advance to closest pixel */
            xadvance = (slot->advance.x + 32) / 64;
            yadvance = (slot->advance.y + 32) / 64;
            
            item->item.width     = (FT_UShort)bitmap->width;
            item->item.height    = (FT_UShort)bitmap->rows;
            item->item.pitch     = (FT_Short)bitmap->pitch;
            item->item.left      = (FT_Short)slot->bitmap_left;
            item->item.top       = (FT_Short)slot->bitmap_top;
            item->item.xadvance  = (FT_Short)xadvance;
            item->item.yadvance  = (FT_Short)yadvance;
            
            if ( access != 0 && access->processBitmap != 0 )
            {
                item->item.buffer = access->processBitmap(access, bitmap, &item->item, gindex, node->family->scaler.face_id,
                                                          node->family->scaler.height, node->family->scaler.width);
                if ((item->item.buffer == 0) && (bitmap->buffer != 0)) {
                    error = FT_ERR(Out_Of_Memory);
                }
            }
            else {
                item->item.buffer = 0;
            }
            
            if ( error == 0 ) {
                if ( addWeight && item->item.buffer != 0 )
                {
                    cache->manager->cur_weight += ((FT_UShort)(FT_ABS( item->item.pitch ))) * item->item.height;
                }

                // The glyph bitmap data is cached only if an accessor is
                // present.
                if (access != 0) {
                    item->done = 1;
                }
            }
        }
    }

    return error;
}

FT_LOCAL_DEF ( void )
FtCgcFamilyAddRef( FtCgcFamily         family,
                   FtCustomGlyphCache  cache )
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(438, "formal parameter list is expected, variable cache must be handed over")
    FT_UNUSED( cache );
    family->refCount ++;
}

FT_UInt32 FtCgcFamilyGetHash( FtCgcFamilyRec const *family )
{
    FT_UInt32       scaler = POINTER_TO_UINT32((FTC_SCALER_HASH(&family->scaler)));
    FT_UInt32       flags = 31 * ((FT_UInt)((family->flags.loadFlags << 4) + family->flags.renderFlags));
    FT_UInt32       hash = (FT_UInt32)(scaler + flags);

    return hash;
}

FT_LOCAL_DEF ( void )
FtCgcFamilyRemRef( FtCgcFamily         family,
                   FtCustomGlyphCache  cache )
{
    family->refCount --;
    if ( family->refCount == 0 )
    {
        FT_Memory memory = cache->base.memory;
        
        if ( family->prev == 0 )
        {
            FT_UInt32       hash = FtCgcFamilyGetHash( family );
            FT_UInt32       idx = hash % FT_CGC_FAMILY_HASH_SIZE;
            
            cache->familyHash[idx] = family->next;
        }
        else
        {
            family->prev->next = family->next;
        }
        
        if ( family->next != 0 )
        {
            family->next->prev = family->prev;
        }
        
        if ( family->access != 0 )
        {
            family->access->refCount--;
            if ( family->access->refCount == 0 && family->access->doneAccess != 0 )
            {
                family->access->doneAccess(family->access);
            }
        }
        
        FT_FREE( family );
    }
}


FT_LOCAL_DEF ( FT_Error )
FtCgcFamilyNew( FtCgcFamily         input,
                FtCustomGlyphCache  cache,
                FtCgcFamily*        output,
                FT_UInt32*          ahash )
{
    FT_Error        error = 0;
    FT_Memory       memory = cache->base.memory;
    FT_UInt32       hash = FtCgcFamilyGetHash( input );
    FT_UInt32       idx = hash % FT_CGC_FAMILY_HASH_SIZE;
    FtCgcFamily     family = cache->familyHash[idx];
    
    FT_UNUSED( input );
    
    while( family != 0 )
    {
        FT_Int identical1 = (FT_Int)(FTC_SCALER_COMPARE( &family->scaler, &input->scaler ));
        FT_Int identical2 = ( family->flags.loadFlags == input->flags.loadFlags ) &&  ( family->flags.renderFlags == input->flags.renderFlags );
        FT_Int identical3 = ( (input->access) == 0 || (family->access == 0) || (family->access == input->access) );

        FT_Int identical =  identical1 && identical2 && identical3;
        if ( identical != 0 )
        {
            if ( family->access == 0 )
            {
                family->access = input->access;
                if (family->access != 0) 
                {
                    family->access->refCount++;
                }
            }
            break;
        }
        
        family = family->next;
    }
    
    if ( family == 0 )
    {
        if ( !FT_NEW( family ) )
        {
            family->refCount = 0;
            family->prev = 0;
            family->next = cache->familyHash[idx];
            
            if ( cache->familyHash[idx] ) {
                cache->familyHash[idx]->prev = family;
            }
            
            cache->familyHash[idx] = family;
            
            family->scaler = input->scaler;
            family->flags = input->flags;
            family->access = input->access;
            
            if ( family->access != 0 )
            {
                family->access->refCount++;
            }
        }
    }
    
    *ahash = hash;
    *output = family;
    
    return error;
}


FT_EXPORT(FT_Error)
FtCustomGlyphCache_New(FTC_Manager manager, FtCustomGlyphCache* cache)
{
    return FTC_Manager_RegisterCache( manager, &FtCustomGlyphCacheClass, FTC_CACHE_P( cache ) );
}

FT_EXPORT( FT_Error )
FtCustomGlyphCache_Lookup( FtCustomGlyphCache   cache,
                           FtCgcFontTypeConst   type,
                           FtCgcAccess          access,
                           FT_UInt              gindex,
                           FtCgcItem            *item,
                           FTC_Node             *anode )
{
    FtCgcQueryRec   query;
    FT_Error        error;
    FTC_Node        node = 0;
    FT_UInt32       hash;
    FT_UInt32       familyHash;
    
    FtCgcFamilyRec family;

    *item = 0;

    FT_UNUSED( type );

    family.access         = access;
    family.flags          = type->flags;
    family.scaler.face_id = type->face_id;
    family.scaler.width   = (FT_UInt)(type->width);
    family.scaler.height  = (FT_UInt)(type->height);
    family.scaler.pixel   = 1;
    family.scaler.x_res   = 0;
    family.scaler.y_res   = 0;

    query.index = gindex;
    error = FtCgcFamilyNew( &family, cache, &query.family, &familyHash );
    
    if ( !error ) {
        hash = familyHash +
               gindex / FT_CGC_ITEMS_PER_NODE;
        
        /*lint --e{925} cast pointer to pointer inside freetype macro */ 
        FTC_CACHE_LOOKUP_CMP( cache, 
                              FtCgcNodeCompare, 
                              hash, 
                              &query, 
                              node, 
                              error );
                              
        if ( !error ){
            FtCgcNode ftCgcNode = DOWN_CAST(FtCgcNode, node);
            FT_UInt idx = gindex - (FT_UInt)ftCgcNode->index;
            *item = &(ftCgcNode->items + idx)->item;
        }
    }
    
    if ( node != 0 && anode != 0 ) {
        node->ref_count++;
        *anode = node;
    }
    
    return error;
}
