//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_FontStore_h)
#define Candera_FontStore_h

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>

namespace Candera {
    namespace TextRendering {
        /** @addtogroup CanderaTextEngine
         *  @{
         */

        /**
         *  Font resource description
         *  The FontResourceDescriptor is used to specify how the font engine
         *  can retrieve font data
         */
        struct FontResourceDescriptor {
            /**
             *  StorageType specifies if resource can be loaded through file interfaces
             *  or memory location or is invalid.
             */
            enum StorageType {
                InvalidResource, ///< invalid resource descriptor
                FileResource,    ///< the font data can be loaded through file interface
                MemoryResource,  ///< the font data can be retrieved from a memory location
                StreamResource   ///< the font data can be loaded through stream interface
            };

            typedef UInt32(*StreamReadFunction)(void* buffer, UInt32 offset, UInt32 count, const void* data);

            struct StorageSpec {
                const Char *fileName;
                struct {
                    const void *baseAddress;
                    UInt32 nBytes;
                } memoryLocation;
                struct StreamSpec {
                    StreamReadFunction readFunction;
                    UInt32 size;
                    void* data;
                } streamSpecification;
            };

            UInt8 uid; ///< unique identifier of the descriptor (descId)
            UInt8 faceIndex; ///< index of the face within the font
            StorageType storageType; ///< defines how to interpret the storage specifier
            StorageSpec storageSpec; ///< font data storage specification
        };

        inline bool operator==(const FontResourceDescriptor &l, const FontResourceDescriptor &r) {
            return l.uid == r.uid;
        }

        inline bool operator!=(const FontResourceDescriptor &l, const FontResourceDescriptor &r) {
            return l.uid != r.uid;
        }

        /**
        * @brief This class is used to store the a font resource.
        *
        * The FontBuffer class is shared both by a Font and a FontStore implementation.
        * Purpose is to not tie the lifetime of font resources to the Font Cache.
        * Implementations of FontStore must handle the lifetime of FontBuffer.
        */
        class FontBuffer {
        public:
            typedef MemoryManagement::SharedPointer<FontBuffer> SharedPointer;
            typedef MemoryManagement::DisposerBase<const void*>::DisposerFunction DisposerFn;

            /**
            * Construct a new shared FontBuffer.
            *
            * @return the new shared FontBuffer.
            */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            virtual ~FontBuffer() {
                Dispose();
            }

            /**
            * Set the font resource data and a disposer function for the data.
            *
            * @param data pointer to valid font resource data.
            * @param disposer disposer function for font resource data.
            */
            void SetData(const void* data, DisposerFn disposer) {
                Dispose();
                m_data = data;
                m_disposer = disposer;
            }

            /**
            * Retrieve the font resource data.
            *
            * @return pointer to font resource data.
            */
            const void * GetData() const {
                return m_data;
            }

            /**
            * Set the size of font resource data.
            *
            * @param size size of font resource data in bytes.
            */
            void SetSize(SizeType size) {
                m_size = size;
            }

            /**
            * Retrieve the size of font resource data.
            *
            * @return size of font resource data in bytes.
            */
            SizeType GetSize() const {
                return m_size;
            }

            /**
            * Set whether font resource data is allocated.
            *
            * @param allocated true, if font resource data is allocated.
            */
            void SetAllocated(bool allocated) {
                m_isAllocated = allocated;
            }

            /**
            * Retrieve whether font resource data is allocated.
            *
            * @return true, if font resource data is allocated.
            */
            bool IsAllocated() const {
                return m_isAllocated;
            }

        private:
            void Dispose();

            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, "Use FontBuffer::Create")
            FontBuffer(): m_data(0), m_size(0), m_isAllocated(false), m_disposer(0) {}

            CANDERA_SHARED_POINTER_DECLARATION();
            const void* m_data;
            SizeType m_size;
            bool m_isAllocated;
            DisposerFn m_disposer;
        };

        /**
        * Create a shared FontBuffer object.
        *
        */
        inline FontBuffer::SharedPointer FontBuffer::Create()
        {
            CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
            FontBuffer* fontBuffer = TEXTENGINE_TRANSIENT_NEW(FontBuffer);
            FontBuffer::SharedPointer sharedPtr(fontBuffer);
            return sharedPtr;
        }

        /**
        * Dispose the data of a FontBuffer object.
        *
        */
        inline void FontBuffer::Dispose() {
            if ((0 != m_data) && (0 != m_disposer))
            {
                m_disposer(m_data);
            }
            m_data = 0;
            m_size = 0;
            m_isAllocated = false;
        }

        /**
         *  @brief FontStore is an abstract interface to retrieve font data
         *  The font store defines unique descriptor id for all the fonts. the descriptor id
         *  must not change for a font during run time.
         */
        class FontStore {
        public:

            /**
             *  Constructor
             */
            FontStore() {}

            /**
             *  Destructor
             */
            virtual ~FontStore() {}

            /**
             *  InvalidDescriptorId - invalid font descriptor id
             */
            enum {
                InvalidDescriptorId = 0 //< invalid font descriptor id
            };

            /**
             *  Returns the descriptor Id and height of the default font.
             *  @param descId Descriptor Id, out param.
             *  @param height Height, out param.
             */
            virtual void DefaultFont(UInt8 &descId, Int16 &height) const = 0;

            /**
            *  Return the descriptor id for a given font name
            *  @param name name of the font face (e.g. OpenSans)
            *  @param faceIndex index of the face in collection(e.g. index of the Italic face). Should be 0 for fonts that contain only one face.
            *  @return descriptor id of the font, InvalidDescriptorId if the font does not exist in the store
            */
            virtual UInt8 GetDescriptorId(const Char *name, UInt8 faceIndex);

            /**
             *  Load the font data into the store
             *  loading must not necessarily mean loading from file, but preparing
             *  what ever is necessary to read the data according to the FontResourceDescriptor
             *  specification.
             *  @param descId descriptor id of the font ( @see GetDescriptorId )
             *  @param descriptor receives the information how to access the font data
             *  @return true if the loading succeeded, false otherwise
             */
            virtual bool Load(UInt8 descId, FontResourceDescriptor &descriptor) = 0;

            /**
             *  Unload the font data from the store.
             *  The descriptor id does not become invalid. A new call to GetDescriptorId
             *  is not required.
             *  @param descId descriptor id of the font to unload.
             *  @return true if unloading succeeded, false otherwise
             */
            virtual bool Unload(UInt8 descId) = 0;

            /**
             *  Returns the face name.
             *  @param descId Descriptor Id of font whose face shall be returned.
             *  @return the face name.
             */
            virtual const Char* GetFaceName(UInt8 descId) = 0;

            /**
            *  Returns the face index.
            *  @param descId Descriptor Id of font whose face shall be returned.
            *  @return the face name.
            */
            virtual UInt8 GetFaceIndex(UInt8 descId);
            
            /**
            *  Returns the shared font buffer for a given face.
            *  @param name name of the font face (e.g. OpenSans)
            *  @param faceIndex index of the face in collection(e.g. index of the Italic face). Should be 0 for fonts that contain only one face.
            *  @return the shared font buffer.
            */
            virtual FontBuffer::SharedPointer GetFontBuffer(const Char * /* name */, UInt8 /* faceIndex */) { return FontBuffer::Create(); }

            /**
            *  Releases all stored shared font buffer objects which are solely referenced by the font store.
            *  
            */
            virtual void ReleaseOrphanedFontBuffer() {}
        };

        /** @} */ // end of CanderaTextEngine
    }// namespace TextRendering

}// namespace Candera

#endif// Candera_FontStore_h

