//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextValidator.h"

#include <FeatStd/MemoryManagement/Heap.h>
#include <FeatStd/Util/DefaultValidationHelper.h>
#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Util/ValidationHelperBase.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif
#include <Candera/TextEngine/Async/TextValidationGroup.h>

namespace Candera {
    namespace TextRendering {




        TextValidator::TextValidator(ITextValidationUser* toBeValidated, TextValidationGroup * attachedValidationGroup) :
            m_valUser(toBeValidated),
            m_validationGroup(attachedValidationGroup),
            m_invalidCount(0),
            m_validCount(0)
        {
            FEATSTD_DEBUG_ASSERT(m_valUser != 0);
        }

        Candera::TextRendering::ITextValidationUser* TextValidator::GetValidationUser()
        {
            FEATSTD_DEBUG_ASSERT(m_valUser != 0);
            return m_valUser;
        }

        void TextValidator::ConfirmValidHandledImplementation()
        {
            // pending result is handled
            if (m_validCount != 0) {
                m_validCount--;
            }
        }

        bool TextValidator::IsSingleValid() const
        {

#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            // invalid text validator always valid so it is not a reason to block
            if (m_valUser == 0) {
                return true;
            }
            // async renderer is deactivated it is always valid (Calc is done as soon as it is requested)
            if (!m_valUser->IsAsyncPreRenderEnabled()) {
                return true;
            }
            if (m_invalidCount == 0)// text has not changed or everything in queue is valid now
            {
                return true;
            }
            return (m_validCount > 0); // invalid item is in queue and valid item waits for being updated
        }


        bool TextValidator::IsPendingValid() const
        {
            return m_validCount != 0;
        }

        void TextValidator::SetInvalidImplementation()
        {
            m_invalidCount++;
        }

        void TextValidator::SetValid()
        {
            SetValidImplementation();
        }

        void TextValidator::SetValidImplementation()
        {
            bool isValidAvailable = false;
            TextValidationGroup* group = 0;
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
                if (m_invalidCount != 0) {
                    m_invalidCount--;
                    m_validCount++;
                    group = m_validationGroup;
                    isValidAvailable = true;
                }
            }
            if ((isValidAvailable) && (group != 0)) {
                group->HandleValidation();
            }
        }

        void TextValidator::ResetImplementation()
        {
            m_validCount = 0;
            m_invalidCount = 0;
        }
        bool TextValidator::IsValid() const
        {
            return IsValidImplementation();
        }

        bool TextValidator::IsValidImplementation() const
        {
            if (m_validationGroup != 0) {
                return m_validationGroup->IsValid();
            }
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            return IsSingleValid();
        }
    }
}


