//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextValidationGroupConnector_h
#define Candera_TextEngine_TextValidationGroupConnector_h

#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/TextEngine/Async/TextValidator.h>
#include <FeatStd/Util/ValidationHelperBase.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSection.h>
#endif

namespace Candera {
    namespace TextRendering {
        class TextValidationGroup;

        /** @addtogroup CanderaTextEngine
        *  @{
        */
        /**
        * TextValidationGroupConnector connects TextValidationGroups together.
        * The connector has to be attached to the groups.
        * Before deleting the connector, it should be detached from the groups.
        * The groups which contain a connector work independent from each other as long as the connector is not invalidated.
        * As soon the connector is invalidated, all connected groups are invalid. And they stay invalid until all of the 
        * groups are internally valid (all validation users are valid).
        */
        class TextValidationGroupConnector {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(TextValidationGroupConnector);
            /**
            * This invalidates the connector and therefore all groups to which the connector is attached to.
            */
            void Invalidate();

            /**
            * Is valid when all attached groups became valid.
            * There is no explicit check if all groups are valid. The groups itself have to trigger that.
            * @return valid when all attached groups are valid.
            */
            bool IsValid() const;

            TextValidationGroupConnector();

            virtual ~TextValidationGroupConnector();

        protected:

#ifdef FEATSTD_THREADSAFETY_ENABLED
            mutable FeatStd::Internal::CriticalSection m_lock;
#endif

            struct ValidationGroupEx {
                TextValidationGroup * m_group;
                bool m_isValid;
            };

            /**
            * Sets state of a single group to valid which is attached to the connector.
            * @param group the group which will be flagged as valid.
            */
            void SetGroupToValid(TextValidationGroup const * group);

            /**
            * returns the internal structure for a group.
            * @param group for which the internal structure is requested.
            * @return internal structure of group
            */
            ValidationGroupEx* GetGroup(TextValidationGroup const * group);

            /**
            * invalidates all status flags of the internal group structures.
            */
            void SetAllGroupsToInvalid();

            /**
            * Sets the state of the connector to valid.
            */
            virtual void SetValid();

            /**
            * Checks whether all required groups are valid. 
            * This can be overridden to customize the intergroup connection.
            * E.g. also 'or' combinations possible with this approach.
            * @param group which triggered this call.
            * @return true to continue the validation process (leads to a trigger validation update)
            */
            virtual bool IsRequiredGroupsValid(TextValidationGroup const* /*group*/);

            /**
            * Checks if all groups are valid. This is the default behaviour of IsRequiredGroupsValid.
            * @return all attached groups are marked as valid or not.
            */
            bool IsAllGroupsValid() const;

            /**
            * Returns the internal structure of a specific group based on the internal index.
            * If index is greater than the maximum size - 0 will be returned.
            * @param idx index of the group.
            * @return the internal structure of a group or 0 if index is out of bounds.
            */
            ValidationGroupEx * GetGroup(SizeType const idx);

            /**
            * Tries to validate the connector. The group which calls this method is valid,
            * only the connector is left to be validated. Internally it checks whether the connector is really valid
            * by calling IsRequiredGroupsValid (or it were never invalid).
            * @param group - a vaild group (all attached nodes are valid - only connectors are left)
            * @return true if connector is valid
            */
            bool TryValidate(TextValidationGroup const * group);
        private:

            SizeType GetGroupIndex(TextValidationGroup const * group);
            

            bool AttachValidationGroup(TextValidationGroup* group);

            bool DetachValidationGroup(TextValidationGroup const* group);

            friend class TextValidationGroup;

            FeatStd::Internal::Vector<ValidationGroupEx> m_attachedGroups;
            bool m_isConnectorValid;


            FEATSTD_MAKE_CLASS_UNCOPYABLE(TextValidationGroupConnector);
            FEATSTD_SHARED_POINTER_DECLARATION();

        };
        /** @} */ // end of CanderaTextEngine
    }// namespace TextRendering
}// namespace Candera

#endif

