//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextValidationGroupConnector.h"

#include <Candera/TextEngine/Async/TextValidationGroup.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);
    namespace TextRendering {

        

        void TextValidationGroupConnector::Invalidate()
        {
            SetAllGroupsToInvalid();
        }

        bool TextValidationGroupConnector::TryValidate(TextValidationGroup const * group)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            if (m_isConnectorValid) {
                return true;
            }
            ValidationGroupEx * groupEx = GetGroup(group);
            if (groupEx == 0) {
                return true;
            }

            groupEx->m_isValid = true;
            if (IsRequiredGroupsValid(group)) {
                m_isConnectorValid = true;
                for (SizeType i = 0; i < m_attachedGroups.Size(); i++)
                {
                    if (m_attachedGroups[i].m_group != group) {
                        m_attachedGroups[i].m_group->SetValid();
                    }
                    FEATSTD_DEBUG_ASSERT(m_attachedGroups[i].m_group->IsValid());
                }
            }
            else {
                return false;
            }

            return true;
        }


        bool TextValidationGroupConnector::IsValid() const
        {
            return m_isConnectorValid;
        }

        TextValidationGroupConnector::TextValidationGroupConnector() : m_isConnectorValid(true) {}


        TextValidationGroupConnector::~TextValidationGroupConnector()
        {
            FEATSTD_DEBUG_ASSERT(m_attachedGroups.Size() == 0);
            if (m_attachedGroups.Size() > 0) {
                FEATSTD_LOG_WARN("The connector is not detached from all groups yet!");
            }
        }

        void TextValidationGroupConnector::SetGroupToValid(TextValidationGroup const * group)
        {
            ValidationGroupEx * groupEx = GetGroup(group);
            FEATSTD_DEBUG_ASSERT(groupEx != 0);
            if (groupEx != 0) {
                groupEx->m_isValid = true;
            }
        }

        TextValidationGroupConnector::ValidationGroupEx* TextValidationGroupConnector::GetGroup(TextValidationGroup const * group)
        {
            ValidationGroupEx * result = 0;
            SizeType i = 0;
            while (i < m_attachedGroups.Size()) {
                if (m_attachedGroups[i].m_group == group) {
                    result = &m_attachedGroups[i];
                    break;
                }
                i++;
            }
            return result;
        }


        Candera::TextRendering::TextValidationGroupConnector::ValidationGroupEx * TextValidationGroupConnector::GetGroup(SizeType const idx)
        {

            if (idx >= m_attachedGroups.Size()) {
                return 0;
            }
            return &(m_attachedGroups[idx]);
        }

        void TextValidationGroupConnector::SetAllGroupsToInvalid()
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            for (SizeType i = 0; i < m_attachedGroups.Size(); i++) {
                m_attachedGroups[i].m_isValid = false;
            }
            m_isConnectorValid = false;
        }

        void TextValidationGroupConnector::SetValid()
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            m_isConnectorValid = true;
        }

        bool TextValidationGroupConnector::IsRequiredGroupsValid(TextValidationGroup const* /*group*/)
        {
            return IsAllGroupsValid();
        }

        Candera::SizeType TextValidationGroupConnector::GetGroupIndex(TextValidationGroup const * group)
        {
            SizeType i = 0;
            while (i < m_attachedGroups.Size()) {
                if (m_attachedGroups[i].m_group == group) {
                    break;
                }
                i++;
            }
            return i;
        }

        bool TextValidationGroupConnector::IsAllGroupsValid() const
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            for (SizeType i = 0; i < m_attachedGroups.Size(); i++) {
                if (!(m_attachedGroups[i].m_isValid)) {
                    return false;
                }
            }
            return true;
        }

        bool TextValidationGroupConnector::AttachValidationGroup(TextValidationGroup* group)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            FEATSTD_DEBUG_ASSERT(group != 0);
            if (group == 0) {
                return false;
            }
            FEATSTD_DEBUG_ASSERT(GetGroupIndex(group) == m_attachedGroups.Size());
            ValidationGroupEx groupEx = {group, true};
            return m_attachedGroups.Add(groupEx);
        }

        bool TextValidationGroupConnector::DetachValidationGroup(TextValidationGroup const * group)
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            SizeType idx = GetGroupIndex(group);
            if (idx < m_attachedGroups.Size()) {
                return m_attachedGroups.Remove(idx);
            }
            return false;
        }

    }
}
