//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextValidationGroup.h"

#include <Candera/TextEngine/Async/TextValidator.h>
#include <Candera/TextEngine/Async/TextValidationEventSources.h>

namespace Candera {
    namespace TextRendering {
        FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);

        Candera::TextRendering::TextValidationGroup& TextValidationGroup::GetDefaultValidator()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(TextValidationGroup, s_defaultValidator);
            return s_defaultValidator;
        }

        TextValidationGroup::TextValidationGroup()
        {

        }

        TextValidationGroup::~TextValidationGroup()
        {
            for (SizeType i = 0; i < m_boundConnector.Size(); i++)
            {
                if (!m_boundConnector[i]->DetachValidationGroup(this)) {
                    FEATSTD_LOG_WARN("Detaching a connector from a validation group failed!");
                }
            }
            m_boundConnector.Clear();
        }

        bool TextValidationGroup::AttachValidationUser(ITextValidationUser * user)
        {
            FeatStd::SizeType i = 0;
            while (i < m_boundUser.Size()) {
                if (m_boundUser[i]->GetValidationUser() == user) {
                    return false;
                }
                ++i;
            }
            TextValidator::SharedPointer newValidator = TextValidator::SharedPointer(TEXTENGINE_TRANSIENT_NEW(TextValidator)(user, this));

            if (!m_boundUser.Add(newValidator)) {
                return false;
            }
            return user->AttachValidationHelper(newValidator.GetPointerToSharedInstance());
        }

        bool TextValidationGroup::DetachValidationUser(ITextValidationUser * user)
        {
            FeatStd::SizeType i = 0;
            while (i < m_boundUser.Size()) {
                if (m_boundUser[i]->GetValidationUser() == user) {
                    break;
                }
                ++i;
            }
            if (i == m_boundUser.Size()) {
                return false;
            }
            TextValidator::SharedPointer oldValidator = m_boundUser[i];
            if (m_boundUser.Remove(i)) {
                if (0 == m_boundUser.Size()) {
                    m_boundUser.Free();
                }
            }
            else {
                return false;
            }
            return user->DetachValidationHelper(oldValidator.GetPointerToSharedInstance());
        }

        bool TextValidationGroup::AttachGroupConnector(TextValidationGroupConnector::SharedPointer& connector)
        {
            if (m_boundConnector.Contains(connector)) {
                return false;
            }

            if (!m_boundConnector.Add(connector)) {
                return false;
            }
            return connector->AttachValidationGroup(this);
        }

        bool TextValidationGroup::DetachGroupConnector(TextValidationGroupConnector::SharedPointer& connector)
        {
            SizeType idx = 0;
            while (idx < m_boundConnector.Size()) {
                if (m_boundConnector[idx] == connector) {
                    if (!m_boundConnector[idx]->DetachValidationGroup(this)) {
                        FEATSTD_LOG_WARN("Detaching a connector from a validation group failed!");
                    }

                    if (m_boundConnector.Remove(idx)) {
                        if (0 == m_boundConnector.Size()) {
                            return true;
                        }
                    }

                    return false;
                }
                ++idx;
            }
            return false;
        }

        void TextValidationGroup::HandleValidation()
        {
            if (IsValid()) {
                TriggerTextUpdates();
            }
        }

        void TextValidationGroup::SetInvalidImplementation()
        {
            // FEATSTD_DEBUG_ASSERT(false);
        }

        void TextValidationGroup::SetValidImplementation()
        {
            //  FEATSTD_DEBUG_ASSERT(false);
            if (IsValid()) {
                TriggerTextUpdates();
            }
        }

        void TextValidationGroup::ResetImplementation()
        {
            // FEATSTD_DEBUG_ASSERT(false);
        }

        bool TextValidationGroup::IsValidImplementation() const
        {
            for (FeatStd::SizeType i = 0; i < m_boundUser.Size(); i++) {
                if (!m_boundUser[i]->IsSingleValid()) {
                    return false;
                }
            }
            bool result = true;
            // All validation user of this validation group are valid - only the connectors are missing
            for (FeatStd::SizeType i = 0; i < m_boundConnector.Size();i++)
            {
                if (!m_boundConnector[i]->TryValidate(this)) {
                    result = false;
                }
            }
            return result;
        }

        void TextValidationGroup::TriggerTextUpdates()
        {
            for (FeatStd::SizeType i = 0; i < m_boundUser.Size(); i++) {
                ITextValidationUser* valUser = m_boundUser[i]->GetValidationUser();
                bool isTriggerUpdate = (valUser != 0);
                if (isTriggerUpdate) {
                    isTriggerUpdate = (valUser->IsTriggerUpdateForced()) ||
                        ((valUser->IsAsyncPreRenderEnabled()) && (m_boundUser[i]->IsPendingValid()));
                    if (isTriggerUpdate) {
                        valUser->TriggerValidUpdate();
                    }
                }
            }
            FeatStd::EventSource& evtSource = Candera::TextRendering::Internal::TextValidationEventSources::GetDefaultValidationEventSource();
            Candera::TextRendering::Internal::TriggerValidationUpdateEvent nodeEvent;
            evtSource.DispatchEvent(nodeEvent);
        }


        void TextValidationGroup::ConfirmValidHandledImplementation()
        {
            FEATSTD_DEBUG_BREAK();
        }

    }
}

